
   #include <math.h>
   #include <sys/time.h>

   #include <mpi.h>

   #include "MPI_em3d.h"
   #include "MPI_em3d_i.c"

   int main(int argc, char **argv) 
   {
      MPI_Comm em3dcomm;
      int i, j, k, l, me, rc, size, color = 0;

      double elapsed_time, barrier_time;
      struct timeval start, end;
      struct timeval startb, endb;

      /*
       * Inputs to the program
       */
      int p, niter;

      gettimeofday(&start, NULL);

      {
         rc = MPI_Init(
                 &argc,
                 &argv
         );

         if (rc != MPI_SUCCESS)
         {
            printf(
               "MAIN:Problems initializing MPI "
               "...Exiting...\n"
            );

            MPI_Abort(MPI_COMM_WORLD, -1);
         }
      }

      {
         int hostnamel = HOSTNAMELENGTH;

         gethostname(
            hostName,
            hostnamel
         );
      }

      {
         rc = MPI_Comm_rank(MPI_COMM_WORLD, &me);
 
         if (rc != MPI_SUCCESS)
         {
            printf(
               "MAIN:Problems getting rank "
               "...Exiting...\n"
            );
 
            MPI_Abort(MPI_COMM_WORLD, -2);
         }
      }

      {
         rc = MPI_Comm_size(MPI_COMM_WORLD, &size);
 
         if (rc != MPI_SUCCESS)
         {
            printf(
               "MAIN:Problems getting size of MPI_COMM_WORLD "
               "...Exiting...\n"
            );
 
            MPI_Abort(MPI_COMM_WORLD, -3);
         }
      }

      if (me == (size - 1))
      {
         color = MPI_UNDEFINED;
      }

      rc = MPI_Comm_split(
              MPI_COMM_WORLD,
              color,
              me,
              &em3dcomm
      );

      if (rc != MPI_SUCCESS)
      {
         printf("Problems with MPI_Comm_split for MPI_COMM_WORLD\n");
         MPI_Abort(MPI_COMM_WORLD, -4);
      }

      if (me == (size - 1))
      {
         MPI_Finalize();
         return 0;
      }

      p = size - 1;
      niter = NUM_ITERATIONS;

      if (me == 0)
      {
         printf("Initializing System\n");
      }

      rc = Initialize_system(me, p);

      if (rc != 0)
      {
         printf("Problems with Initializing system\n");
         MPI_Abort(MPI_COMM_WORLD, -4);
      }

      if (me == 0)
      {
         printf("System initialized\n");
      }

      {
         int myb, ind, tnum;
         Body* b;

         if (me == 0)
         {                       
            myb = 0;
         }
                
         if (me > 0)
         {    
            myb = 1;
         }                     
             
         for (i = 0; i < niter; i++) 
         {
             /* 
              * Gather the dependency H values 
              */
             for (j = 0, ind = 0; j < p; j++) 
             {
                if (dependencies[me*p + j] == 0)
                {
                   continue;
                }

                b = bodies[ind++];
                tnum = b->nnum;

                for (k = 0; k < tnum; k++)
                {
                    if (b->ntype == E) 
                    {
                       continue; 
                    }

                    if (j == me)
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] != me)
                           {
                              MPI_Status status;
                              rc = MPI_Recv(
                                      &b->values[l], 
                                      1, 
                                      MPI_DOUBLE, 
                                      b->dbodies[l], 
                                      MPI_MSG_TAG, 
                                      em3dcomm, 
                                      &status
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems receiving H value"
                                     "...Exiting...\n"
                                  );

                                  MPI_Abort(MPI_COMM_WORLD, -6);
                              }
                           }
                       }
                    }
                    else
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] == me) 
                           {
                              double value = GetNodalValue(
                                                b->dnodes[l],
                                                bodies[myb]
                              );

                              rc = MPI_Send(
                                      &value, 
                                      1,
                                      MPI_DOUBLE, 
                                      j, 
                                      MPI_MSG_TAG, 
                                      em3dcomm
                              );

                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems sending H value"
                                     "...Exiting...\n"
                                  );

                                  MPI_Abort(MPI_COMM_WORLD, -7);
                              }
                           }
                       }
                    }

                    b = b->next;
                }
             }

             /* 
              * Computations for E nodes 
              */
             b = bodies[myb];
             tnum = b->nnum;

             for (j = 0; j < tnum; j++)
             {
                 if (b->ntype == H) 
                 {
                    continue; 
                 }

                 for (k = 0; k < b->edge_count; k++)
                 {
                     b->value -= (b->values[k])*(b->coeffs[k]);
                 }

                 b = b->next;
             }

             /* 
              * Gather the dependency E values 
              */
             for (j = 0, ind = 0; j < p; j++) 
             {
                if (dependencies[me*p + j] == 0)
                {
                   continue;
                }

                b = bodies[ind++];
                tnum = b->nnum;

                for (k = 0; k < tnum; k++)
                {
                    if (b->ntype == H) 
                    {
                       continue; 
                    }

                    if (j == me)
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] != me)
                           {
                              MPI_Status status;
                              rc = MPI_Recv(
                                      &b->values[l], 
                                      1, 
                                      MPI_DOUBLE, 
                                      b->dbodies[l], 
                                      MPI_MSG_TAG, 
                                      em3dcomm, 
                                      &status
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems receiving H value"
                                     "...Exiting...\n"
                                  );

                                  MPI_Abort(MPI_COMM_WORLD, -8);
                              }
                           }
                       }
                    }
                    else
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] == me) 
                           {
                              double value = GetNodalValue(
                                                b->dnodes[l],
                                                bodies[myb]
                              );
                              rc = MPI_Send(
                                      &value, 
                                      1,
                                      MPI_DOUBLE, 
                                      j, 
                                      MPI_MSG_TAG, 
                                      em3dcomm
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems sending H value"
                                     "...Exiting...\n"
                                  );

                                  MPI_Abort(MPI_COMM_WORLD, -9);
                              }
                           }
                       }
                    }

                    b = b->next;
                }
             }

             /* 
              * Computations for H nodes 
              */
             b = bodies[myb];
             tnum = b->nnum;

             for (j = 0; j < tnum; j++)
             {
                 if (b->ntype == E) 
                 {
                    continue; 
                 }

                 for (k = 0; k < b->edge_count; k++)
                 {
                     b->value -= (b->values[k])*(b->coeffs[k]);
                 }

                 b = b->next;
             }
         }
      }

      if (me == 0)
      {
         printf("Destroying system\n");
      }

      Destroy_system(p, me);

      if (me == 0)
      {
         printf("System destroyed\n");
      }

      /*
       * Calculate the barrier time for possible use.
       */
      {
        double tstart, tend;

        gettimeofday(&startb, NULL);

        for (i = 0; i < BARRIER_ITERATIONS; i++)
        {
           rc = MPI_Barrier(em3dcomm);

           if (rc != MPI_SUCCESS)
           {
              return rc;
           }
        }

        gettimeofday(&endb, NULL);

        tstart = startb.tv_sec + (startb.tv_usec/pow(10, 6));
        tend = endb.tv_sec + (endb.tv_usec/pow(10, 6));

        /*
         * There is a barrier at the end
         */
        barrier_time = (tend - tstart)/(BARRIER_ITERATIONS+1);
      }

      if (me != (size - 1))
      {
         rc = MPI_Comm_free(
                 &em3dcomm
         );

         if (rc != MPI_SUCCESS)
         {
            return rc;
         }
      }

      /*
       * Record the final time and record the elapsed time.
       */
      gettimeofday(&end, NULL);

      if (me == 0)
      {
         double tstart = start.tv_sec + (start.tv_usec/pow(10, 6));
         double tend = end.tv_sec + (end.tv_usec/pow(10, 6));

         double elapsed_time = (tend
                                - tstart
                                - barrier_time
         );

         printf(
            "Total number of nodes=%d, Total_number_of_dependencies=%d, time(sec)=%0.6f\n",
            Total_number_of_nodes,
            Total_number_of_dependencies,
            elapsed_time
         );
      }

      MPI_Finalize(); 
   }

