

/* vec.c Simple vector manipulation routines.
 */

#include "errors.h"
 /* Standard error codes. */

#include "vec.h"
/* Our type and prototype definitions */

/* ========================= Set to a constant ============================== */

/* Set a vector to zero */
error veczero (int size, vector result) {
  if (!result) return ERRnil;  /* must exist */
  if (size<1)  return ERRsize; /* and have positive size*/

  while(size--) *result++ = 0.0;

  return NoError;
}

/* ========================= Copy =========================================== */

/* ------------- Straight copy  ---------------- */
error veccpy (int size, vector result, vector source) {
    if (!result || !source) return ERRnil;  /* They must exist */
    if (size < 1)           return ERRsize; /* and have positive size */

    while (size--) *result++ = *source++;

    return NoError;
}

/* --------------- swap ------------- */
/* Swap two vectors. */
error vecswp (int size, vector first, vector second) {
   double temp;

   if (!first || !second) return ERRnil;  /* they must exist */
   if (size<1)            return ERRsize; /* they must have reasonable size */

   while (size--) { temp = *first; *first++ = *second; *second++ = temp; }

   return NoError;
}

/* =================  Vector scale ========================================== */

/* Scale a vector */
error vecscl (int size, vector result, vector given, double scalar) {
    if (!result || !given)   return ERRnil;  /* Must be vectors */
    if (size < 1)            return ERRsize; /* of reasonable size */

    if (scalar == 1.0)       return NoError; /* Nothing to do. */

    while (size--) *result++ = scalar * (*given++);

    if (scalar == 0.0)       return ERRmeaning; /* It is not meaningful
                                                 * to scale by zero.
                                                 */
    return NoError;
}

/* ========================== Add =========================================== */

/* result = A + B */
error vecadd (int size, vector result, vector A, vector B) {
    if (!result || !A || !B)  return ERRnil;   /* Must have vectors */
    if (size < 1)             return ERRsize;  /* of reasonable size */

    while (size--) { *result++ = *A++ + *B++; }
    return NoError;
}

/* result = A - B */
error vecsub (int size, vector result, vector A, vector B) {
    if (!result || !A || !B)  return ERRnil;   /* Must have vectors */
    if (size < 1)             return ERRsize;  /* of reasonable size */

    while (size--) { *result++ = *A++ - *B++; }
    return NoError;
}


/* Scaled addition of rows and columns. */
/* result = A + B*scale; */
error vecadds (int size, vector result, vector A, vector B, double scale) {
    if (!result || !A || !B)  return ERRnil;   /* Must have vectors  */
    if (size < 1)             return ERRsize;  /* of reasonable size */

    while (size--) { *result++  =  *A++   +   scale * (*B++); }
    if (scale == 0.0) return ERRmeaning; /* Scaling by zero is meaningless */

    return NoError;
}

/* ======================== Multiply ======================================== */
/* Dot product */

error vecdot (int size, double *result, vector A, vector B) {
    double sum;
    if (!result)   return ERRnil;  /* We have to put the result somewhere, */
    if (!A || !B)  return ERRnil;  /* and must have vectors  */
    if (size < 1)  return ERRsize; /* of positive size */

    sum = 0.0;
    while (size--) { sum  +=   *A++   *   *B++ ; }
    *result = sum;

    return NoError;

}

/* ========================================================================== */
