
   #include <math.h>
   #include <sys/time.h>

   #include <hmpi.h>
   #include "HMPI_em3d.h"
   #include "HMPI_recon.c"
   #include "HMPI_em3d_i.c"

   int main(int argc, char **argv) 
   {
      MPI_Comm *gcomm;
      MPI_Comm *em3dcomm;
      HMPI_Group gid;
      int i, j, k, l, me, rc;
      double temp;

      double elapsed_time, barrier_time;
      struct timeval start, end;
      struct timeval startb, endb;

      double *rperf, *rearranged_rperf;

      int* model_params;
      int param_count; 

      /*
       * Inputs to the program
       */
      int p, niter;

      gettimeofday(&start, NULL);

      {
         rc = HMPI_Init(
                  &argc,
                  &argv
         );

         if (rc != HMPI_OK)
         {
            printf(
                "Problems initializing HMPI runtime "
                "...Exiting...\n"
            );

            HMPI_Finalize(-1);
         }
      }

      {
         int hostnamel = HOSTNAMELENGTH;

         gethostname(
            hostName,
            hostnamel
         );
      }

      p = HMPI_Get_number_of_processors();
      niter = HMPI_NUM_ITERATIONS;

      if (HMPI_Is_recon_required)
      {
         if (HMPI_Is_member(HMPI_COMM_WORLD_GROUP)) 
         {
            int output_p;
            Body recon_body;
            Construct_recon_params(RECON_NUM_NODES, &recon_body);
            rc = HMPI_Recon(&Serial_em3d, 
                            &recon_body, 
                            1, 
                            &output_p
            );
 
            if (rc != HMPI_OK)
            {
               printf(
                  "MAIN:Problems reconning the network"
                  "...Exiting...\n"
               );
 
               HMPI_Finalize(-2);
            }

            Destroy_recon_params(RECON_NUM_NODES, &recon_body);
         }

         if (HMPI_Is_host())
         {
            printf("Processor performances refreshed using recon\n");
         }
      }

      /*
       * get performances
       */
      rperf = (double*)malloc(
                       sizeof(double)
                       *
                       p
      );

      rc = HMPI_Get_processors_info(
               rperf
      );

      if (rc != HMPI_OK)
      {
         printf(
             "MAIN:Problems getting the performances"
             "...Exiting...\n"
         );

         HMPI_Finalize(-3);
      }

      temp = rperf[0];

      for (i = 1; i < p; i++)
      {
          if (rperf[i] < temp)
          {
             temp = rperf[i];
          }
      }

      for (i = 0; i < p; i++)
      {
         rperf[i] = ceil(rperf[i]/temp);
      }

      if (HMPI_Is_host())
      {
         printf("Performances are :\n");

         for (i = 0; i < p; i++)
         {
            printf("%0.2f, ", rperf[i]);
         }

         printf("\n");
      }

      gcomm = (MPI_Comm*)HMPI_Get_comm(HMPI_COMM_WORLD_GROUP);

      MPI_Comm_rank(*gcomm, &me);

      if (HMPI_Is_host())
      {
         printf("Initializing system\n");
      }

      rc = Initialize_system(me, p, rperf);

      if (rc != HMPI_OK)
      {
         HMPI_Finalize(-4);
      }

      if (HMPI_Is_host())
      {
         printf("System initialized\n");
      }

      if (HMPI_Is_host())
      {
         int d[p], dep[p*p];

         Construct_model_params(
                  p,
                  rperf,
                  d,
                  dep
         );

         param_count = 1 + 1 + 1 + p + p*p;

         model_params = (int*)malloc(
                              sizeof(int)
                              *
                              param_count
         );

         if (model_params == NULL)
         {
            printf("Can't allocate memory(Model params), hostname is %s\n", hostName);
            HMPI_Finalize(-11);
         }

         model_params[0] = p;
         model_params[1] = HMPI_NUM_ITERATIONS;
         model_params[2] = RECON_NUM_NODES;

         for (i = 0; i < p; i++)
         {
             model_params[1 + 1 + 1 + i] = d[i];
         }

         for (i = 0; i < (p*p); i++)
         {
             model_params[1 + 1 + 1 + p + i] = dep[i];
         }
      }

      if (HMPI_Is_host())
      {
         printf("Creating group\n");

         rc = HMPI_Group_create(
                  &gid, 
                  &MPC_NetType_Em3d,
                  model_params, 
                  param_count
         );

         if (rc != HMPI_OK)
         {
            printf(
               "MAIN:Problems creating the group"
               "...Exiting...\n"
            );

            HMPI_Finalize(-5);
         }
      }

      if (HMPI_Is_free())
      {
         rc = HMPI_Group_create(
                  &gid, 
                  &MPC_NetType_Em3d,
                  NULL,
                  0
         );

         if (rc != HMPI_OK)
         {
            printf(
               "MAIN:Problems creating the group"
               "...Exiting...\n"
            );

            HMPI_Finalize(-5);
         }
      }

      if (HMPI_Is_host())
      {
         printf("Destroying system\n");
      }

      if (HMPI_Is_member(&gid)) 
      {
         Destroy_system(p, me);
      }

      if (HMPI_Is_host())
      {
         printf("System destroyed\n");
      }


      if (HMPI_Is_free())
      {
         free(rearranged_rperf);
         free(rperf);
         free(dependencies);
 
         HMPI_Finalize(0);
      }

      me = HMPI_Group_rank(&gid);

      /*
       * get performances
       */
      rearranged_rperf = (double*)malloc(
                                  sizeof(double)
                                  *
                                  p
      );

      em3dcomm = (MPI_Comm*)HMPI_Get_comm(&gid);

      //
      // The other way to get the performances is to use
      // HMPI_Group_performances
      for (i = 0; i < p; i++)
      {
         int translated_rank;

         Translate_from_rank(
             i,
             em3dcomm,
             &HMPI_COMM_WORLD,
             &translated_rank
         );

         rearranged_rperf[i] = rperf[translated_rank];
      }

      /*
       * print the mapping
       */
      if (HMPI_Is_host())
      {
         printf("New mapping is \n");
      }

      {
         printf(
            "My group rank is %d, group performance is %0.1f, "
            "My global rank is %d, My global perf is %0.1f\n", 
            me,
            rearranged_rperf[me],
            HMPI_Group_rank(HMPI_COMM_WORLD_GROUP),
            rperf[HMPI_Group_rank(HMPI_COMM_WORLD_GROUP)]
         );
      }

      if (HMPI_Is_host())
      {
         printf("Initializing system after rearrangement\n");
      }

      free(dependencies);

      rc = Initialize_system(me, p, rearranged_rperf);

      if (rc != HMPI_OK)
      {
         HMPI_Finalize(-4);
      }

      if (HMPI_Is_host())
      {
         printf("System initialized\n");
      }

      if (HMPI_Is_member(&gid)) 
      {
         int myb, ind, tnum;
         Body* b;

         if (me == 0)
         {                       
            myb = 0;
         }
                
         if (me > 0)
         {    
            myb = 1;
         }                     
             
         for (i = 0; i < niter; i++) 
         {
             /* 
              * Gather the dependency H values 
              */
             for (j = 0, ind = 0; j < p; j++) 
             {
                if (dependencies[me*p + j] == 0)
                {
                   continue;
                }

                b = bodies[ind++];
                tnum = b->nnum;

                for (k = 0; k < tnum; k++)
                {
                    if (b->ntype == E) 
                    {
                       continue; 
                    }

                    if (j == me)
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] != me)
                           {
                              MPI_Status status;
                              rc = MPI_Recv(
                                      &b->values[l], 
                                      1, 
                                      MPI_DOUBLE, 
                                      b->dbodies[l], 
                                      HMPI_MSG_TAG, 
                                      *em3dcomm, 
                                      &status
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems receiving H value"
                                     "...Exiting...\n"
                                  );

                                  HMPI_Finalize(-6);
                              }
                           }
                       }
                    }
                    else
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] == me) 
                           {
                              double value = GetNodalValue(
                                                b->dnodes[l],
                                                bodies[myb]
                              );

                              rc = MPI_Send(
                                      &value, 
                                      1,
                                      MPI_DOUBLE, 
                                      j, 
                                      HMPI_MSG_TAG, 
                                      *em3dcomm
                              );

                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems sending H value"
                                     "...Exiting...\n"
                                  );

                                  HMPI_Finalize(-7);
                              }
                           }
                       }
                    }

                    b = b->next;
                }
             }

             /* 
              * Computations for E nodes 
              */
             b = bodies[myb];
             tnum = b->nnum;

             for (j = 0; j < tnum; j++)
             {
                 if (b->ntype == H) 
                 {
                    continue; 
                 }

                 for (k = 0; k < b->edge_count; k++)
                 {
                     b->value -= (b->values[k])*(b->coeffs[k]);
                 }

                 b = b->next;
             }

             /* 
              * Gather the dependency E values 
              */
             for (j = 0, ind = 0; j < p; j++) 
             {
                if (dependencies[me*p + j] == 0)
                {
                   continue;
                }

                b = bodies[ind++];
                tnum = b->nnum;

                for (k = 0; k < tnum; k++)
                {
                    if (b->ntype == H) 
                    {
                       continue; 
                    }

                    if (j == me)
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] != me)
                           {
                              MPI_Status status;
                              rc = MPI_Recv(
                                      &b->values[l], 
                                      1, 
                                      MPI_DOUBLE, 
                                      b->dbodies[l], 
                                      HMPI_MSG_TAG, 
                                      *em3dcomm, 
                                      &status
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems receiving H value"
                                     "...Exiting...\n"
                                  );

                                  HMPI_Finalize(-8);
                              }
                           }
                       }
                    }
                    else
                    {
                       for (l = 0; l < b->edge_count; l++)
                       {
                           if (b->dbodies[l] == me) 
                           {
                              double value = GetNodalValue(
                                                b->dnodes[l],
                                                bodies[myb]
                              );
                              rc = MPI_Send(
                                      &value, 
                                      1,
                                      MPI_DOUBLE, 
                                      j, 
                                      HMPI_MSG_TAG, 
                                      *em3dcomm
                              );
                              if (rc != MPI_SUCCESS)
                              {
                                  printf(
                                     "MAIN:Problems sending H value"
                                     "...Exiting...\n"
                                  );

                                  HMPI_Finalize(-9);
                              }
                           }
                       }
                    }

                    b = b->next;
                }
             }

             /* 
              * Computations for H nodes 
              */
             b = bodies[myb];
             tnum = b->nnum;

             for (j = 0; j < tnum; j++)
             {
                 if (b->ntype == E) 
                 {
                    continue; 
                 }

                 for (k = 0; k < b->edge_count; k++)
                 {
                     b->value -= (b->values[k])*(b->coeffs[k]);
                 }

                 b = b->next;
             }
         }
      }

      free(rearranged_rperf);
      free(rperf);
      free(dependencies);
 
      if (HMPI_Is_host())
      {
         free(model_params);
      }

      if (HMPI_Is_host())
      {
         printf("Destroying system\n");
      }

      if (HMPI_Is_member(&gid)) 
      {
         Destroy_system(p, me);
      }

      if (HMPI_Is_host())
      {
         printf("System destroyed\n");
      }

      /*
       * Calculate the barrier time for possible use.
       */
      if (HMPI_Is_member(&gid)) 
      {
        double tstart, tend;

        gettimeofday(&startb, NULL);

        for (i = 0; i < BARRIER_ITERATIONS; i++)
        {
           rc = HMPI_Barrier(&gid);

           if (rc != MPI_SUCCESS)
           {
              return rc;
           }
        }

        gettimeofday(&endb, NULL);

        tstart = startb.tv_sec + (startb.tv_usec/pow(10, 6));
        tend = endb.tv_sec + (endb.tv_usec/pow(10, 6));

        /*
         * There is a barrier at the end
         */
        barrier_time = (tend - tstart)/(BARRIER_ITERATIONS+1);
      }

      if (HMPI_Is_member(&gid)) 
      {
         rc = HMPI_Group_free(&gid);

         if (rc != HMPI_OK)
         {
            printf(
               "MAIN:Freeing the group"
               "...Exiting...\n"
            );

            HMPI_Finalize(-10);
         }
      }

      /*
       * Record the final time and record the elapsed time.
       */
      if (HMPI_Is_host())
      {
	 gettimeofday(&end, NULL);
      }

      if (HMPI_Is_host())
      {
         double tstart = start.tv_sec + (start.tv_usec/pow(10, 6));
         double tend = end.tv_sec + (end.tv_usec/pow(10, 6));

         double elapsed_time = (tend
                                - tstart
                                - barrier_time
         );

         printf(
            "Total number of nodes=%d, Total_number_of_dependencies=%d, time(sec)=%0.6f\n",
            Total_number_of_nodes,
            Total_number_of_dependencies,
            elapsed_time
         );
      }

      HMPI_Finalize(0); 
   }

