/*****************************************************************************/
/* mpCexchg - contains send-receive, functions                               */
/* Coded by Dm. Arapov for A.Lastovtsky 1995-1996                            */
/* Release 1.10.96                                                           */
/*****************************************************************************/
/********** mpCexchg.c $Author: kusic $ $Revision: 1.3 $ $State: Exp $ $Date: 1998/01/28 14:13:28 $ $Locker:  $ **************/
#include <stdlib.h>
#include <stdio.h>
#include "mpC.h"
#include <mpi.h>
#include "mpCtags.h"
#include "mpCtype.h"
#include "mpCtree.h"


/******************* MPC_rank_check ******************************************/

Int MPC_rank_check(
  const MPC_Net* net,
  const Int addr)
  {
    if(addr < 0 || addr >= net->power) {
      char mes[50];
      sprintf(mes,"inavlid rank %d",addr);
      MPC_Error_raise(mes);
    }
    return MPC_OK;
  } /* MPC_rank_check */

/******************* iMPC_Send ***********************************************/  
Int iMPC_Send(
  const MPC_Net* net,    /* Net, both source and destination must belong IN */
  const void* message,   /* Message to send */
  MPC_Datatype datatype, /* Message datatype */
  Int dest)              /* rank in the net of the message destination */
  {
    Int RC;
    MPC_Rts_datatype type;
    MPI_Datatype mpi_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    type = MPC_Get_datatype(datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = MPC_rank_check(net,dest);
    RC = I2C(MPI_Send((void*)message,1,mpi_type,dest,MPC_MSG_TAG,(*(MPI_Comm*)(net->pweb))));
    if (RC!=MPC_OK) return RC;
    return MPC_OK;
  } /* iMPC_Send */

/******************* MPC_Send ************************************************/  
Int MPC_Send(
  const MPC_Net* net,    /* Net, both source and destination must belong IN */
  const void* message,   /* Message to send */
  MPC_Datatype datatype, /* Message datatype */
  Int dest)              /* rank in the net of the message destination */
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG){
      MPC_Debug_printf( "---[%2d]-> MPC_Send, rank = %d, dest = %d\n", MPC_Net_global.rank, net->rank, dest );
    } /* if */
    RC = iMPC_Send(net,message,datatype,dest);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
        MPC_Debug_printf( "<--[%2d]-- MPC_Send\n", MPC_Net_global.rank );
    } /* if */
    return RC;    
  } /* MPC_Send */

/******************* iMPC_Recv ***********************************************/  
Int iMPC_Recv(
  const MPC_Net* net,    /* Net, both source and destination must belong */
  void* message,         /* Message to receive */
  MPC_Datatype datatype, /* Message datatype */
  Int source )           /* rank in the net of the message source */
  {
    Int RC;
    MPC_Rts_datatype type;
    MPI_Status status;
    MPI_Datatype mpi_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;    
    type = MPC_Get_datatype(datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = MPC_rank_check(net,source);
    RC=I2C(MPI_Recv(message, 1, mpi_type, source, MPC_MSG_TAG, (*(MPI_Comm*)(net->pweb)), &status));
    if (RC!=MPC_OK) return RC;
    return MPC_OK;
  } /* iMPC_Recv */

/******************* MPC_Recv ************************************************/  
Int MPC_Recv(
  const MPC_Net* net,    /* Net, both source and destination must belong */
  void* message,         /* Message to receive */
  MPC_Datatype datatype, /* Message datatype */
  Int source )           /* rank in the net of the message source */
  {  
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Recv, rank = %d, source = %d\n", MPC_Net_global.rank, net->rank, source );
    } /* if */
    RC = iMPC_Recv(net,message,datatype,source);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Recv\n", MPC_Net_global.rank );
    } /* if */
    return RC;
  } /* MPC_Recv */

/******************* iMPC_Send_copy *******************************************/
Int iMPC_Send_copy(
  const MPC_Net* net,    /* Net, both source and destination must belong IN */
  const void* send_message,   /* Message to send */
  void* recv_message,
  MPC_Datatype datatype, /* Message datatype */
  Int dest)              /* rank in the net of the message destination */
  {
    Int RC;

    if (MPC_Node_rank(net) == dest) {
        RC = MPC_Elem_copy( recv_message, send_message, datatype );
        if (RC != MPC_OK) return RC;
    }
    else {
      RC = iMPC_Send(net, send_message,datatype, dest );
      if (RC != MPC_OK) return RC;
    } /* if */
    return MPC_OK; 
  } /* iMPC_Send_copy */

/******************* MPC_Send_copy ********************************************/
Int MPC_Send_copy(
  const MPC_Net* net,    /* Net, both source and destination must belong IN */
  const void* send_message,   /* Message to send */
  void* recv_message,
  MPC_Datatype datatype, /* Message datatype */
  Int dest)              /* rank in the net of the message destination */
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_copy, rank = %d, dest = %d\n", MPC_Net_global.rank, net->rank, dest );
    } /* if */
    RC = iMPC_Send_copy(net,send_message,recv_message,datatype,dest);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_copy\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_copy\n", MPC_Net_global.rank );
    } /* if */
    return RC;    
  } /* MPC_Send_copy */

/******************* iMPC_Recv_copy *******************************************/
Int iMPC_Recv_copy(
  const MPC_Net* net,
  void* message,
  MPC_Datatype datatype,
  Int source )
  {
    int RC;
    if (source != MPC_Node_rank(net)) {
      RC = iMPC_Recv(net,message,datatype,source);
      if (RC != MPC_OK) return RC;
     } /* if */
    return MPC_OK; 
  } /* iMPC_Recv_copy */

/******************* MPC_Recv_copy ********************************************/
Int MPC_Recv_copy(
  const MPC_Net* net,
  void* message,
  MPC_Datatype datatype,
  Int source )
  {
        Int RC;
    char err_str[MAX_ERR_STR];
    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Recv_copy, rank = %d, source = %d\n", MPC_Net_global.rank, net->rank, source );
    } /* if */
    RC = iMPC_Recv_copy(net,message,datatype,source);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv_copy\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Recv_copy\n", MPC_Net_global.rank );
    } /* if */
    return RC;
  } /* MPC_Recv_copy */

/******************* iMPC_Send_all ********************************************/
Int iMPC_Send_all( 
  MPC_Net* net,    /* Net to broadcast over */
  const void* send_message,   /* Message to broadcast */
  void* recv_message,    /* Message to receive */
  MPC_Datatype datatype) /* Message datatype */
  {
    Int RC;       
    MPC_Rts_datatype type;
    MPI_Datatype mpi_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Change_root(net,net->rank);
    if (RC != MPC_OK) return RC;
    type = MPC_Get_datatype(datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = I2C(MPI_Bcast((void*)send_message,1,mpi_type,net->rank,(*(MPI_Comm*)(net->pweb)))); 
    if (RC != MPC_OK) return RC;
    RC = MPC_Elem_copy( recv_message, send_message, datatype );
    if (RC != MPC_OK) return RC;
    return MPC_OK;
  } /* iMPC_Send_all */

/******************* MPC_Send_all *********************************************/
Int MPC_Send_all( 
  MPC_Net* net,    /* Net to broadcast over */
  const void* send_message,   /* Message to broadcast */
  void* recv_message,    /* Message to receive */ 
  MPC_Datatype datatype) /* Message datatype */
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_all, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Send_all(net,send_message,recv_message,datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_all\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_all\n", MPC_Net_global.rank );
    } /* if */
    return RC; 
  } /* MPC_Send_all */

/******************* iMPC_Recv_all *******************************************/  
Int iMPC_Recv_all( 
  MPC_Net* net,          /* Net, both source and destination must belong */
  void* message,         /* broadcasted message */
  MPC_Datatype datatype, /* Message datatype, must be same as in MPC_Msg_send_all */
  Int source )           /* rank of broadcating node */
  {
    Int RC;
    MPC_Rts_datatype type;
    MPI_Datatype mpi_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Change_root(net,source);
    if (RC != MPC_OK) return RC;
    type = MPC_Get_datatype(datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = MPC_rank_check(net,source);
    RC = I2C(MPI_Bcast(message,1,mpi_type,source,(*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;
    return MPC_OK;
  } /* iMPC_Recv_all */                           

/******************* MPC_Recv_all ********************************************/ 
Int MPC_Recv_all( 
  MPC_Net* net,          /* Net, both source and destination must belong */
  void* message,         /* broadcasted message */
  MPC_Datatype datatype, /* Message datatype, must be same as in MPC_Msg_send_all */
  Int source )           /* rank of broadcating node */
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Recv_all, rank = %d, source = %d\n", MPC_Net_global.rank, net->rank, source );
    } /* if */
    RC = iMPC_Recv_all(net,message,datatype,source);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv_all\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Recv_all\n", MPC_Net_global.rank );
    } /* if */
    return RC;     
  } /* MPC_Recv_all */                           

/******************* iMPC_Send_scatter ****************************************/  
Int iMPC_Send_scatter( 
  MPC_Net* net,               /* Net, both source and destination must belong */
  const void* send_message,   /* address of the whole message (all parts) */
  void* recv_message,         /* place for local part of message */
  MPC_Datatype datatype)      /* Message datatype */                             
  {
    Int RC;
    Int* counts;
    Int* displs;
    MPC_Rts_datatype elem_type;
    Int step;
    Int count;
    Int i;
    MPI_Datatype mpi_elem_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Change_root(net,net->rank);
    if (RC != MPC_OK) return RC;
    elem_type = MPC_Get_element_type( datatype );
    mpi_elem_type = *(MPI_Datatype*)(elem_type);
    count = MPC_Get_count( datatype );
    if ((count != MPC_UNDEFINED_COUNT) && (count != net->power)) return MPC_ERR_COUNT;
    if (count == MPC_UNDEFINED_COUNT) count = net->power;
    step  = MPC_Get_step( datatype );
    if ((counts=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    if ((displs=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    for (i=0;i<count;i++) {
      counts[i] = 1;
      displs[i] = i*step;
    } /* for */           
    RC = I2C(MPI_Scatterv((void*)send_message, counts, displs, mpi_elem_type, recv_message, 1, mpi_elem_type, net->rank, (*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;                      
    free(counts);
    free(displs);
    return MPC_OK;
  } /* iMPC_Send_scatter */

/******************* MPC_Send_scatter ****************************************/
Int MPC_Send_scatter( 
  MPC_Net* net,               /* Net, both source and destination must belong */
  const void* send_message,   /* address of the whole message (all parts) */
  void* recv_message,         /* place for local part of message */
  MPC_Datatype datatype)      /* Message datatype */                             
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_scatter, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Send_scatter(net,send_message,recv_message,datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_scatter\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_scatter\n", MPC_Net_global.rank );
    } /* if */
    return RC; 
  } /* MPC_Send_scatter */  

/******************* iMPC_Recv_scatter ***************************************/    
Int iMPC_Recv_scatter( 
  MPC_Net* net,               /* Net, both source and destination must belong */
  void* message,              /* address of destination buffer */
  MPC_Datatype datatype,      /* Message datatype, must be same as in MPC_Msg_send_scatter */
  Int source )                /* rank of scattering node */
  {
    Int RC;
    MPC_Rts_datatype elem_type;
    MPI_Datatype mpi_elem_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Change_root(net,source);
    if (RC != MPC_OK) return RC;
    elem_type = MPC_Get_datatype( datatype );
    mpi_elem_type = *(MPI_Datatype*)(elem_type);
    RC = MPC_rank_check(net,source);
    RC = I2C(MPI_Scatterv(NULL,NULL,NULL,mpi_elem_type,message,1,mpi_elem_type,source,(*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;
    return MPC_OK;
  } /* iMPC_Recv_scatter */

/******************* MPC_Recv_scatter ****************************************/    
Int MPC_Recv_scatter( 
  MPC_Net* net,               /* Net, both source and destination must belong */
  void* message,              /* address of destination buffer */
  MPC_Datatype datatype,      /* Message datatype, must be same as in MPC_Msg_send_scatter */
  Int source )                /* rank of scattering node */
    {
      Int RC;
      char err_str[MAX_ERR_STR];

      if (MPC_DEBUG) {
        MPC_Debug_printf( "---[%2d]-> MPC_Recv_scatter, rank = %d, source = %d\n", MPC_Net_global.rank, net->rank, source );
      } /* if */
      RC = iMPC_Recv_scatter(net,message,datatype,source);
      if (RC != MPC_OK) {        
        RC2STR( err_str, RC );
        MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv_scatter\n", MPC_Net_global.rank, err_str );
        exit(RC);
      } /* if */
      if (MPC_DEBUG) {
        MPC_Debug_printf( "<--[%2d]-- MPC_Recv_scatter\n", MPC_Net_global.rank );
      } /* if */
      return RC; 
    } /* MPC_Recv_scatter */

/******************* iMPC_Send_gather ****************************************/
Int iMPC_Send_gather( 
  const MPC_Net* net,          /* Net, both source and destination must belong */
  const void* send_message,    /* address of the local part of the message */
  MPC_Datatype datatype,       /* Message datatype, must be same as in MPC_Msg_recv_gather */
  Int dest )                   /* number of the node in the net to send the message */
  {
    Int RC;
    MPC_Rts_datatype elem_type;
    MPI_Datatype mpi_elem_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;                                        
    elem_type = MPC_Get_datatype( datatype );                                       
    mpi_elem_type = *(MPI_Datatype*)(elem_type);
    RC = MPC_rank_check(net,dest);
    RC = I2C(MPI_Gatherv((void*)send_message,1,mpi_elem_type,NULL,NULL,NULL,mpi_elem_type,dest,(*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK ) return RC;
    return MPC_OK;
  } /* iMPC_Send_gather */                   

/******************* MPC_Send_gather *****************************************/
Int MPC_Send_gather( 
  const MPC_Net* net,          /* Net, both source and destination must belong */
  const void* send_message,    /* address of the local part of the message */
  MPC_Datatype datatype,       /* Message datatype, must be same as in MPC_Msg_recv_gather */
  Int dest )                   /* number of the node in the net to send the message */
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_gather, rank = %d, dest = %d\n", MPC_Net_global.rank, net->rank, dest );
    } /* if */
    RC = iMPC_Send_gather(net,send_message,datatype,dest);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_gather\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_gather\n", MPC_Net_global.rank );
    } /* if */
    return RC; 
  } /* MPC_Send_gather */                   
  
/******************* iMPC_Recv_gather ****************************************/ 
Int iMPC_Recv_gather( 
  const MPC_Net* net,          /* Net, both source and destination must belong */
  const void* send_message,    /* Local part of the message */         
  void* recv_message,
  MPC_Datatype datatype )         
  {
    Int RC;
    Int* counts;
    Int* displs;
    MPC_Rts_datatype elem_type;
    MPI_Datatype mpi_elem_type;
    Int step;
    Int count;
    Int i;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    elem_type = MPC_Get_element_type( datatype );
    mpi_elem_type = *(MPI_Datatype*)(elem_type);
    count = MPC_Get_count( datatype );
    if ((count != MPC_UNDEFINED_COUNT) && (count != net->power)) return MPC_ERR_COUNT;
    if (count == MPC_UNDEFINED_COUNT) count = net->power;
    step  = MPC_Get_step( datatype );
    if ((counts=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    if ((displs=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    for (i=0;i<count;i++) {
      counts[i] = 1;
      displs[i] = i*step;
    } /* for */  
    RC = I2C(MPI_Gatherv( (void*)send_message,1,mpi_elem_type,recv_message,counts,displs,mpi_elem_type,net->rank,(*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;                     
    return MPC_OK;
  } /* iMPC_Recv_gather */ 

/******************* MPC_Recv_gather *****************************************/                                                                     Int MPC_Recv_gather( 
  const MPC_Net* net,          /* Net, both source and destination must belong */
  const void* send_message,    /* Local part of the message */         
  void* recv_message,
  MPC_Datatype datatype )         
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Recv_gather, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Recv_gather(net,send_message,recv_message,datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv_gather\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Recv_gather\n", MPC_Net_global.rank );
    } /* if */
    return RC;  
  } /* MPC_Recv_gather */

/******************* iMPC_Send_copy_conv ***************************************/
Int iMPC_Send_copy_conv(
  const MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype,
  Int dest)
  {
    Int RC;
    
    if (MPC_Node_rank(net) == dest) {
      RC = MPC_Elem_copy_conv( recv_message, send_message, recv_datatype, send_datatype );
      if (RC != MPC_OK) return RC;
    }
    else {
      RC = iMPC_Send(net, send_message, send_datatype, dest );
      if (RC != MPC_OK) return RC;
    } /* if */
    return MPC_OK; 
  } /* iMPC_Send_copy_conv */

/******************* MPC_Send_copy_conv ***************************************/
Int MPC_Send_copy_conv(
  const MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype,
  Int dest)
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_copy_conv, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Send_copy_conv(net,send_message,recv_message,send_datatype,recv_datatype,dest);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_copy_conv\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_copy_conv\n", MPC_Net_global.rank );
    } /* if */
    return RC;  
  } /* MPC_Send_copy_conv */

/******************* iMPC_Send_all_conv ***************************************/
Int iMPC_Send_all_conv(
  MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype)
  {
    Int RC;       
    MPC_Rts_datatype type;
    MPI_Datatype mpi_type;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Elem_copy_conv( recv_message, send_message, recv_datatype,send_datatype );
    if (RC != MPC_OK) return RC;
    RC = MPC_Change_root(net, net->rank);
    if (RC != MPC_OK) return RC;
    type = MPC_Get_datatype(recv_datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = I2C(MPI_Bcast((void*)recv_message,1,mpi_type,net->rank,(*(MPI_Comm*)(net->pweb)))); 
    return MPC_OK;
  } /* iMPC_Send_all_conv */

/******************* MPC_Send_all_conv ***************************************/
Int MPC_Send_all_conv(
  MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype)
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_all_conv, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Send_all_conv(net,send_message,recv_message,send_datatype,recv_datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_all_conv\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_all_conv\n", MPC_Net_global.rank );
    } /* if */
    return RC;
  } /* MPC_Send_all_conv */

/******************* iMPC_Send_scatter_conv ***************************************/
Int iMPC_Send_scatter_conv(
  MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype )
  {
    Int RC;
    MPC_Datatype send_elem_type;
    MPC_Datatype recv_type;
    MPI_Datatype mpi_send_elem_type;
    MPI_Datatype mpi_recv_type;
    Int count;
    Int step;
    Int* counts;
    Int* displs;
    Int i;

    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    RC = MPC_Change_root(net, net->rank);
    if (RC != MPC_OK) return RC;
    send_elem_type = MPC_Get_element_type( send_datatype );
    mpi_send_elem_type = *(MPI_Datatype*)(send_elem_type);
    recv_type = MPC_Get_datatype(recv_datatype);
    mpi_recv_type = *(MPI_Datatype*)(recv_type);
    count = MPC_Get_count( send_datatype );
    if ((count != MPC_UNDEFINED_COUNT) && (count != net->power)) return MPC_ERR_COUNT;
    if (count == MPC_UNDEFINED_COUNT) count = net->power;
    step  = MPC_Get_step( send_datatype );
    if ((counts=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    if ((displs=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    for (i=0;i<count;i++) {
      counts[i] = 1;
      displs[i] = i*step;
    } /* for */           
    RC = I2C(MPI_Scatterv((void*)send_message, counts, displs, mpi_send_elem_type, recv_message, 1, mpi_recv_type, net->rank, (*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;                      
    free(counts);
    free(displs);
    return MPC_OK;
  } /* iMPC_Send_scatter_conv */

/******************* MPC_Send_scatter_conv ***************************************/
Int MPC_Send_scatter_conv(
  MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype )
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Send_scatter_conv, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Send_scatter_conv(net,send_message,recv_message,send_datatype,recv_datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Send_scatter_conv\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Send_scatter_conv\n", MPC_Net_global.rank );
    } /* if */
    return RC;
  } /* MPC_Send_scatter_conv */

/******************* iMPC_Recv_gather_conv ***************************************/
Int iMPC_Recv_gather_conv(
  const MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype )
  {
    Int RC;
    Int* counts;
    Int* displs;
    MPC_Rts_datatype elem_type;
    MPC_Rts_datatype send_type;   
    MPI_Datatype mpi_recv_elem_type;
    MPI_Datatype mpi_send_type;
    Int step;
    Int count;
    Int i;
    
    if ((RC = MPC_Test_net(net)) != MPC_OK) return RC;
    elem_type = MPC_Get_element_type( recv_datatype );
    mpi_recv_elem_type = *(MPI_Datatype*)(elem_type);
    send_type = MPC_Get_datatype(send_datatype);
    mpi_send_type = *(MPI_Datatype*)(send_type);    
    count = MPC_Get_count( recv_datatype );
    if ((count != MPC_UNDEFINED_COUNT) && (count != net->power)) return MPC_ERR_COUNT;
    if (count == MPC_UNDEFINED_COUNT) count = net->power;
    step  = MPC_Get_step( recv_datatype );
    if ((counts=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    if ((displs=malloc((int)(sizeof(Int)*count)))==NULL) return MPC_ERR_NOMEM;
    for (i=0;i<count;i++) {
      counts[i] = 1;
      displs[i] = i*step;
    } /* for */  
    RC = I2C(MPI_Gatherv( (void*)send_message,1,mpi_send_type,recv_message,counts,displs,mpi_recv_elem_type,net->rank,(*(MPI_Comm*)(net->pweb))));
    if (RC != MPC_OK) return RC;                     
    return MPC_OK;
  } /* MPC_Recv_gather_conv */ 

/******************* MPC_Recv_gather_conv ***************************************/
Int MPC_Recv_gather_conv(
  const MPC_Net* net,
  const void* send_message,
  void* recv_message,
  MPC_Datatype send_datatype,
  MPC_Datatype recv_datatype )
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Recv_gather_conv, rank = %d\n", MPC_Net_global.rank, net->rank );
    } /* if */
    RC = iMPC_Recv_gather_conv(net,send_message,recv_message,send_datatype,recv_datatype);
    if (RC != MPC_OK) {        
      RC2STR( err_str, RC );
      MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Recv_gather_conv\n", MPC_Net_global.rank, err_str );
      exit(RC);
    } /* if */
    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Recv_gather_conv\n", MPC_Net_global.rank );
    } /* if */
    return RC;
  } /* MPC_Recv_gather_conv */ 

/******************** MPC_Elem_copy_conv *************************************/
Int MPC_Object_copy_conv( void* dest, const void* source, MPC_Datatype recv_datatype, MPC_Datatype send_datatype);

Int MPC_Elem_copy_conv( void* dest, const void* source, MPC_Datatype recv_datatype, MPC_Datatype send_datatype)
  {
    Int RC;
    /**/
    if (MPC_DEBUG) {
      MPC_Debug_printf( "---[%2d]-> MPC_Object_copy_conv\n", MPC_Net_global.rank );
    }

    RC = MPC_Object_copy_conv(dest,source,recv_datatype,send_datatype);

    if (MPC_DEBUG) {
      MPC_Debug_printf( "<--[%2d]-- MPC_Object_copy_conv\n", MPC_Net_global.rank );
    }
    return RC;
    /**-/
    MPC_Rts_datatype type;
    MPI_Datatype mpi_type;
    MPI_Status status;
    MPI_Request req;
    
    type = MPC_Get_datatype(send_datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = I2C(MPI_Isend(source,1,mpi_type,0,MPC_MSG_TAG,MPI_COMM_SELF,&req)); 
    if (RC != MPC_OK) return RC;
    type = MPC_Get_datatype(recv_datatype);
    mpi_type = *(MPI_Datatype*)(type);
    RC = I2C(MPI_Recv(dest,1,mpi_type,0,MPC_MSG_TAG,MPI_COMM_SELF,&status)); 
      
    return MPC_OK;
/**/
  } /* MPC_Elem_copy_conv */

/******************** MPC_Object_copy_conv *************************************/
Int MPC_Object_copy_conv( void* dest, const void* source, MPC_Datatype recv_datatype, MPC_Datatype send_datatype){

  if((send_datatype->Kind != recv_datatype->Kind)||
     (recv_datatype == NoMPC_Type)||(recv_datatype == NoMPC_Type))
                                                   return( MPC_ERR_TYPE );
  if(send_datatype->Kind != recv_datatype->Kind) {
    MPC_Debug_printf("ERROR: MPC_Object_copy_conv: types of nodes are not equal!");
    return( MPC_ERR_TYPE );
  }

  if((recv_datatype->Base.blindCopy==send_datatype->Base.blindCopy)&&
     (recv_datatype->Base.blindCopy==1)) {
    if(dest!=source)
      {
        if (MPC_DEBUG) MPC_Debug_printf( "---[%2d]-- MPC_Object_copy_conv BLIND\n", MPC_Net_global.rank);
        memmove(dest,source,recv_datatype->Base.Size);
      }
    return MPC_OK;
  }

  if (MPC_DEBUG) {
    MPC_Debug_printf( "---[%2d]-- MPC_Object_copy_conv recvKind=%d\n", MPC_Net_global.rank, recv_datatype->Kind);
  }
  
  switch (recv_datatype->Kind) {

    /*  case  kMPC_Type:*/
  case  kMPC_Basic:
  case  kMPC_Struct:
    memmove(dest,source,recv_datatype->Base.Size);
    break;
    /*  case  kMPC_Enum:*/
    /*  case  kMPC_Typedef:*/
    /*  case  kMPC_BitField:*/
  case  kMPC_Array:
    {
      Int send_step;
      Int recv_step;
      Int send_count;
      Int recv_count;
      char *send_offset;
      char *recv_offset;
      MPC_Datatype elem_type;
      Int elem_size;
      Int i;

      send_count = MPC_Get_count(send_datatype);
      recv_count = MPC_Get_count(recv_datatype);
      if (send_count != recv_count) return( MPC_ERR_TYPE );
      send_offset = 0;
      recv_offset = 0;
      send_step = MPC_Get_step(send_datatype);
      recv_step = MPC_Get_step(recv_datatype);
      elem_type = MPC_Get_element(send_datatype);
      elem_size = MPC_Type_size(elem_type);

      if (elem_type->Kind==kMPC_Basic||elem_type->Kind==kMPC_Struct){
        /* simple element case */
        if (recv_step==send_step&&recv_step==1) {
          /* blind copy case*/
          if(dest!=source){
            memmove(dest,source,recv_datatype->Base.Size);
          }
        } else {
          /* non-blind copy case*/
          recv_offset = (char*)dest;
          send_offset = (char*)source;
          send_step *= elem_size;
          recv_step *= elem_size;
          if ( (recv_offset<=send_offset)&&
               (send_offset<recv_offset+recv_datatype->Base.Size)||
               (send_offset<=recv_offset)&&
               (recv_offset<send_offset+send_datatype->Base.Size) ) {
            /* interlaped */
            char *tbuf,*tbuf_offset;
            if ((tbuf=malloc(elem_size*recv_count))==NULL) return MPC_ERR_NOMEM;
            for (i=0,tbuf_offset=tbuf;i<send_count;
                 i++,tbuf_offset += elem_size,send_offset += send_step) { 
              memmove( tbuf_offset,send_offset,elem_size);
            } /* for */
            
            for (i=0,tbuf_offset=tbuf;i<send_count;
                 i++,tbuf_offset += elem_size,recv_offset += recv_step) { 
              memmove( recv_offset,tbuf_offset,elem_size);
            } /* for */
            
          } else {
            /* non-interlaped */
            for (i=0;i<send_count;
                 i++,recv_offset += recv_step,send_offset += send_step) { 
              memmove( recv_offset,send_offset, elem_size);
            } /* for */
          }
          
        }
        return MPC_OK;
      } else {
        /* compound element case */
        int RC;
        recv_offset = (char*)dest;
        send_offset = (char*)source;
        for (i=0;i<send_count;
             i++,recv_offset += elem_size,send_offset += elem_size) { 

          if (MPC_DEBUG) {
            MPC_Debug_printf( "---[%2d]-- MPC_Object_copy_conv recv=%d send=%d\n", MPC_Net_global.rank, recv_offset,send_offset);
          }
          RC=MPC_Object_copy_conv((void*)recv_offset,
                                  (void*)send_offset,
                                  MPC_Get_element(recv_datatype),
                                  MPC_Get_element(send_datatype));

          if (MPC_DEBUG) {
            MPC_Debug_printf( "---[%2d]-- MPC_Object_copy_conv recv=%d send=%d\n", MPC_Net_global.rank, recv_offset,send_offset);
          }
          
          if (RC != MPC_OK) return RC;
        } /* for */

      }
    }
    break;
    /*  case  kMPC_Pointer:*/
    /*  case  kMPC_Union:*/
    /*  case  kMPC_Function:*/
    /*  case  kMPC_List:*/
  default:
    MPC_Debug_printf( "---[%2d]-- ERROR: MPC_Object_copy_conv recvKind=%d: Sorry, not realized yet!\n", MPC_Net_global.rank, recv_datatype->Kind);
  }
  return MPC_OK;
} /* MPC_Object_copy_conv */
