
#include "mpc_adv.h"
#include "mpc_diag.h"
#include "options.h"
#include "Tree.h"

#include <stdio.h>
#include <string.h>

#define  MAX_COM	128
#define  MAX_EXPR	32

#define  QUIT		1
#define  WRITE		2
#define  READ		3

#define  FULL		1
#define  PART		0


extern char *CURR_ADVICE_FILE;
extern tPosition ADVICE_POS;

 /* The following declarations will be used later by the integrated version: */
static int QUIET = NO,	  /* if YES, don't write anything except position    */	/* - */
	   COMMENT = YES, /* if YES, write comments like "unary expression"  */	/* +- */
	   RECOMPILE = NO;


static tTree	Locate		ARGS((tString File, tPosition Pos));		/* + */
static tTree	NextGlob	ARGS((tTree Tree));				/* - */
static tTree	NextGlobInFile	ARGS((tTree Tree));				/* - */

static void	Diagnose	ARGS((tTree Mess, int Mode));			/* (+-) */
static void	MoreInfo	ARGS((tTree Mess));				/* (-) */
static void	Investigate	ARGS((tTree Tree));				/* + */

static void	Write_MPC	ARGS((tTree Tree, int Mode));			/* (+-) */

static void	Write_Class	ARGS((tTree Tree));				/* + */
static void	Write_Declarator ARGS((tIdent Ident, tTree Type, tTree Distr));	/* + */
static void	Write_MPC_Type	ARGS((tTree Type, int Mode));			/* + */
static void	Write_Distribution ARGS((tTree Tree));				/* +- */
static void	Write_Condition	ARGS((tTree Expr, int Mode));			/* + */
static void	Write_NestedStat ARGS((tTree Stat));				/* + */
static void	Write_OpCode	ARGS((int OpCode));				/* + */
static int	Is_PostfixOperator ARGS((int OpCode));				/* + */
static void	Write_NodeType	ARGS((tTree Node));				/* + */

static int	PrintableSize = MAX_EXPR;  /* max printable control expression, bytes; */
static int	Is_Printable	ARGS((tTree Tree));				/* -+ */

static int	PROMPT_MODE = PART;			/* may be FULL or PART */
static void	Write_Prompt	ARGS((tTree Tree));				/* + */

static char	Command[MAX_COM];
static int	ComLength = -1;

static bool	FileIs		ARGS((tString File1, tString File2));		/* + */
			/* -- compare two file names (also including path); */
static bool	CommandIs	ARGS((tString Pattern));			/* + */
                	/* -- compare "Command" to "Pattern"; */
static void	Store_Location	ARGS((tPosition Pos));				/* + */

/** Stack ('Before') and Queue ('After') functions for Write_Declarator **/
static void	PushBefore	ARGS((tTree Decl));
static void	WriteBefore	ARGS((void));
static void	PushAfter	ARGS((tTree Decl));
static void	WriteAfter	ARGS((void));

static tTree	TerminalType	ARGS((tTree Type));

static int	IsGlobal = YES; /* -- for the current declaration; */



/************/
int mpc_Advice
/************/
#if defined __STDC__ | defined __cplusplus
	(int Mode, tString File, tPosition Pos)
#else
	(Mode, File, Pos)
	int Mode;
	tString File;
	tPosition Pos;
#endif
{
  tTree Root;

  RECOMPILE = NO;

  if(!QUIET) {
    (void)printf("\nmpC Advisor: trying to explore the file %s ", File);
    if(!Compare(Pos, NoPosition) || Pos.Line==0)
      (void)printf("from beginning...\n\n");
    else {
      (void)printf("at position (");
      WritePosition(stdout,Pos);
      (void)printf(")...\n\n");
    }
  }

  switch(Mode) {
    case INTERACT_MODE:
      Root = Locate(File,Pos);
      if(Root==NoTree) break;
      Investigate(Root);
      break;
    case ERRINTER_MODE:
      Diagnose(TreeRoot->MPC_Root.Messages, ERRINTER_MODE);
  /***fprintf(stderr,"\nmpC Advisor: Sorry, mode 'E' is currently not supported.\n\n");***/
      break;
    case DIAGNOST_MODE:
      fprintf(stderr,"\nmpC Advisor: Sorry, mode 'D' is currently not supported.\n\n");
      break;
    default:
      break;
  }

  return RECOMPILE;
}

/*****************/
static tTree Locate
/*****************/
#if defined __STDC__ | defined __cplusplus
	(tString File, tPosition Pos)
#else
	(File, Pos)
	tString File;
	tPosition Pos;
#endif
{
  tTree node;
  tString src;
  tPosition p;

  node = TreeRoot->MPC_Root.Decls;
  if(node==NoTree || node->Kind==kMPC_FreeNode) {
    fprintf(stderr, "\nmpC Advisor: generated program is empty.\n\n");
    return NoTree;
  }
  src  = SourceFile(node->MPC_Decls.Pos);
  p = RealPosition(node->MPC_Decls.Pos);

  while( !( FileIs(File,src) &&
	    (Compare(p,Pos)>0 || !Compare(p,Pos)) ) ) {
    if(node->Kind==kMPC_FreeNode || node->MPC_Decls.Next->Kind==kMPC_FreeNode) break;
    node = node->MPC_Decls.Next;
    src  = SourceFile(node->MPC_Decls.Pos);
    p = RealPosition(node->MPC_Decls.Pos);
  }

  if(!FileIs(File,src)) {
    (void)fprintf(stderr, "mpC Advisor: can't locate line %d in the file '%s'.\n"
		  "(no such line in this file or no such file in compilation)\n\n",
		  Pos.Line, File);
  }
  if(!QUIET) {
    (void)printf("mpC Advisor: exploring the file %s ", src);
    if(!Compare(p, NoPosition) || p.Line==0)
      (void)printf("from beginning\n\n");
    else {
      (void)printf("at position (");
      WritePosition(stdout,p);
      (void)printf(")\n\n");
    }
  }

  return node;
}

/*******************/
static tTree NextGlob
/*******************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{}

/*************************/
static tTree NextGlobInFile
/*************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{}


static int State;  /* QUIT, READ, WRITE; */

/******************/
static void Diagnose
/******************/
# if defined __STDC__ | defined __cplusplus
 (tTree Mess, int Mode)
# else
 (Mess, Mode) tTree Mess;
# endif
{
  char Ch;
  tTree PrevMess=NoTree, NextMess=NoTree;
  int Save_State;

  if(Mess==NoTree || Mess->Kind==kMPC_FreeNode) {
    (void)printf("\n   -- No diagnostics --\n\n");
    return;
  }

  PrevMess = Mess->MPC_Messages.Prev;

  State = WRITE;
  for (;;) {
    switch (State) {
      case QUIT:	return;
      case WRITE:
	{
	  bool More_Messages=true;
	  int ErrorClass;

	  (void)fprintf(stderr, "\nMessage(s) generated for source file '%s'\n\n",
			SourceFile(Mess->MPC_Messages.Pos));
	  NextMess = Mess;
	  while(More_Messages && NextMess!=NoTree && NextMess->Kind!=kMPC_FreeNode) {
	    Print_Message(NextMess);
	    NextMess = NextMess->MPC_Messages.Next;
	    if(NextMess==NoTree || NextMess->Kind==kMPC_FreeNode)
	      More_Messages = false;
	    else {
	    ErrorClass = NextMess->MPC_Messages.ErrorClass;
	      switch(ErrorClass) {
	        case xxFatal:
	        case xxRestriction:
	        case xxError:
	        case xxWarning:
		  More_Messages = false;
	      }
	    }
	  } /* end while */
	}
	State = READ;
      case READ:
	Write_Prompt(Mess);
	ComLength = -1; Ch = getc (stdin);
	while (Ch!='\n' && Ch>0) {
	  Command[++ComLength] = Ch;
	  Ch = getc (stdin);
	}
	if (Ch<0) {
	  (void)fputs ("mpC Advisor: eof reached\n", stderr);
	  State = QUIT;
	  return;
	}
	if(CommandIs("back") || CommandIs("up")) { State = WRITE; return; }
	if(CommandIs("quit"))
	  { State = QUIT; return; }
	if(CommandIs("recompile"))
	  { State = QUIT; RECOMPILE = YES; return; }
	if(CommandIs("prompt"))
	  if(PROMPT_MODE==FULL)
	    PROMPT_MODE = PART;
	  else
	    PROMPT_MODE = FULL;
	if(CommandIs("help")) {  /*** tmp - Quick Reference: ***/
	  (void)printf("\nThe following commands are everywhere legal in aE mode:\n");
	  (void)printf("-------------------------------------------------------\n");
	  (void)printf("help      - get this quick reference\n");
	  (void)printf("            (full HELP is currently not available);\n");
	  (void)printf("prompt    - toggle prompt mode;\n");
	  (void)printf("            (all commands / only message specific commands);\n");
	  (void)printf("back      - return to previous message;\n");
	  (void)printf("quit      - exit from mpC Advisor;\n");
	  (void)printf("recompile - recall the mpC compiler for the same source\n");
	  (void)printf("            with the same options;\n");
	  (void)printf("\nPrompt format\n");
	  (void)printf("-------------\n");
	  (void)printf("Every prompt line contains a set of commands.\n");
	  (void)printf("Some commands are everywhere legal, the others are message\n");
	  (void)printf("specific and may be used to get more information and to\n");
	  (void)printf("explore the subtree, for which is this message generated.\n");
	  (void)printf("    --------------------------------------------------\n\n");
	}
	if(Mess!=NoTree) {
	  switch(Mess->Kind) {
	    case kMPC_ErrorMessage:
	    case kMPC_ErrorMessageI:
		if(CommandIs("next"))		Diagnose(NextMess,Mode);
		if(CommandIs("prev"))	Diagnose(PrevMess,Mode);
		break;
	    case kMPC_Message:
	    case kMPC_MessageI:
		if(CommandIs("next"))			Diagnose(NextMess,Mode);
		if(CommandIs("prev"))		Diagnose(PrevMess,Mode);
		if(CommandIs("info"))		MoreInfo(Mess);
		if(CommandIs("investigate")) {
		  Save_State = State;
		  Investigate(Mess->MPC_Messages.Tree);
		  if(State==QUIT)
		    return;
		  else
		    State = Save_State;
		}
		break;
          } /* end switch(Tree->Kind) */
	} /* end else if */

    } /* end switch(State) */
  } /* end for(;;) */

}

/******************/
static void MoreInfo
/******************/
# if defined __STDC__ | defined __cplusplus
 (tTree Mess)
# else
 (Mess) tTree Mess;
# endif
{
  (void)printf("\nSorry, more information is currently not available.\n\n");
}

/*********************/
static void Investigate
/*********************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{
  char Ch;

  if(Tree==NoTree || Tree->Kind==kMPC_FreeNode) {
    (void)printf("\n   -- Nothing to investigate here --\n\n");
    return;
  }

  State = WRITE;
  for (;;) {
    switch (State) {
      case QUIT:	return;
      case WRITE:
	Write_MPC(Tree,FULL); State = READ;
      case READ:
	Write_Prompt(Tree);
	ComLength = -1; Ch = getc (stdin);
	while (Ch!='\n' && Ch>0) {
	  Command[++ComLength] = Ch;
	  Ch = getc (stdin);
	}
	if (Ch<0) {
	  (void)fputs ("mpC Advisor: eof reached\n", stderr);
	  State = QUIT;
	  return;
	}
	if(CommandIs("back") || CommandIs("up")) { State = WRITE; return; }
	if(CommandIs("quit"))
	  { State = QUIT; return; }
	if(CommandIs("recompile"))
	  { State = QUIT; RECOMPILE = YES; return; }
	if(CommandIs("prompt"))
	  if(PROMPT_MODE==FULL)
	    PROMPT_MODE = PART;
	  else
	    PROMPT_MODE = FULL;
	if(CommandIs("comment"))
	  if(COMMENT==YES)
	    (void)printf("   comment mode set to NO\n"),
	    COMMENT = NO;
	  else
	    (void)printf("   comment mode set to YES\n"),
	    COMMENT = YES;
	if(CommandIs("help")) {  /*** tmp - Quick Reference: ***/
	  (void)printf("\nThe following commands are everywhere legal:\n");
	  (void)printf("--------------------------------------------\n");
	  (void)printf("help      - get this quick reference\n");
	  (void)printf("            (full HELP is currently not available);\n");
	  (void)printf("prompt    - toggle prompt mode;\n");
	  (void)printf("            (all commands / only node specific commands);\n");
	  (void)printf("comment   - toggle comment mode\n");
	  (void)printf("            (add comment to each printed mpC construction);\n");
	  (void)printf("back      - return to previous node;\n");
	  (void)printf("quit      - exit from mpC Advisor;\n");
	  (void)printf("recompile - recall the mpC compiler for the same source\n");
	  (void)printf("            with the same options;\n");
	  (void)printf("\nPrompt format\n");
	  (void)printf("-------------\n");
	  (void)printf("Every prompt line contains set of commands and set of\n");
	  (void)printf("attributes; attributes are placed in [ ].\n");
	  (void)printf("Some commands are everywhere legal, the others are node\n");
	  (void)printf("specific and may be used to explore the subtree.\n");
	  (void)printf("\nAttributes\n");
	  (void)printf("----------\n");
	  (void)printf("[type] attributes are printed in C declaration style;\n");
	  (void)printf("[storenet] of expression (distribution of value) is printed\n");
	  (void)printf("as a net or subnet in mpC style: [host], [*], [net1], etc.\n");
	  (void)printf("[evalnet] of expression and [distribution] of statement are\n");
	  (void)printf("printed as set of nets or subnets: { [net1], [net2], ... }\n");
	  (void)printf("    --------------------------------------------------\n\n");
	}
/***  { fprintf(stderr, "\nmpC Advisor: Sorry, 'help' is currently not available\n\n"); } ***/
	if(Tree!=NoTree) {
	  switch(Tree->Kind) {
	    case kMPC_Root:
	    case kMPC_Source:
	    case kMPC_ErrorMessage:
	    case kMPC_ErrorMessageI:
	    case kMPC_Message:
	    case kMPC_MessageI:
	    case kMPC_BasicType:		/*** <<< ??? ***/
	    case kMPC_ArrayType:
	    case kMPC_VectorType:
	    case kMPC_PointerType:
	    case kMPC_FunctionType:
	    case kMPC_TopoFunctions:
	    case kMPC_EnumConst:
	    case kMPC_SU_MemberDecl:
	    case kMPC_SU_Member:		/*** ??? >>> ***/
	    /* ... */
		fprintf(stderr,"mpC Advisor: Invalid kind of tree node\n");
		break;
	    case kMPC_EnumType:
	    case kMPC_Typedef:
	    case kMPC_StructType:
	    case kMPC_UnionType:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		break;
	    case kMPC_VarDecl:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("var"))	Investigate(Tree->MPC_VarDecl.Var);
		break;
	    case kMPC_Function:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("param")) {
		  IsGlobal = NO;
		  Investigate(Tree->MPC_Function.ParamList);
		  IsGlobal = YES;
		}
		else if(CommandIs("body")) {
		  IsGlobal = NO;
		  Investigate(Tree->MPC_Function.Stats);
		  IsGlobal = YES;
		}
		else if(CommandIs("type"))	Write_MPC(Tree->MPC_Function.Type, FULL);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Function.Distribution);
		break;
	    case kMPC_Ellipsis:
		if(CommandIs("prev"))		Investigate(Tree->MPC_Decls.Prev);
		break;
	    case kMPC_NetDecl:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("topology"))	Investigate(Tree->MPC_NetDecl.NetTypeSpecifier);
		else if(CommandIs("net"))	Investigate(Tree->MPC_NetDecl.Net);
		break;
	    case kMPC_SubnetDecl:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("subnet"))	Investigate(Tree->MPC_SubnetDecl.Subnet);
		break;
	    case kMPC_RelDecl:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("relation"))	Investigate(Tree->MPC_RelDecl.Relation);
		break;
	    case kMPC_NetType:
		if(CommandIs("next"))		Investigate(Tree->MPC_Decls.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Decls.Prev);
		else if(CommandIs("param"))	Investigate(Tree->MPC_NetType.ParamList);
		else if(CommandIs("coord"))	Investigate(Tree->MPC_NetType.CoordDecl);
		else if(CommandIs("node"))	Investigate(Tree->MPC_NetType.NodeDecl);
		else if(CommandIs("link"))	Investigate(Tree->MPC_NetType.LinkDecl);
		else if(CommandIs("parent"))	Investigate(Tree->MPC_NetType.StartDecl);
		break;
	    case kMPC_NetTypeSpecifier:
		if(CommandIs("nettype"))	Investigate(Tree->MPC_NetTypeSpecifier.NetType);
		else if(CommandIs("arguments")) Investigate(Tree->MPC_NetTypeSpecifier.ArgList);
		break;
	    case kMPC_Var:
		if(CommandIs("next"))		Investigate(Tree->MPC_Var.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Var.Prev);
		else if(CommandIs("type"))	Write_MPC(Tree->MPC_Var.Type,FULL);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Var.Distribution);
		break;
	    case kMPC_CoordDecl:
		if(CommandIs("next"))		Investigate(Tree->MPC_CoordDecl.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_CoordDecl.Prev);
		else if(CommandIs("range"))	Investigate(Tree->MPC_CoordDecl.Range);
		break;
	    case kMPC_Node:
		if(CommandIs("next"))		Investigate(Tree->MPC_Node.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Node.Prev);
		else if(CommandIs("predicate"))	Investigate(Tree->MPC_Node.Predicate);
		break;
	    case kMPC_LinkDecl:
		if(CommandIs("freecoord"))	Investigate(Tree->MPC_LinkDecl.FreeCoord);
		else if(CommandIs("linklist"))
			Investigate(Tree->MPC_LinkDecl.LinkDeclaringList);
		break;
	    case kMPC_LinkDeclaringList:
		if(CommandIs("next"))		Investigate(Tree->MPC_LinkDeclaringList.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_LinkDeclaringList.Prev);
		else if(CommandIs("predicate"))
			Investigate(Tree->MPC_LinkDeclaringList.Predicate);
		break;
	/** case kMPC_LinkDeclarator: -- ? **/
	    case kMPC_NetList:
		if(CommandIs("next"))		Investigate(Tree->MPC_NetList.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_NetList.Prev);
		else if(CommandIs("net"))	Investigate(Tree->MPC_NetList.Net);
		break;
	    case kMPC_Net:
		if(CommandIs("next"))		Investigate(Tree->MPC_Net.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Net.Prev);
		else if(CommandIs("topology"))	Investigate(Tree->MPC_Net.NetType);
		else if(CommandIs("distribution"))
			Investigate(Tree->MPC_Net.Distribution);
		break;
	    case kMPC_Subnet:
		if(CommandIs("next"))		Investigate(Tree->MPC_Subnet.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Subnet.Prev);
		else if(CommandIs("supernet"))	Investigate(Tree->MPC_Subnet.Distribution);
		else if(CommandIs("predicate"))	Investigate(Tree->MPC_Subnet.Predicate);
		break;
	    case kMPC_Relation:
		if(CommandIs("next"))		Investigate(Tree->MPC_Relation.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Relation.Prev);
		else if(CommandIs("left"))	Investigate(Tree->MPC_Relation.LNet);
		else if(CommandIs("right"))	Investigate(Tree->MPC_Relation.RNet);
		break;
	    case kMPC_ExprStat:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_ExprStat.Expr);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_If:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_If.Expr);
		else if(CommandIs("then"))	Investigate(Tree->MPC_If.Stats);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_IfElse:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_IfElse.Expr);
		else if(CommandIs("then"))	Investigate(Tree->MPC_IfElse.Then);
		else if(CommandIs("else"))	Investigate(Tree->MPC_IfElse.Else);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Switch:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_Switch.Expr);
		else if(CommandIs("body"))	Investigate(Tree->MPC_Switch.Stats);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_While:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_While.Expr);
		else if(CommandIs("body"))	Investigate(Tree->MPC_While.Stats);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_DoWhile:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("body"))	Investigate(Tree->MPC_DoWhile.Stats);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_DoWhile.Expr);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_For:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("init"))	Investigate(Tree->MPC_For.Init);
		else if(CommandIs("cond"))	Investigate(Tree->MPC_For.Cond);
		else if(CommandIs("reinit"))	Investigate(Tree->MPC_For.Reinit);
		else if(CommandIs("body"))	Investigate(Tree->MPC_For.Stats);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Goto:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Continue:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Break:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_BreakFan:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Return:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_Return.Expr);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Compound:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("decls"))	Investigate(Tree->MPC_Compound.Decls);
		else if(CommandIs("stats"))	Investigate(Tree->MPC_Compound.Stats);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_Fan:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stat.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stat.Prev);
		else if(CommandIs("body"))	Investigate(Tree->MPC_Fan.Body);
		else if(CommandIs("distribution")) Investigate(Tree->MPC_Stat.ExecNet);
		break;
	    case kMPC_IdentLabel:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stats.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stats.Prev);
		break;
	    case kMPC_CaseLabel:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stats.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stats.Prev);
		break;
	    case kMPC_Default:
		if(CommandIs("next"))		Investigate(Tree->MPC_Stats.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Stats.Prev);
		break;
	    case kMPC_IntConst:
	    case kMPC_UIntConst:
	    case kMPC_FloatConst:
		if(CommandIs("type"))		Investigate(Tree->MPC_Expr.Type);
	/***	else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet); ***/
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_StringLiteral:
		if(CommandIs("next"))		Investigate(Tree->MPC_StringLiteral.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_StringLiteral.Prev);
	/***	else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type); ***/
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_Ident:
		if(CommandIs("decl"))		Investigate(Tree->MPC_Ident.Store);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
	/***	else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet); ***/
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_CastExpr:
		if(CommandIs("operand"))	Investigate(Tree->MPC_CastExpr.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_NetCastExpr:
		if(CommandIs("operand"))	Investigate(Tree->MPC_NetCastExpr.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_CoordExpr:
		if(CommandIs("operand"))	Investigate(Tree->MPC_CoordExpr.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_Size_Of_Value:
		if(CommandIs("operand"))	Investigate(Tree->MPC_Size_Of_Value.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_Size_Of_Type:
		if(CommandIs("operand"))	Investigate(Tree->MPC_Size_Of_Type.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_UnaryExpr:
		if(CommandIs("operand"))	Investigate(Tree->MPC_UnaryExpr.Operand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_BinaryExpr:
		if(CommandIs("left"))		Investigate(Tree->MPC_BinaryExpr.Loperand);
		else if(CommandIs("right"))	Investigate(Tree->MPC_BinaryExpr.Roperand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_TernaryExpr:
		if(CommandIs("first"))		Investigate(Tree->MPC_TernaryExpr.Foperand);
		else if(CommandIs("second"))	Investigate(Tree->MPC_TernaryExpr.Soperand);
		if(CommandIs("third"))		Investigate(Tree->MPC_TernaryExpr.Toperand);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_CallExpr:
		if(CommandIs("function"))	Investigate(Tree->MPC_CallExpr.Function);
		else if(CommandIs("args"))	Investigate(Tree->MPC_CallExpr.ArgList);
		else if(CommandIs("netargs") && Tree->MPC_CallExpr.NetworkArgList!=NoTree &&
			Tree->MPC_CallExpr.NetworkArgList->Kind!=kMPC_FreeNode)
						Investigate(Tree->MPC_CallExpr.NetworkArgList);
		else if(CommandIs("type"))	Investigate(Tree->MPC_Expr.Type);
		else if(CommandIs("evalnet"))	Investigate(Tree->MPC_Expr.EvalNet);
		else if(CommandIs("storenet"))	Investigate(Tree->MPC_Expr.StoreNet);
		break;
	    case kMPC_Exprs:
		if(CommandIs("next"))		Investigate(Tree->MPC_Exprs.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_Exprs.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_Exprs.Expr);
		break;
	    case kMPC_SimpleInit:
		if(CommandIs("next"))		Investigate(Tree->MPC_SimpleInit.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_SimpleInit.Prev);
		else if(CommandIs("expr"))	Investigate(Tree->MPC_SimpleInit.Expr);
		break;
	    case kMPC_InitList:
		if(CommandIs("next"))		Investigate(Tree->MPC_InitList.Next);
		else if(CommandIs("prev"))	Investigate(Tree->MPC_InitList.Prev);
		else if(CommandIs("list"))	Investigate(Tree->MPC_InitList.List);
		break;
          } /* end switch(Tree->Kind) */
	} /* end else if */

    } /* end switch(State) */
  } /* end for(;;) */

}

/*******************/
static void Write_MPC
/*******************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree, int Mode)
# else
 (Tree, Mode)
 tTree Tree;
 int Mode;
# endif
{

# define  FULL_WritePosition(pos)  if(Mode==FULL)if(Compare(pos,NoPosition)){Store_Location(pos);WritePosition(stdout,RealPosition(pos)),(void)printf(":  ");}else{(void)printf("           ");}
/****
if(Mode==FULL)
  if(Compare(pos,NoPosition)) {
    Store_Location(pos);
    WritePosition(stdout,RealPosition(pos)),(void)printf(":  ");
  }
  else {
    (void)printf("           ");
  }
****/

# define  FULL_PRINT(arg)    if(Mode==FULL)(void)printf(arg)
# define  PART_PRINT(arg)    if(Mode==PART)(void)printf(arg)
# define  PRINT_COMMENT(arg) if(COMMENT&&Mode==FULL)(void)printf(arg)

  tTree node,  /* wrk var for writing of lists; */
	tmp;   /* wrk var for evaluation of subexpressions; */

  if(Tree==NoTree || Tree->Kind==kMPC_FreeNode)  return;
  FULL_PRINT("\n");

  switch(Tree->Kind) {
    case kMPC_BasicType:
    case kMPC_EnumType:
    case kMPC_Typedef:
    case kMPC_ArrayType:
    case kMPC_VectorType:
    case kMPC_PointerType:
    case kMPC_StructType:
    case kMPC_UnionType:
    case kMPC_FunctionType:
	FULL_WritePosition(Tree->MPC_Decls.Pos);
/***	WritePosition(stdout, RealPosition(Tree->MPC_Decls.Pos)); (void)printf(":  "); ***/
	Write_MPC_Type(Tree, FULL); (void)printf(";\n"); break;
    case kMPC_VarDecl:
	FULL_WritePosition(Tree->MPC_Decls.Pos);
	Write_Class(Tree);
	Write_MPC_Type(Tree->MPC_VarDecl.DeclSpecifier, PART);
	node = Tree->MPC_VarDecl.Var;
	while(node!=NoTree && node->Kind==kMPC_Var) {
	  Write_MPC(node, PART);
	  if(node->MPC_Var.Next!=NoTree && node->MPC_Var.Next->Kind==kMPC_Var)
	    (void)printf(", ");
	  node = node->MPC_Var.Next;
	}
	FULL_PRINT(";");
	break;
    case kMPC_Function:
	FULL_WritePosition(Tree->MPC_Function.Pos);
	Write_Class(Tree);
	tmp = Tree->MPC_Function.Type->MPC_FunctionType.ResultType;
	Write_MPC_Type(TerminalType(tmp), PART);
	Write_Declarator(Tree->MPC_Function.Name->MPC_Var.Ident,
			 Tree->MPC_Function.Type, Tree->MPC_Function.Distribution);
	(void)printf(" { ... }");
	switch(Tree->MPC_Function.Kind) {
	  case BASIC:	PRINT_COMMENT("   /* basic function */"); break;
	  case NODAL:	PRINT_COMMENT("   /* nodal function */"); break;
	  case NETWORK:	PRINT_COMMENT("   /* network function */"); break;
	}
	break;
    case kMPC_Ellipsis:
	FULL_WritePosition(Tree->MPC_Decls.Pos);
	(void)printf("...");
    case kMPC_NetDecl:
	FULL_WritePosition(Tree->MPC_NetDecl.Pos);
	Write_Class(Tree);
	Write_MPC(Tree->MPC_NetDecl.NetTypeSpecifier, PART);
	node = Tree->MPC_NetDecl.Net;
	while(node!=NoTree && node->Kind==kMPC_Net) {
	  Write_MPC(node, PART);
	  if(node->MPC_Net.Next!=NoTree && node->MPC_Net.Next->Kind==kMPC_Net)
	    (void)printf(", ");
	  node = node->MPC_Net.Next;
	}
	(void)printf(";");
	break;
    case kMPC_SubnetDecl:
	FULL_WritePosition(Tree->MPC_SubnetDecl.Pos);
	Write_Class(Tree);
	(void)printf("subnet ");
	node = Tree->MPC_SubnetDecl.Subnet;
	while(node!=NoTree && node->Kind==kMPC_Subnet) {
	  Write_MPC(node, PART);
	  if(node->MPC_Subnet.Next!=NoTree && node->MPC_Subnet.Next->Kind==kMPC_Subnet)
	    (void)printf(", ");
	  node = node->MPC_Subnet.Next;
	}
	(void)printf(";");
	break;
    case kMPC_RelDecl:
	FULL_WritePosition(Tree->MPC_RelDecl.Pos);
	(void)printf("relation ");
	node = Tree->MPC_RelDecl.Relation;
	while(node!=NoTree && node->Kind==kMPC_Relation) {
	  Write_MPC(node, PART);
	  if(node->MPC_Relation.Next!=NoTree && node->MPC_Relation.Next->Kind==kMPC_Relation)
	    (void)printf(", ");
	  node = node->MPC_Relation.Next;
	}
	(void)printf(";");
	break;
    case kMPC_NetType:
	FULL_WritePosition(Tree->MPC_RelDecl.Pos);
	Write_Class(Tree);
	(void)printf("nettype ");
	WriteIdent(stdout, Tree->MPC_NetType.Ident);
	if(Tree->MPC_NetType.ParamList!=NoTree &&
	   Tree->MPC_NetType.ParamList->Kind!=kMPC_FreeNode) {
	  (void)printf("(");
	  node = Tree->MPC_NetType.ParamList;
	  while(node!=NoTree && node->Kind==kMPC_Var) {
	    Write_MPC(node, PART);
	    if(node->MPC_Var.Next!=NoTree && node->MPC_Var.Next->Kind==kMPC_Var)
	      (void)printf(", ");
	    node = node->MPC_Var.Next;
	  }
	(void)printf(") ");
	}
	(void)printf(" { ... };");
	break;
    case kMPC_NetTypeSpecifier:
	FULL_WritePosition(Tree->MPC_Decls.Pos);
	(void)printf("net ");
	WriteIdent(stdout, Tree->MPC_NetTypeSpecifier.Ident);
	if(Tree->MPC_NetTypeSpecifier.ArgList!=NoTree &&
	   Tree->MPC_NetTypeSpecifier.ArgList->Kind==kMPC_Exprs) {
	  (void)printf("(");
	  node = Tree->MPC_NetTypeSpecifier.ArgList;
	  while(node!=NoTree && node->Kind==kMPC_Exprs) {
	    Write_MPC(node->MPC_Exprs.Expr, PART);
	    if(node->MPC_Exprs.Next!=NoTree && node->MPC_Exprs.Next->Kind==kMPC_Exprs)
	      (void)printf(", ");
	    node = node->MPC_Exprs.Next;
	  }
	  (void)printf(")");
	}
	(void)printf(" ");
	break;
    case kMPC_EnumConst:
	FULL_WritePosition(Tree->MPC_EnumConst.Pos);
	WriteIdent(stdout, Tree->MPC_EnumConst.Name);
	if(Tree->MPC_EnumConst.WithValue)
	  (void)printf("=%d", Tree->MPC_EnumConst.Value);
	break;
    case kMPC_SU_MemberDecl:
	Write_MPC(Tree->MPC_SU_MemberDecl.Member, FULL);
	break;
    case kMPC_SU_Member:
	if(Mode==FULL) {
	  FULL_WritePosition(Tree->MPC_SU_Member.Pos);
	  Write_MPC_Type(TerminalType(Tree->MPC_SU_Member.Type), FULL);
	}
	Write_Declarator(Tree->MPC_SU_Member.Name, Tree->MPC_Var.Type, NO_NET);
	if(Tree->MPC_SU_Member.BitFieldLength!=UNDEFINED)
	  (void)printf(":"),Write_MPC(Tree->MPC_SU_Member.BitFieldSize, PART);
	break;
    case kMPC_Var:
	if(Mode==FULL) {
	  FULL_WritePosition(Tree->MPC_Var.Pos);
	  Write_MPC_Type(TerminalType(Tree->MPC_Var.Type), FULL);
	}
	Write_Declarator(Tree->MPC_Var.Ident, Tree->MPC_Var.Type, Tree->MPC_Var.Distribution);
	break;
    case kMPC_CoordDecl:
	FULL_WritePosition(Tree->MPC_CoordDecl.Pos);
	if(Tree->MPC_CoordDecl.Prev==NoTree ||
	   Tree->MPC_CoordDecl.Prev->Kind==kMPC_FreeNode)
	  (void)printf("coord ");
	else
	  (void)printf("coord ..., ");
	WriteIdent(stdout, Tree->MPC_CoordDecl.Ident);
	(void)printf("=");
	Write_MPC(Tree->MPC_CoordDecl.Range, PART);
	if(Tree->MPC_CoordDecl.Next!=NoTree &&
	   Tree->MPC_CoordDecl.Next->Kind==kMPC_CoordDecl) {
	  (void)printf(", ...");
	}
	  (void)printf(" ;");
	break;
    case kMPC_Node:
	FULL_WritePosition(Tree->MPC_Node.Pos);
	(void) printf("node {\n");
	if(Tree->MPC_Node.Prev!=NoTree && Tree->MPC_Node.Prev->Kind==kMPC_Node)
	  (void)printf("               ...\n");
	(void)printf("             ");
	if(Tree->MPC_Node.Predicate)
	  Write_MPC(Tree->MPC_Node.Predicate, PART);
	else
	  (void)printf("default");
	(void)printf(": ");
	Write_NodeType(Tree);
	(void)printf("\n");
	if(Tree->MPC_Node.Next!=NoTree && Tree->MPC_Node.Next->Kind==kMPC_Node)
	  (void)printf("               ...\n");
	(void) printf("           };");
	break;

    /** +++++++++ **/

    /*******

    case kMPC_:
	FULL_WritePosition(Tree->MPC_.Pos);
	break;

    case kMPC_CoordDecl:
	...
    case kMPC_LinkLengthSpecifier:
    *******/

    case kMPC_NetList:
	Write_Distribution(Tree);
	break;
    case kMPC_Net:
	FULL_WritePosition(Tree->MPC_Net.Pos);
	Write_Distribution(Tree->MPC_Net.Distribution);
	if(Tree==HOST)
	  (void)printf("[host]");
	else if(Tree==COMPUTING_SPACE || IsGlobal && Tree==SINGLE_NODE)
	  (void)printf("[*]");
	else if(Tree==SINGLE_NODE && !IsGlobal) {
	  (void)printf("[<node>]");
	  PRINT_COMMENT("   /* distribution of nodal object */");
	}
	else if(Tree==CONST_NET) {
	  (void)printf("[<context-dependent>]");
	  PRINT_COMMENT("   /* distribution of constant expression */");
	}
	else
	  WriteIdent(stdout, Tree->MPC_Net.Ident);
	break;
    case kMPC_Subnet:
	FULL_WritePosition(Tree->MPC_Subnet.Pos);
	/*** Write_Distribution(Tree);   --- ??? ***/
	(void)printf("[");
	WriteIdent(stdout,Tree->MPC_Subnet.NetOrSubnetName);
	(void)printf(":");
	Write_MPC(Tree->MPC_Subnet.Predicate, PART);
	(void)printf("]");
	if(Tree->MPC_Subnet.Ident!=NoIdent)
	  WriteIdent(stdout, Tree->MPC_Subnet.Ident);
	break;
    case kMPC_Relation:
	FULL_WritePosition(Tree->MPC_Relation.Pos);
	WriteIdent(stdout, Tree->MPC_Relation.LNet->MPC_NetOrSubnet.Ident);
	Write_OpCode(Tree->MPC_Relation.RelCode);
	WriteIdent(stdout, Tree->MPC_Relation.RNet->MPC_NetOrSubnet.Ident);
	break;
    case kMPC_ExprStat:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	Write_MPC(Tree->MPC_ExprStat.Expr, PART);
	(void)printf(" ;");
	break;
    case kMPC_If:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("if");
	Write_Condition(Tree->MPC_If.Expr, FULL);
	Write_NestedStat(Tree->MPC_If.Stats);
	break;
    case kMPC_IfElse:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("if");
	Write_Condition(Tree->MPC_IfElse.Expr, FULL);
	Write_NestedStat(Tree->MPC_IfElse.Then);
	(void)printf("           ");
	(void)printf("else");
	Write_NestedStat(Tree->MPC_IfElse.Else);
	break;
    case kMPC_Switch:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("switch");
	Write_Condition(Tree->MPC_Switch.Expr, FULL);
	Write_NestedStat(Tree->MPC_Switch.Stats);
	break;
    case kMPC_While:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("while");
	Write_Condition(Tree->MPC_While.Expr, FULL);
	Write_NestedStat(Tree->MPC_While.Stats);
	break;
    case kMPC_DoWhile:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("do");
	Write_NestedStat(Tree->MPC_If.Stats);
	(void)printf("           ");
	(void)printf("while");
	Write_Condition(Tree->MPC_DoWhile.Expr, FULL);
	(void)printf(";");
	break;
    case kMPC_For:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("for(");
	Write_Condition(Tree->MPC_For.Init, PART); (void)printf(";");
	Write_Condition(Tree->MPC_For.Cond, PART); (void)printf(";");
	Write_Condition(Tree->MPC_For.Reinit, PART); (void)printf(")");
	Write_NestedStat(Tree->MPC_For.Stats);
	break;
    case kMPC_Goto:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("goto ");
	WriteIdent(stdout, Tree->MPC_Goto.Ident);
	(void)printf(";");
	break;
    case kMPC_Continue:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("continue;");
	break;
    case kMPC_Break:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("break;");
	break;
    case kMPC_BreakFan:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("break?;");
	break;
    case kMPC_Return:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("return ");
	Write_MPC(Tree->MPC_Return.Expr, PART);
	(void)printf(";");
	break;
    case kMPC_Compound:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("{ ");
	if(Tree->MPC_Compound.Decls!=NoTree &&
	   Tree->MPC_Compound.Decls->Kind!=kMPC_FreeNode)
	  (void)printf("<declarations> ");
	if(Tree->MPC_Compound.Stats!=NoTree &&
	   Tree->MPC_Compound.Stats->Kind!=kMPC_FreeNode)
	  (void)printf("<statements> ");
	(void)printf("}");
	break;

/************
    case kMPC_Fan:
	if(Mode==FULL)
	  WritePosition(stdout, RealPosition(Tree->MPC_Stats.Pos)), (void)printf(":  ");
	break;
************/

    case kMPC_IdentLabel:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	WriteIdent(stdout, Tree->MPC_IdentLabel.Ident);
	(void)printf(":");
	break;
    case kMPC_CaseLabel:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("case ");
	Write_MPC(Tree->MPC_CaseLabel.Expr, PART);
	(void)printf(":");
	break;
    case kMPC_Default:
	FULL_WritePosition(Tree->MPC_Stats.Pos);
	(void)printf("default:");
	break;
    case kMPC_IntConst:
    case kMPC_UIntConst:
    case kMPC_FloatConst:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	(void)printf("%s", Tree->MPC_Const.SymbolicForm);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* constant */");
	break;
    case kMPC_StringLiteral:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	if(Tree->MPC_StringLiteral.Prev!=NoTree &&
	   Tree->MPC_StringLiteral.Prev->Kind!=kMPC_FreeNode)
	  (void)printf(" ... ");
	(void)printf("%s", Tree->MPC_StringLiteral.String);
	if(Tree->MPC_StringLiteral.Next!=NoTree &&
	   Tree->MPC_StringLiteral.Next->Kind!=kMPC_FreeNode)
	  (void)printf(" ... ");
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	if(Tree->MPC_StringLiteral.Prev!=NoTree &&
	   Tree->MPC_StringLiteral.Prev->Kind!=kMPC_FreeNode ||
	   Tree->MPC_StringLiteral.Next!=NoTree &&
	   Tree->MPC_StringLiteral.Next->Kind!=kMPC_FreeNode)
	  PRINT_COMMENT("   /* string literal list */");
	else
	  PRINT_COMMENT("   /* string literal */");
	break;
    case kMPC_Ident:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	WriteIdent(stdout, Tree->MPC_Ident.Ident);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* identifier */");
	break;
    case kMPC_CastExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	(void)printf("(");
	Write_MPC_Type(Tree->MPC_CastExpr.TypeName, PART);
	(void)printf(")");
	Write_MPC(Tree->MPC_CastExpr.Operand, PART);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* cast expression */");
	break;
    case kMPC_NetCastExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	if(Tree->MPC_NetCastExpr.Flexible)
	  Write_MPC(Tree->MPC_NetCastExpr.StoreNet, PART);
	else
	  Write_Distribution(Tree->MPC_NetCastExpr.StoreNet);
	Write_MPC(Tree->MPC_NetCastExpr.Operand, PART);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* network cast expression */");
	break;
    case kMPC_CoordExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	tmp = Tree->MPC_CoordExpr.Operand;
	if(tmp->Kind==kMPC_Ident && tmp->MPC_Ident.Store->Kind==kMPC_CoordDecl &&
	   tmp->MPC_Ident.Ident==Tree->MPC_CoordExpr.CoordName)
	  WriteIdent(stdout, Tree->MPC_CoordExpr.CoordName);
	else {
	  WriteIdent(stdout, Tree->MPC_CoordExpr.CoordName);
	  (void)printf(" coordof ");
	  Write_MPC(Tree->MPC_CoordExpr.Operand, PART);
	}
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* coordof expression */");
	break;
    case kMPC_Size_Of_Value:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	(void)printf("sizeof ");
	Write_MPC(Tree->MPC_Size_Of_Value.Operand, PART);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* sizeof expression */");
	break;
    case kMPC_Size_Of_Type:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	(void)printf("sizeof(");
	Write_MPC_Type(Tree->MPC_Size_Of_Type.Operand, PART);
	(void)printf(")");
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* sizeof expression */");
	break;
    case kMPC_UnaryExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	if(Is_PostfixOperator(Tree->MPC_UnaryExpr.OpCode)) {
	  Write_MPC(Tree->MPC_UnaryExpr.Operand, PART);
	  Write_OpCode(Tree->MPC_UnaryExpr.OpCode);
	}
	else {
	  Write_OpCode(Tree->MPC_UnaryExpr.OpCode);
	  Write_MPC(Tree->MPC_UnaryExpr.Operand, PART);
	}
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* unary expression */");
	break;
    case kMPC_BinaryExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	if(Tree->MPC_BinaryExpr.OpCode==INDEX) {
	  Write_MPC(Tree->MPC_BinaryExpr.Loperand, PART);
	  (void)printf("[");
	  Write_MPC(Tree->MPC_BinaryExpr.Roperand, PART);
	  (void)printf("]");
	}
	else if(Tree->MPC_BinaryExpr.OpCode==DISTR_ASS_CALL) {
	  Write_MPC(Tree->MPC_BinaryExpr.Loperand, PART);
	  (void)printf("[>");
	  Write_MPC(Tree->MPC_BinaryExpr.Roperand, PART);
	  (void)printf("<]");
	}
	else if(Tree->MPC_BinaryExpr.OpCode==DISTR_COMM_CALL) {
	  Write_MPC(Tree->MPC_BinaryExpr.Loperand, PART);
	  (void)printf("[<");
	  Write_MPC(Tree->MPC_BinaryExpr.Roperand, PART);
	  (void)printf(">]");
	}
	else {
	  Write_MPC(Tree->MPC_BinaryExpr.Loperand, PART);
	  Write_OpCode(Tree->MPC_BinaryExpr.OpCode);
	  Write_MPC(Tree->MPC_BinaryExpr.Roperand, PART);
	}
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* binary expression */");
	break;
    case kMPC_TernaryExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
	if(Tree->MPC_TernaryExpr.OpCode==VECT_EVAL) {
	  Write_MPC(Tree->MPC_TernaryExpr.Foperand, PART);
	  (void)printf("[");
	  Write_MPC(Tree->MPC_TernaryExpr.Soperand, PART);
	  (void)printf("#");
	  Write_MPC(Tree->MPC_TernaryExpr.Toperand, PART);
	  (void)printf("]");
	}
	else if(Tree->MPC_TernaryExpr.OpCode==GRID) {
	  Write_MPC(Tree->MPC_TernaryExpr.Foperand, PART);
	  (void)printf("[");
	  Write_MPC(Tree->MPC_TernaryExpr.Soperand, PART);
	  (void)printf(":");
	  Write_MPC(Tree->MPC_TernaryExpr.Toperand, PART);
	  (void)printf("]");
	}
	else if(Tree->MPC_TernaryExpr.OpCode==CONDITION) {
	  Write_MPC(Tree->MPC_TernaryExpr.Foperand, PART);
	  (void)printf(" ? ");
	  Write_MPC(Tree->MPC_TernaryExpr.Soperand, PART);
	  (void)printf(" : ");
	  Write_MPC(Tree->MPC_TernaryExpr.Toperand, PART);
	}
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* ternary expression */");
	break;
    case kMPC_CallExpr:
	FULL_WritePosition(Tree->MPC_Expr.Pos);
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf("(");
     /** +++  ADD ALL FOR CALL OF THE NETWORK FUNCTION !!!  +++ **/
	Write_MPC(Tree->MPC_CallExpr.Function, PART);
	(void)printf("(");
	node = Tree->MPC_CallExpr.ArgList;
	while(node!=NoTree && node->Kind==kMPC_Exprs) {
	  Write_MPC(node->MPC_Exprs.Expr, PART);
	  if(node->MPC_Exprs.Next!=NoTree && node->MPC_Exprs.Next->Kind==kMPC_Exprs)
	    (void)printf(", ");
	  node = node->MPC_Exprs.Next;
	}
	(void)printf(")");
	if(Tree->MPC_Expr.Flag.InParentheses) (void)printf(")");
	PRINT_COMMENT("   /* call expression */");
	break;
    case kMPC_Exprs:
	FULL_WritePosition(Tree->MPC_Exprs.Pos);
	if(Tree->MPC_Exprs.Prev!=NoTree &&
	   Tree->MPC_Exprs.Prev->Kind==kMPC_Exprs) (void)printf("(...), ");
	(void)printf("(");
	Write_MPC(Tree->MPC_Exprs.Expr, PART);
	(void)printf(")");
	if(Tree->MPC_Exprs.Next!=NoTree &&
	   Tree->MPC_Exprs.Next->Kind==kMPC_Exprs) (void)printf(", (...)");
	PRINT_COMMENT("   /* list of expressions */");
	break;
    case kMPC_SimpleInit:
	FULL_WritePosition(Tree->MPC_Initializer.Pos);
	Write_MPC(Tree->MPC_SimpleInit.Expr, PART);
	break;
    case kMPC_InitList:
	FULL_WritePosition(Tree->MPC_Initializer.Pos);
	(void)printf("{...}");
	break;
    default:
	(void)fprintf(stderr,"mpC Advisor: Sorry, can't write this tree node...\n");
	/**WriteTreeNode(stdout, Tree);**/ /*** TMP - for all other nodes ***/
  } /*** end switch(Tree->Kind) ***/

  FULL_PRINT("\n\n");
}

/*********************/
static void Write_Class
/*********************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{
  tClass Class=EXTERN;
  if(Tree->Kind==kMPC_Function) {
    if(Tree->MPC_Function.Flag.Static)		(void)printf("static ");
    else if(Tree->MPC_Function.Flag.Extern)	(void)printf("extern ");
  }
  else {
    if(Tree->Kind==kMPC_VarDecl)	Class = Tree->MPC_VarDecl.Class;
    if(Tree->Kind==kMPC_NetDecl)	Class = Tree->MPC_NetDecl.NetClass;
    if(Tree->Kind==kMPC_SubnetDecl)	Class = Tree->MPC_SubnetDecl.SubnetClass;
    if(Tree->Kind==kMPC_NetType)	Class = Tree->MPC_NetType.Class;
    switch(Class) {
      case EXTERN:		if(Tree->MPC_VarDecl.Flag.Extern)
				  (void)printf("extern ");  break;
      case STATIC:
      case GLOBAL_STATIC:
      case LOCAL_STATIC:	(void)printf("static "); break;
      case AUTO:		(void)printf("auto "); break;
      case REGISTER:		(void)printf("register "); break;
      case TYPEDEF:		(void)printf("typedef "); break;
      case FLEXIBLE:		(void)printf("flex "); break;
    }
  }
}

/**************************/
static void Write_Declarator
/**************************/
# if defined __STDC__ | defined __cplusplus
 (tIdent Ident, tTree Type, tTree Distr)
# else
 (Ident, Type, Distr)
 tIdent Ident;
 tTree Type, Distr;
# endif
{
  switch(Type->Kind) {
    case kMPC_BasicType:
    case kMPC_EnumType:
    case kMPC_Typedef:
    case kMPC_StructType:
    case kMPC_UnionType:
	WriteBefore();  /* all from stack; */
	Write_Distribution(Distr);
	WriteIdent(stdout, Ident);
	WriteAfter();	/* all from queue; */
	break;
    case kMPC_ArrayType:
	PushAfter(Type);
	Write_Declarator(Ident, Type->MPC_ArrayType.ElementType, Distr);
	break;
    case kMPC_VectorType:
	PushAfter(Type);
	Write_Declarator(Ident, Type->MPC_VectorType.ElementType, Distr);
	break;
    case kMPC_FunctionType:
	PushAfter(Type);
	Write_Declarator(Ident, Type->MPC_FunctionType.ResultType, Distr);
	break;
    case kMPC_PointerType:
	PushBefore(Type);
	if(Type->MPC_PointerType.ElementType->Kind==kMPC_ArrayType ||
	   Type->MPC_PointerType.ElementType->Kind==kMPC_VectorType ||
	   Type->MPC_PointerType.ElementType->Kind==kMPC_FunctionType) {
	  PushBefore(NoTree);	/* '(' --> stack */
	  PushAfter(NoTree);	/* ')' --> queue */
	}
	Write_Declarator(Ident, Type->MPC_PointerType.ElementType, Distr);
	break;
  } /** end switch(Type->Kind) **/
}

/************************/
static void Write_MPC_Type
/************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Type, int Mode)
# else
 (Type, Mode)
 tTree Type;
# endif
{
  tTree node;

  if(!Type) {
    (void)printf("<NoType> ");
    return;
  }
  /*** ADD: print type qualifiers!!! ***/
  if(Type->MPC_Type.Flag.Const)		(void)printf("const ");
  if(Type->MPC_Type.Flag.Volatile)	(void)printf("volatile ");
  if(Type->MPC_Type.Flag.Repl)		(void)printf("repl ");
  switch(Type->Kind) {
    case kMPC_BasicType:
	switch(Type->MPC_BasicType.TypeConstructor) {
	  case NO_TYPE:		(void)printf("<NoType> "); break;
	  case VOID:		(void)printf("void "); break;
	  case CHAR:		(void)printf("char "); break;
	  case SIGNED_CHAR:	(void)printf("signed char "); break;
	  case UNSIGNED_CHAR:	(void)printf("unsigned char "); break;
	  case SHORT:		(void)printf("short "); break;
	  case UNSIGNED_SHORT:	(void)printf("unsigned short "); break;
	  case INT:		(void)printf("int "); break;
	  case UNSIGNED:	(void)printf("unsigned int "); break;
	  case LONG:		(void)printf("long "); break;
	  case UNSIGNED_LONG:	(void)printf("unsigned long "); break;
	  case FLOAT:		(void)printf("float "); break;
	  case DOUBLE:		(void)printf("double "); break;
	  case LONG_DOUBLE:	(void)printf("long double "); break;
	}
/***********/
	WriteBefore();
	WriteAfter();
/***********/
	break;
    case kMPC_EnumType:
	(void)printf("enum ");
	if(Type->MPC_EnumType.EnumTag!=NoIdent)
	  WriteIdent(stdout, Type->MPC_EnumType.EnumTag), (void)printf(" ");
	if(Mode==FULL && Type->MPC_EnumType.EnumList!=NoTree &&
	   Type->MPC_EnumType.EnumList->Kind==kMPC_EnumConst) (void)printf("{...} ");
	WriteBefore();
	WriteAfter();
	break;
    case kMPC_Typedef:
	if(Mode==FULL) {
	  (void)printf("typedef  ");
	  Write_MPC_Type(Type->MPC_Typedef.Type, PART); (void)printf(" ");
	}
	WriteIdent(stdout, Type->MPC_Typedef.TypedefName); (void)printf(" ");
	WriteBefore();
	WriteAfter();
	break;
/********/
    case kMPC_ArrayType:
	PushAfter(Type);
	Write_MPC_Type(Type->MPC_ArrayType.ElementType, PART);
	break;
    case kMPC_VectorType:
	PushAfter(Type);
	Write_MPC_Type(Type->MPC_VectorType.ElementType, PART);
	break;
    case kMPC_FunctionType:
	PushAfter(Type);
	Write_MPC_Type(Type->MPC_FunctionType.ResultType, PART);
	break;
    case kMPC_PointerType:
	PushBefore(Type);
	if(Type->MPC_PointerType.ElementType->Kind==kMPC_ArrayType ||
	   Type->MPC_PointerType.ElementType->Kind==kMPC_VectorType ||
	   Type->MPC_PointerType.ElementType->Kind==kMPC_FunctionType) {
	  PushBefore(NoTree);	/* '(' --> stack */
	  PushAfter(NoTree);	/* ')' --> queue */
	}
	Write_MPC_Type(Type->MPC_PointerType.ElementType, PART);
	break;
    case kMPC_StructType:
	(void)printf("struct ");
	if(Type->MPC_StructType.SU_Tag!=NoIdent)
	  WriteIdent(stdout, Type->MPC_StructType.SU_Tag), (void)printf(" ");
	if(Mode==FULL && Type->MPC_StructType.MemberDecls!=NoTree &&
	   Type->MPC_StructType.MemberDecls->Kind==kMPC_SU_MemberDecl) (void)printf("{...} ");
	break;
    case kMPC_UnionType:
	(void)printf("union ");
	if(Type->MPC_UnionType.SU_Tag!=NoIdent)
	  WriteIdent(stdout, Type->MPC_UnionType.SU_Tag), (void)printf(" ");
	if(Mode==FULL && Type->MPC_UnionType.MemberDecls!=NoTree &&
	   Type->MPC_UnionType.MemberDecls->Kind==kMPC_SU_MemberDecl) (void)printf("{...} ");
	break;

  } /** end switch(Type->Kind) **/

}

/****************************/
static void Write_Distribution
/****************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{
  tTree node;

  if(!Tree) return;
  if(Tree_IsType(Tree, kMPC_NetOrSubnet))
    if(Tree==SINGLE_NODE || Tree==CONST_NET) ;  /* nothing to print; */
    else if(Tree==HOST)
      (void)printf("[host]");
    else if(Tree==COMPUTING_SPACE)
      (void)printf("[*]");
    else {  /* NetOrSubnet; --ADD ACTIONS FOR FLEXIBLE SUBNETS !!! */
      (void)printf("[");
      if(Tree->Kind==kMPC_Net && Tree->MPC_Net.Topology==PARAMETER)
	Write_MPC(Tree->MPC_Net.NetType, PART);
      WriteIdent(stdout,Tree->MPC_NetOrSubnet.Ident);
      (void)printf("]");
      if(Tree->Kind==kMPC_Net && Tree->MPC_Net.Topology==PARAMETER)
	(void)printf(" ");
    }
  else if(Tree->Kind==kMPC_NetList) {
    (void)printf("{ ");
    while(Tree!=NoTree && Tree->Kind==kMPC_NetList) {
      Write_Distribution(Tree->MPC_NetList.Net);
      if(Tree->MPC_NetList.Next!=NoTree && Tree->MPC_NetList.Next->Kind==kMPC_NetList)
	(void)printf(", ");
      Tree = Tree->MPC_NetList.Next;
    }
    (void)printf(" }");
  }
}

/*************************/
static void Write_Condition
/*************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Expr, int Mode)
# else
 (Expr, Mode) tTree Expr;
# endif
{
  if(Mode==FULL) (void)printf("(");
  if(Is_Printable(Expr))
    Write_MPC(Expr, PART);
  else
    (void)printf("<expression>");
  if(Mode==FULL) (void)printf(")");
}

/**************************/
static void Write_NestedStat
/**************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Stat)
# else
 (Stat) tTree Stat;
# endif
{
  if(Stat==NoTree || Stat->Kind==kMPC_FreeNode) return;
  switch(Stat->Kind) {
    case kMPC_Goto:
    case kMPC_Continue:
    case kMPC_Break:
    case kMPC_BreakFan:
    case kMPC_Return:
	(void)printf("\n");
	(void)printf("           ");
	Write_MPC(Stat, PART);
	break;
    case kMPC_Compound:
	(void)printf(" {\n");
	(void)printf("           ");
	(void)printf("...\n");
	(void)printf("           ");
	(void)printf("}\n");
	break;
    default:
	(void)printf("\n");
	(void)printf("           ");
	(void)printf("   <statement>\n");
  }
}

/**********************/
static void Write_OpCode
/**********************/
# if defined __STDC__ | defined __cplusplus
 (int OpCode)
# else
 (OpCode)
# endif
	/***********************************************************/
	/** Write only unary and binary operators (except INDEX): **/
	/***********************************************************/
{
  switch(OpCode) {
    case VECT_CONCAT:		(void)printf("<+>"); break;
    case BRACKETS:		(void)printf("[]"); break;
    case DOT:			(void)printf("."); break;
    case ARROW:			(void)printf("->"); break;
    case POST_INC:		(void)printf("++ "); break;
    case POST_DEC:		(void)printf("-- "); break;
    case PRE_INC:		(void)printf(" ++"); break;
    case PRE_DEC:		(void)printf(" --"); break;
    case ADDRESS:		(void)printf(" &"); break;
    case STAR:			(void)printf(" *"); break;
    case NEG:			(void)printf(" -"); break;
    case NOT:			(void)printf(" ~"); break;
    case LOG_NOT:		(void)printf(" !"); break;
    case MULT:			(void)printf(" * "); break;
    case DIV:			(void)printf(" / "); break;
    case MOD:			(void)printf(" %c ",'%'); break; /*******/
    case MAX:			(void)printf(" ?> "); break;
    case MIN:			(void)printf(" ?< "); break;
    case PLUS:			(void)printf(" + "); break;
    case MINUS:			(void)printf(" - "); break;
    case LEFT_SHIFT:		(void)printf(" << "); break;
    case RIGHT_SHIFT:		(void)printf(" >> "); break;
    case LESS_THEN:		(void)printf(" < "); break;
    case GREAT_THEN:		(void)printf(" > "); break;
    case LESS_OR_EQ:		(void)printf(" <= "); break;
    case GREAT_OR_EQ:		(void)printf(" >= "); break;
    case EQUAL:			(void)printf(" == "); break;
    case NOT_EQUAL:		(void)printf(" != "); break;
    case AND:			(void)printf(" & "); break;
    case EXCL_OR:		(void)printf(" ^ "); break;
    case INCL_OR:		(void)printf(" | "); break;
    case LOG_AND:		(void)printf(" && "); break;
    case LOG_OR:		(void)printf(" || "); break;
    case ASSIGN:		(void)printf(" = "); break;
    case MULT_ASSIGN:		(void)printf(" *= "); break;
    case DIV_ASSIGN:		(void)printf(" /= "); break;
    case MOD_ASSIGN:		(void)printf(" %= "); break;
    case PLUS_ASSIGN:		(void)printf(" += "); break;
    case MINUS_ASSIGN:		(void)printf(" -= "); break;
    case LSHIFT_ASSIGN:		(void)printf(" <<= "); break;
    case RSHIFT_ASSIGN:		(void)printf(" >>= "); break;
    case AND_ASSIGN:		(void)printf(" &= "); break;
    case EXCL_OR_ASSIGN:	(void)printf(" ^= "); break;
    case INCL_OR_ASSIGN:	(void)printf(" |= "); break;
    case COMMA:			(void)printf(" , "); break;

    case LIN_MULT:		(void)printf(" [*]"); break;
    case LIN_MAX:		(void)printf(" [?>]"); break;
    case LIN_MIN:		(void)printf(" [?<]"); break;
    case LIN_PLUS:		(void)printf(" [+]"); break;
    case LIN_AND:		(void)printf(" [&]"); break;
    case LIN_EXCL_OR:		(void)printf(" [^]"); break;
    case LIN_INCL_OR:		(void)printf(" [|]"); break;
    case LIN_LOG_AND:		(void)printf(" [&&]"); break;
    case LIN_LOG_OR:		(void)printf(" [||]"); break;

    case DISTR_LIN_MULT:	(void)printf("[*] "); break;
    case DISTR_LIN_MAX:		(void)printf("[?>] "); break;
    case DISTR_LIN_MIN:		(void)printf("[?<] "); break;
    case DISTR_LIN_PLUS:	(void)printf("[+] "); break;
    case DISTR_LIN_AND:		(void)printf("[&] "); break;
    case DISTR_LIN_EXCL_OR:	(void)printf("[^] "); break;
    case DISTR_LIN_INCL_OR:	(void)printf("[|] "); break;
    case DISTR_LIN_LOG_AND:	(void)printf("[&&] "); break;
    case DISTR_LIN_LOG_OR:	(void)printf("[||] "); break;
    default:			(void)printf("<NoOperator>");
  }
}

/***************************/
static int Is_PostfixOperator
/***************************/
# if defined __STDC__ | defined __cplusplus
 (int OpCode)
# else
 (OpCode)
#endif
{
  switch(OpCode) {
    case BRACKETS:
    case POST_INC:
    case POST_DEC:
    case DISTR_LIN_MULT:
    case DISTR_LIN_MAX:
    case DISTR_LIN_MIN:
    case DISTR_LIN_PLUS:
    case DISTR_LIN_AND:
    case DISTR_LIN_EXCL_OR:
    case DISTR_LIN_INCL_OR:
    case DISTR_LIN_LOG_AND:
    case DISTR_LIN_LOG_OR:
	return YES;
  }
  return NO;
}

/************************/
static void Write_NodeType
/************************/
# if defined __STDC__ | defined __cplusplus
 (tTree Node)
# else
 (Node) tTree Node;
# endif
{
  tTree Qual;

  if(!Node || Node->Kind!=kMPC_Node) return;

  Qual = Node->MPC_Node.NodeQual;
  if(Qual!=NoTree && Qual->Kind==kMPC_NodeTypeQual) {
    if(Qual->MPC_NodeTypeQual.Fast)
      (void)printf("fast ");
    else
      (void)printf("slow ");
    if(Qual->MPC_NodeTypeQual.CapacityExpr!=NoTree &&
       Qual->MPC_NodeTypeQual.CapacityExpr->Kind!=kMPC_FreeNode) {
      (void)printf("* ");
      Write_MPC(Qual->MPC_NodeTypeQual.CapacityExpr, PART);
    }
  }

  switch(BasicNodeType(Node->MPC_Node.NodeType)) {
    case VOID_NODE:	(void)printf(" void;"); break;
    case MEMORY_NODE:	(void)printf(" memory;"); break;
    case SCALAR_NODE:	(void)printf(" scalar;"); break;
    case VECTOR_NODE:	(void)printf(" vector;"); break;
    default:		(void)printf(" <unknown node type>;"); break;
  }
}

/*********************/
static int Is_Printable
/*********************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{
  return NO;  /* TMP */
}

/**********************/
static void Write_Prompt
/**********************/
# if defined __STDC__ | defined __cplusplus
 (tTree Tree)
# else
 (Tree) tTree Tree;
# endif
{
  if(QUIET || Tree->Kind==kMPC_FreeNode)  return;
  if(PROMPT_MODE==FULL)
    (void)printf("help-prompt-comment-back-recompile-quit---");
 /**(void)printf("help-prompt-back-up-compile-quit---");****/
  switch(Tree->Kind) {
    case kMPC_ErrorMessage:
    case kMPC_ErrorMessageI:	(void)printf("next-prev"); break;
    case kMPC_Message:
    case kMPC_MessageI:		(void)printf("next-prev-info-investigate"); break;
    case kMPC_EnumType:
    case kMPC_Typedef:
    case kMPC_StructType:
    case kMPC_UnionType:	(void)printf("next-prev"); break;
    case kMPC_VarDecl:		(void)printf("next-prev-var"); break;
    case kMPC_Function:		(void)printf("next-prev-param-body-[type]-[distribution]"); break;
    case kMPC_Ellipsis:		(void)printf("prev"); break;
    case kMPC_NetDecl:		(void)printf("next-prev-topology-net"); break;
    case kMPC_SubnetDecl:	(void)printf("next-prev-subnet"); break;
    case kMPC_RelDecl:		(void)printf("next-prev-relation"); break;
    case kMPC_NetType:		(void)printf("next-prev-param-coord-node-link-parent"); break;
    case kMPC_NetTypeSpecifier: (void)printf("nettype-arguments"); break;
    case kMPC_Var:		(void)printf("next-prev-[type]-[distribution]"); break;
    case kMPC_CoordDecl:	(void)printf("next-prev-range"); break;
    case kMPC_Node:		(void)printf("next-prev-predicate"); break;
    case kMPC_LinkDecl:		(void)printf("freecoord-linklist"); break;
    case kMPC_LinkDeclaringList:(void)printf("next-prev-predicate"); break;
/** case kMPC_LinkDeclarator:	(void)printf(""); break; **/
    case kMPC_NetList:		(void)printf("next-prev-net"); break;
    case kMPC_Net:		(void)printf("next-prev-topology-distribution"); break;
    case kMPC_Subnet:		(void)printf("next-prev-supernet-predicate"); break;
    case kMPC_Relation:		(void)printf("next-prev-left-right"); break;
    case kMPC_ExprStat:		(void)printf("next-prev-expr-[distribution]"); break;
    case kMPC_If:		(void)printf("next-prev-expr-then-[distribution]"); break;
    case kMPC_IfElse:		(void)printf("next-prev-expr-then-else-[distribution]"); break;
    case kMPC_Switch:		(void)printf("next-prev-expr-body-[distribution]"); break;
    case kMPC_While:		(void)printf("next-prev-expr-body-[distribution]"); break;
    case kMPC_DoWhile:		(void)printf("next-prev-body-expr-[distribution]"); break;
    case kMPC_For:	(void)printf("next-prev-init-cond-reinit-body-[distribution]"); break;
    case kMPC_Goto:
    case kMPC_Continue:
    case kMPC_Break:
    case kMPC_BreakFan:		(void)printf("next-prev-[distribution]"); break;
    case kMPC_Return:		(void)printf("next-prev-expr-[distribution]"); break;
    case kMPC_Compound:		(void)printf("next-prev-decls-stats-[distribution]"); break;
    case kMPC_Fan:		(void)printf("next-prev-body-[distribution]"); break;
    case kMPC_IdentLabel:
    case kMPC_CaseLabel:
    case kMPC_Default:		(void)printf("next-prev"); break;
    case kMPC_IntConst:
    case kMPC_UIntConst:
    case kMPC_FloatConst:	(void)printf("[type]-[storenet]"); break;
    case kMPC_StringLiteral:	(void)printf("next-prev-[storenet]"); break;
    case kMPC_Ident:		(void)printf("decl-[type]-[storenet]"); break;
    case kMPC_CastExpr:
    case kMPC_NetCastExpr:
    case kMPC_CoordExpr:	(void)printf("operand-[type]-[evalnet]-[storenet]"); break;
    case kMPC_Size_Of_Value:
    case kMPC_Size_Of_Type:
    case kMPC_UnaryExpr:	(void)printf("operand-[type]-[evalnet]-[storenet]"); break;
    case kMPC_BinaryExpr:	(void)printf("left-right-[type]-[evalnet]-[storenet]"); break;
    case kMPC_TernaryExpr:
			(void)printf("first-second-third-[type]-[evalnet]-[storenet]"); break;
    case kMPC_CallExpr:		(void)printf("function-args");
				if(Tree->MPC_CallExpr.NetworkArgList!=NoTree &&
				   Tree->MPC_CallExpr.NetworkArgList->Kind!=kMPC_FreeNode)
				  (void)printf("-netargs");
				(void)printf("-[type]-[evalnet]-[storenet]"); break;
    case kMPC_Exprs:
    case kMPC_SimpleInit:	(void)printf("next-prev-expr"); break;
    case kMPC_InitList:		(void)printf("next-prev-list"); break;
  }
  (void)printf("-? ");
}

/****************/
static bool FileIs
/****************/
# if defined __STDC__ | defined __cplusplus
 (tString File1, tString File2)
# else
 (File1, File2)
 tString File1, File2;
# endif
{
  bool equal=true;
  char f1[256], f2[256];
  int i1, i2;

  if(File1[0]=='"') {
    strcpy(f1,File1+1);
    f1[strlen(f1)-1] = '\0';
  }
  else
    strcpy(f1,File1);
  if(File2[0]=='"') {
    strcpy(f2,File2+1);
    f2[strlen(f2)-1] = '\0';
  }
  else
    strcpy(f2,File2);

  for(i1=strlen(f1)-1,i2=strlen(f2)-1; i1>=0 && i2>=0; i1--,i2--) {
    if(f1[i1]!=f2[i2]) { equal = false; break; }
  }

  /*** equal = ! strcmp(f1,f2);  *** TMP ***/
/***/if(DEBUG)printf("FileIs() : compare %s to %s  return %d\n",f1,f2,equal);
  return equal;
}

/*******************/
static bool CommandIs
/*******************/
# if defined __STDC__ | defined __cplusplus
 (tString Pattern)
# else
 (Pattern) tTree Pattern;
# endif
{
  int i, plength;
# define clength ComLength+1

  plength = strlen(Pattern);
  if(!(clength) && plength) return false;  /* no command; */
  for(i=0; i<clength; i++) {
    if(i>=plength && Command[i]!=' ' || Pattern[i]!=Command[i])
      return false;
    else if(i>=plength && Command[i]==' ' || i==plength && i==clength)
      break;
  }
  return true;
}

/************************/
static void Store_Location
/************************/
# if defined __STDC__ | defined __cplusplus
	(tPosition Pos)
# else
	(Pos) tPosition Pos;
# endif
{
  tString file;

  if(Compare(Pos,NoPosition)) {
    file = SourceFile(Pos);
    ADVICE_POS  = RealPosition(Pos);
  }
  else
    file = CURR_ADVICE_FILE;

  if(!QUIET && !FileIs(file, CURR_ADVICE_FILE) ) {
    (void)printf("\nmpcAdvisor: exploring the file %s :\n\n", file);
    CURR_ADVICE_FILE = file;
  }
}

/***************************************************************/
/** Stack ('Before') and Queue ('After') for Write_Declarator **/
/***************************************************************/

static tTree Before[32], After[32];
static int Before_Top = 0, After_Top = 0;

/********************/
static void PushBefore
/********************/
#if defined __STDC__ || defined __cplusplus
 (tTree Decl)
#else
 (Decl) tTree Decl;
#endif
{
  Before[Before_Top++] = Decl;
}

/*********************/
static void WriteBefore
/*********************/
 ARGS((void))
{
  while(Before_Top) {
    Before_Top--;
    if(Before[Before_Top]==NoTree)
      (void)printf("(");
    else if(Before[Before_Top]->Kind==kMPC_PointerType) {
      (void)printf("*");
      if(Before[Before_Top]->MPC_PointerType.Step!=1)
	(void)printf(":%d", Before[Before_Top]->MPC_PointerType.Step);
      if(Before[Before_Top]->MPC_Type.Flag.Const)	(void)printf("const ");
      if(Before[Before_Top]->MPC_Type.Flag.Volatile)	(void)printf("volatile ");
      if(Before[Before_Top]->MPC_Type.Flag.Repl)	(void)printf("repl ");
    }
    else
      fprintf(stderr,"mpC Advisor: Internal error");
  } /* end while */
}

/*******************/
static void PushAfter
/*******************/
#if defined __STDC__ || defined __cplusplus
 (tTree Decl)
#else
 (Decl) tTree Decl;
#endif
{
  After[After_Top++] = Decl;
}

/********************/
static void WriteAfter
/********************/
 ARGS((void))
{
  int i=0;
  tTree node;

  while(i<After_Top) {
    if(After[i]==NoTree)
      (void)printf(")");
    else
      switch(After[i]->Kind) {
	case kMPC_ArrayType:
		if(After[i]->MPC_ArrayType.NumberOfComponents)  /* const or undef. dimension: */
		  if(After[i]->MPC_ArrayType.NumberOfComponents==UNDEFINED)
		    (void)printf("[");
		  else
		    (void)printf("[%d", After[i]->MPC_ArrayType.NumberOfComponents);
		else  /* non-const (parameter) dimension: */
		  (void)printf("["),Write_MPC(After[i]->MPC_ArrayType.ArraySize, PART);
		if(After[i]->MPC_ArrayType.Step!=1)
		  (void)printf(":%d", After[i]->MPC_ArrayType.Step);
		(void)printf("]");
		break;
	case kMPC_VectorType:
		(void)printf("[%d]", After[i]->MPC_VectorType.NumberOfComponents);
		break;
	case kMPC_FunctionType:
		(void)printf("(...)");
		break;
	/********  -- this code contains one error: the queue 'After' must be 'flexible'
		      (like 'TypeStack') for correct printing of the function specifier !!!
		(void)printf("(");
		node = After[i]->MPC_FunctionType.ParamList;
		while(node!=NoTree && node->Kind!=kMPC_FreeNode) {
		  Write_MPC(node, PART);
		  if(node->MPC_VarDecl.Next!=NoTree &&
		     node->MPC_VarDecl.Next->Kind!=kMPC_FreeNode)
		    (void)printf(", ");
		  node = node->MPC_VarDecl.Next;
		}
		(void)printf(")");
		break;
	********/
	default:
		fprintf(stderr,"mpC Advisor: Internal error");
      } /* end switch; */
  i++;
  } /* end while; */
  After_Top = 0;
}

/***********************/
static tTree TerminalType
/***********************/
#if defined __STDC__ || defined __cplusplus
 (tTree Type)
#else
 (Type) tTree Type;
#endif
{
  if(Type==NoTree) return NoTree;
  switch(Type->Kind) {
    case kMPC_ArrayType:	return TerminalType(Type->MPC_ArrayType.ElementType);
    case kMPC_VectorType:	return TerminalType(Type->MPC_VectorType.ElementType);
    case kMPC_PointerType:	return TerminalType(Type->MPC_PointerType.ElementType);
    case kMPC_FunctionType:	return TerminalType(Type->MPC_FunctionType.ResultType);
    default:			return Type;
  }
}
