
   /************************************************/
   /* Implementation of Partitioning Interfaces of */
   /*        Sets                                  */
   /*                                              */
   /* Revision history                             */
   /* 20-05-2003  --      Initial version          */
   /************************************************/
    
   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>

   #include <math.h>
   #include <hmpi.h>

   /*-----------------------------------------------------*/

   /*
    * All the processors are homogeneous. That is they exhibit
    * same speeds. However each processor has an upper bound on
    * number of elements it can store.
    * This is of complexity O(p*p)
    */
   int __HMPI_Homogeneous_distribution_with_mlimits
   ( 
       int p,
       int n,
       const int* mlimits,
       int *np
   )
   {
       int i, j, rc;
       int sum = 0;
       int bound_exceeded = 0;

       for (i = 0; i < p; i++)
       {
           np[i] = n/p;
       }

       np[0] += n%p;

       for (i = 0; i < p; i++)
       {
           if (np[i] > mlimits[i])
           {
              np[i] = mlimits[i];
              bound_exceeded = 1;
              break;
           }
       }

       /*
        * For all the processors whose upper bounds 
        * are exceeded, we assign the number of elements 
        * equal to their upper bounds. However we proceed
        * by one processor at the time. TBD
        */
       if (bound_exceeded == 0)
       {
          return HMPI_OK;
       }

       {
          int *boundsm, *npm;
          int nm = n - mlimits[i];
          int ind = 0;

          npm = (int*)malloc(
                      sizeof(int)
                      *
                      (p-1)
          );

          if (npm == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          boundsm = (int*)malloc(
                          sizeof(int)
                          *
                          (p-1)
          );

          if (boundsm == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (j = 0; j < p; j++)
          {
             if (j == i)
             {
                continue;
             }

             npm[ind] = np[j];
             boundsm[ind] = mlimits[j];
             ind++;
          }

          rc = __HMPI_Homogeneous_distribution_with_mlimits(
                   p-1,
                   nm,
                   boundsm,
                   npm
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          for (ind  = 0, j = 0; j < p; j++)
          {
             if (j == i)
             {
                continue;
             }

             np[j] = npm[ind];
             ind++;
          }

          free(boundsm);
          free(npm);
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * This is for ordered sets.
    * All the processors are homogeneous. That is they exhibit
    * same speeds. However each processor has an upper bound on
    * number of elements it can store. The sum of weights in each
    * partition should be the same.
    */
   int __HMPI_Homogeneous_distribution_with_mlimits_and_weights_ordered_sets
   ( 
       int p,
       int n,
       const int* mlimits,
       const int* w,
       int *np
   )
   {
       int i, j, k, rc;
       double sumw = 0;
       double sumwcum = 0;
       int *Size_of_bin;
       double *wallocations;
       double sumcum = 0;

       wallocations = (double*)malloc(
                            sizeof(double)
                            *
                            (p+1)
       );

       if (wallocations == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       Size_of_bin  = (int*)malloc(
                            sizeof(int)
                            *
                            p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < n; i++)
       {
           sumw += w[i];
       }

       for (i = 0; i < p; i++)
       {
           Size_of_bin[i] = sumw/p;
           sumwcum += Size_of_bin[i];
           np[i] = 0;
       }

       Size_of_bin[0] += sumw - sumwcum;

       wallocations[0] = 0.0;
       for (i = 1; i <= p; i++)
       {
           wallocations[i] = wallocations[i-1] + Size_of_bin[i-1];
       }

       for (i = 0; i < n; i++)
       {
           sumcum += w[i];

           for (j = 0; j < p; j++)
           {
               if ((sumcum > wallocations[j])
                   && (sumcum <= wallocations[j+1]
                  )
               )
               {
                  if (j == (p-1))
                  {
                     np[j] = n-i;

                     free(wallocations);
                     free(Size_of_bin);

                     if (np[j] > mlimits[j])
                     {
                        return HMPI_ERR_PARTITION_SET;
                     }  

                     return HMPI_OK;
                  }

                  np[j]++;

                  if (np[j] > mlimits[j])
                  {
                     np[j] = mlimits[j];

                     free(wallocations);
                     free(Size_of_bin);

                     return __HMPI_Homogeneous_distribution_with_mlimits_and_weights_ordered_sets(
                              p-(j+1),
                              n-i,
                              mlimits+j+1,
                              w+i,
                              np+j+1
                     );
                  }
               }
           }
      }

      free(wallocations);
      free(Size_of_bin);

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Partition_set_homogeneous
   (
       int p,
       const int *mlimits,
       int n,
       const int *w,
       int ordering,
       int processor_ordering,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, rc;
       int sum = 0;
       double sumd = 0;

       /*
        * Homogeneous distribution for unordered sets.
        *
	* The following criterion and restriction should 
	* be satisfied:
	* The number of elements in each partition should be 
	* proportional to the speed of the processor 
	* owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
	*/
       if ((w == NULL) 
           && (mlimits == NULL) 
           && (ordering == 0
          )
       )
       {
          if (n < p)
          {
             for (i = 0; i < n; i++)
             {
                 np[i] = 1;
             }

             for (i = n; i < p; i++)
             {
                 np[i] = 0;
             }

             return HMPI_OK;
          }

          for (i = 0; i < p; i++)
          {
             np[i] = n/p;
          }

          np[0] += n%p;

	  return HMPI_OK;
       }

       /*
        * There is an upper bound on the number of elements
        * that each processor can store.
        */
       if ((w == NULL) 
           && (mlimits != NULL) 
           && (ordering == 0
          )
       )
       {
          for (i = 0; i < p; i++)
          {
             sumd += mlimits[i];
          }

          if (sumd < n)
          {
             printf(
               "This problem size %d cannot be solved, "
               "sum of upper bounds on the number of elements exceeded\n",
               n
             );

             return HMPI_ERR_MLIMITS;
          }

          if (sumd == n)
          {
             for (i = 0; i < p; i++)
             {
                np[i] = mlimits[i];
             }

             return HMPI_OK;
          }

          if (n < p)
          {
             for (i = 0; i < n; i++)
             {
                 np[i] = 1;
             }
            
             for (i = n; i < p; i++)
             {
                 np[i] = 0;
             }
            
             return HMPI_OK;
          }

          rc = __HMPI_Homogeneous_distribution_with_mlimits(
                   p,
                   n,
                   mlimits,
                   np
          );

          if (rc != HMPI_OK)
          {
             printf(
               "Problems with homogeneous partitioning of set "
               "with upper bounds on the number of elements "
               "that can be stored by each processor\n"
             );

             return HMPI_ERR_PARTITION_SET;
          }

	  return HMPI_OK;
       }

       /*
        * This is a NP-hard problem.
        * The set should be split such that the sum 
        * of the weights in each subset is the same.
        * A naive implementation is provided here
        * This is of complexity O(n*n).
        * Total complexity = O(n*n) + O(n*p)
        *                    ^^^^^^
        *              sorting of weights
        */
       if ((w != NULL)
           && (mlimits == NULL)
           && (ordering == 0
          )
       )
       {
          int *Size_of_bin, *Current_bin_capacity;
          double sumw = 0;
          double sumwcum = 0;
          int *rearranged_weights;
          double *speeds;
          int *rearrangedw;
          int temp, temp_number, *allocations, *chosen;

          Size_of_bin  = (int*)malloc(
                               sizeof(int)
                               *
                               p
          );

          if (Size_of_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          Current_bin_capacity = (int*)malloc(
                                       sizeof(int)
                                       *
                                       p
          );

          if (Current_bin_capacity == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumw += w[i];
          }

          for (i = 0; i < p; i++)
          {
             Size_of_bin[i] = sumw/p;
             sumwcum += Size_of_bin[i];
             Current_bin_capacity[i] = 0;
          }

          Size_of_bin[0] += sumw - sumwcum;

          /*
	   * We rearrange the element weights 
	   * in descending order.
	   */
          {
             rearranged_weights = (int*)malloc(
                                        sizeof(int)
		      		        *
				        n
             );

             if (rearranged_weights == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rearrangedw = (int*)malloc(
                                 sizeof(int)
		   	         *
			         n
             );

             if (rearrangedw == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < n; i++)
             {
                 rearrangedw[i] = i;
                 rearranged_weights[i] = w[i];
             }

             for (i = 0; i < n; i++)
             {
                 for (j = 1; j < n; j++)
                 {
                     if (rearranged_weights[j-1] < rearranged_weights[j])
                     {
                        temp = rearranged_weights[j-1];
		        rearranged_weights[j-1] = rearranged_weights[j];
		        rearranged_weights[j] = temp;

		        temp_number = rearrangedw[j-1];
		        rearrangedw[j-1] = rearrangedw[j];
		        rearrangedw[j] = temp_number;
                     }
                 }
             }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              n
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          chosen = (int*)malloc(
                         sizeof(int)
                         *
                         n
          );

          if (chosen == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             chosen[i] = 0;
          }

          /*
           * Fill into the bins until the bins are full
           * or until any addition exceeds the capacity
           */
          for (i = 0; i < p; i++)
          {
             for (j = 0; j < n; j++)
             {
                if (chosen[j] == 1)
                {
                   continue;
                }

                if (Current_bin_capacity[i] == Size_of_bin[i])
                {
                   continue;
                }

                if ((Current_bin_capacity[i] + rearranged_weights[j]) > Size_of_bin[i])
                {
                   continue;
                }

                Current_bin_capacity[i] += rearranged_weights[j];
                allocations[j] = i;
                chosen[j] = 1;
             }
          }

          /*
           * Fill into the bin that causes the minumum 
           * waste
           */
          for (i = 0; i < n; i++)
          {
              int temp = INT_MAX;
              int optimal_bin;

              if (chosen[i] == 1)
              {
                 continue;
              }

              for (j = 0; j < p; j++)
              {
                 int waste = Current_bin_capacity[j] 
                             + 
                             rearranged_weights[i]
                             -
                             Size_of_bin[j];

                 if (waste < temp)
                 {
                    temp = waste;
                    optimal_bin = j;
                 }
              }

              Current_bin_capacity[optimal_bin] += rearranged_weights[i];
              allocations[i] = optimal_bin;
              chosen[i] = 1;
          }

          for (i = 0; i < n; i++)
          {
             np[rearrangedw[i]] = allocations[i];
          }

          free(rearranged_weights);
          free(rearrangedw);
          free(allocations);
          free(chosen);

          if (metric == NULL)
          {
             free(Size_of_bin);
             free(Current_bin_capacity);

             return HMPI_OK;
          }

          speeds = (double*)malloc(
                            sizeof(double)
                            *
                            p
          );

          if (speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             speeds[i] = 1.0;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for array elements
           * of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
               {
                  *metric = (*umf)(
                             p,
                             speeds,
                             Current_bin_capacity,
                             Size_of_bin
                  );
               }
               break;
             case SYSTEM_DEFINED:
               {
                  *metric = __HMPI_System_defined_metric(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
               }
               break;
             default:
               {
                  return HMPI_ERR_METRIC;
               }
               break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(speeds);

          return HMPI_OK;
       }

       /*
        * This is an NP-hard problem.
        * The set should be split such that the 
        * sum of the weights in each subset is the same and the 
        * number of elements assigned to each processor 
        * must not exceed the upper bound it can store.
        * A naive implementation is provided here
        */
       if ((w != NULL)
           && (mlimits != NULL)
           && (ordering == 0
          )
       )
       {
          int *Size_of_bin, *Current_bin_capacity;
          double *speeds;
          int suml = 0;
          double sumw = 0;
          double sumwcum = 0;
          int *rearranged_weights;
          int temp, temp_number, temp_mlimit, *allocations, *chosen, *Number_in_bin;
          int *rearrangedw;
          int *rearranged_mlimits, *rearrangedp;

          for (i = 0; i < p; i++)
          {
              sumd += mlimits[i];
          }

          if (sumd < n)
          {
             printf(
                "This problem %d size cannot be solved, "
                "sum of upper bounds exceeded\n"
                , n
             );

             return HMPI_ERR_MLIMITS;
          }

          Size_of_bin  = (int*)malloc(
                               sizeof(int)
                               *
                               p
          );

          if (Size_of_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          Current_bin_capacity = (int*)malloc(
                                       sizeof(int)
                                       *
                                       p
          );

          if (Current_bin_capacity == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumw += w[i];
          }

          for (i = 0; i < p; i++)
          {
             Current_bin_capacity[i] = 0;
             Size_of_bin[i] = sumw/p;
             sumwcum += Size_of_bin[i];
          }

          Size_of_bin[0] += sumw - sumwcum;

          /*
	   * We rearrange the element weights 
	   * in descending order.
	   */
          {
             rearranged_weights = (int*)malloc(
                                        sizeof(int)
		      		        *
				        n
             );

             if (rearranged_weights == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rearrangedw = (int*)malloc(
                                 sizeof(int)
		   	         *
			         n
             );

             if (rearrangedw == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < n; i++)
             {
                 rearrangedw[i] = i;
                 rearranged_weights[i] = w[i];
             }

             for (i = 0; i < n; i++)
             {
                 for (j = 1; j < n; j++)
                 {
                     if (rearranged_weights[j-1] < rearranged_weights[j])
                     {
                        temp = rearranged_weights[j-1];
		        rearranged_weights[j-1] = rearranged_weights[j];
		        rearranged_weights[j] = temp;

		        temp_number = rearrangedw[j-1];
		        rearrangedw[j-1] = rearrangedw[j];
		        rearrangedw[j] = temp_number;
                     }
                 }
             }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              n
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          speeds = (double*)malloc(
                            sizeof(double)
                            *
                            p
          );

          if (speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             speeds[i] = 1.0;
          }

          if (sumd == n)
          {
             int ind = 0;

             /*
              * TBD:
              * This looks like a NP-hard problem.
              * We know the number of elements in each subset
              * given by the upper bound.
              * The sum of weights of the elements in each
              * subset should be the same.
              * We provide a naive implementation here.
              * This is of complexity O(n*n).
              * We arrange the processors in increasing
              * order of their upper bounds and we arrange
              * the weights in decreasing order.
              */
             rearranged_mlimits = (int*)malloc(
                                        sizeof(int)
                                        *
                                        p
             );

             if (rearranged_mlimits == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rearrangedp = (int*)malloc(
                                 sizeof(int)
		      	         *
			         p
             );

             if (rearrangedp == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < p; i++)
             {
                 rearrangedp[i] = i;
                 rearranged_mlimits[i] = mlimits[i];
             }

             for (i = 0; i < p; i++)
             {
                 for (j = 1; j < p; j++)
                 {
                     if (rearranged_mlimits[j-1] > rearranged_mlimits[j])
                     {
		        temp_number = rearrangedp[j-1];
		        rearrangedp[j-1] = rearrangedp[j];
		        rearrangedp[j] = temp_number;

                        temp_mlimit = rearranged_mlimits[j-1];
                        rearranged_mlimits[j-1] = rearranged_mlimits[j];
                        rearranged_mlimits[j] = temp_mlimit;
                     }
                 }
             }
                 
             for (i = 0; i < p; i++)
             {
                for (j = 0; j < rearranged_mlimits[i]; j++)
                {
                    allocations[ind] = i;
                    Current_bin_capacity[rearrangedp[i]] += rearranged_weights[ind];
                    ind++;
                }
             }

             for (i = 0; i < n; i++)
             {
                np[rearrangedw[i]] = rearrangedp[allocations[i]];
             }

             free(rearranged_weights);
             free(rearrangedw);
             free(rearranged_mlimits);
             free(rearrangedp);
             free(allocations);

             if (metric == NULL)
             {
                free(Size_of_bin);
                free(Current_bin_capacity);
                free(speeds);

                return HMPI_OK;
             }

             /*
              * The ideal sum of weights is given by
              * elements of array Size_of_bin and the
              * actual sum of weights is calculated for array elements
              * of Current_bin_capacity.
              */
             switch (type_of_metric)
             {
                case USER_SPECIFIED:
                  {
                     *metric = (*umf)(
                                p,
                                speeds,
                                Current_bin_capacity,
                                Size_of_bin
                     );
                  }
                  break;
                case SYSTEM_DEFINED:
                  {
                     *metric = __HMPI_System_defined_metric(
                               p,
                               speeds,
                               Current_bin_capacity,
                               Size_of_bin
                     );
                  }
                  break;
                default:
                  {
                     return HMPI_ERR_METRIC;
                  }
                  break;
             }

             free(Size_of_bin);
             free(Current_bin_capacity);
             free(speeds);

             return HMPI_OK;
          }

          Number_in_bin = (int*)malloc(
                                sizeof(int)
                                *
                                p
          );

          if (Number_in_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             Number_in_bin[i] = 0;
          }

          chosen = (int*)malloc(
                         sizeof(int)
                         *
                         n
          );

          if (chosen == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             chosen[i] = 0;
          }

          for (i = 0; i < p; i++)
          {
             for (j = 0; j < n; j++)
             {
                if (chosen[j] == 1)
                {
                   continue;
                }

                if (Current_bin_capacity[i] == Size_of_bin[i])
                {
                   continue;
                }

                if ((Current_bin_capacity[i] + rearranged_weights[j]) > Size_of_bin[i])
                {
                   continue;
                }

                if ((Number_in_bin[i] + 1) > mlimits[i])
                {
                   break;
                }

                Number_in_bin[i]++;
                Current_bin_capacity[i] += rearranged_weights[j];
                allocations[j] = i;
                chosen[j] = 1;
             }
          }

          for (i = 0; i < n; i++)
          {
              int temp = INT_MAX;
              int optimal_bin = -1;

              if (chosen[i] == 1)
              {
                 continue;
              }

              for (j = 0; j < p; j++)
              {
                 int waste;

                 if (Number_in_bin[j] >= mlimits[j])
                 {
                    continue;
                 }
                 
                 waste = Current_bin_capacity[j] 
                         + 
                         rearranged_weights[i]
                         -
                         Size_of_bin[j];

                 if (waste < temp)
                 {
                    temp = waste;
                    optimal_bin = j;
                 }
              }

              Current_bin_capacity[optimal_bin] += rearranged_weights[i];
              allocations[i] = optimal_bin;
              chosen[i] = 1;
          }

          for (i = 0; i < n; i++)
          {
             np[rearrangedw[i]] = allocations[i];
          }

          free(rearranged_weights);
          free(rearrangedw);
          free(allocations);
          free(chosen);
          free(Number_in_bin);

          if (metric == NULL)
          {
             free(Size_of_bin);
             free(Current_bin_capacity);
             free(speeds);

             return HMPI_OK;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for 
           * array elements of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
               {
                  *metric = (*umf)(
                             p,
                             speeds,
                             Current_bin_capacity,
                             Size_of_bin
                  );
               }
               break;
             case SYSTEM_DEFINED:
               {
                  *metric = __HMPI_System_defined_metric(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
               }
               break;
             default:
               {
                  return HMPI_ERR_METRIC;
               }
               break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(speeds);

          return HMPI_OK;
       }

       /*
        * Homogeneous distribution for ordered sets
        *
	* The following criterion and restriction should 
	* be satisfied:
	* The number of elements in each partition should be 
	* proportional to the speed of the processor 
	* owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
        *
        */
       if ((w == NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 1
          )
       )
       {
          if (n < p)
          {
             np[0] = 0;

             for (i = 1; i <= n; i++)
             {
                np[i] = np[i-1] + 1;
             }
            
             for (i = n+1; i <= p; i++)
             {
                np[i] = np[n];
             }
            
             return HMPI_OK;
          }

          np[0] = 0;
          np[1] = n/p + n%p;

          for (i = 2; i <= p; i++)
          {
             np[i] = np[i-1] + (n/p);
          }

	  return HMPI_OK;
       }

       if ((w == NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 1
          )
       )
       {
          int *allocations;

          for (i = 0; i < p; i++)
          {
              sumd += mlimits[i];
          }

          if (sumd < n)
          {
             printf(
               "This problem size %d cannot be solved, "
               "sum of memory bounds exceeded\n",
               n
             );

             return HMPI_ERR_MLIMITS;
          }

          if (sumd == n)
          {
             np[0] = 0;

             for (i = 1; i <= p; i++)
             {
                np[i] = np[i-1] + mlimits[i-1];
             }

             return HMPI_OK;
          }

          if (n < p)
          {
             np[0] = 0;

             for (i = 1; i <= n; i++)
             {
                np[i] = np[i-1] + 1;
             }

             for (i = n+1; i <= p; i++)
             {
                np[i] = np[n];
             }

             return HMPI_OK;
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Homogeneous_distribution_with_mlimits(
                   p,
                   n,
                   mlimits,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             printf(
               "Problems with homogeneous partitioning of set "
               "with memory bounds on the number of elements "
               "that can be stored by each processor\n"
             );

             return HMPI_ERR_PARTITION_SET;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);

	  return HMPI_OK;
       }

       /*
	* The following criterion and restriction should 
	* be satisfied:
	* The sum of weights of the elements in each 
	* partition should be proportional to the speeda
	* of the processor owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
        *
        * This case is of complexity O(n*p)
        */
       if ((w != NULL)
           && (mlimits == NULL)
           && (ordering == 1
          )
       )
       {
          int *Size_of_bin;
          double *wallocations;
          int *allocations;
          double sumcum = 0;
          double sumwcum = 0;
          double sumw = 0;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          wallocations = (double*)malloc(
                               sizeof(double)
                               *
                               (p+1)
          );

          if (wallocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          Size_of_bin  = (int*)malloc(
                               sizeof(int)
                               *
                               p
          );

          if (Size_of_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumw += w[i];
          }

          for (i = 0; i < p; i++)
          {
             Size_of_bin[i] = sumw/p;
             sumwcum += Size_of_bin[i];
             allocations[i] = 0;
          }

          Size_of_bin[0] += sumw - sumwcum;

          wallocations[0] = 0.0;
          for (i = 1; i <= p; i++)
          {
              wallocations[i] = wallocations[i-1] + Size_of_bin[i-1];
          }

          for (i = 0; i < n; i++)
          {
              sumcum += w[i];

              for (j = 0; j < p; j++)
              {
                  if ((sumcum > wallocations[j])
                      && (sumcum <= wallocations[j+1]
                     )
                  )
                  {
                     allocations[j]++;
                     break;
                  }
              }
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);
          free(wallocations);
          free(Size_of_bin);

          return HMPI_OK;
       }

       /*
        * This looks like a NP-hard problem.
        * Processors cannot be reordered.
        * We allocate the elements taking into
        * account the upper bounds of the processors and
        * also the proportionality of the speeds of the  
        * processors to the sum of weights of the elements.
        * It could happen that at the final step, the upper 
        * upper bound of the final processor is exceeded in 
        * which case we just allocate the number of elements 
        * equal to their upper bounds right from the start.
        * This is just an approximation.
        */
       if ((w != NULL) 
           && (mlimits != NULL) 
           && (ordering == 1)
           && (processor_ordering == 0
          )
       )
       {
          int *Size_of_bin;
          int *Current_bin_capacity;
          int *allocations;
          double sumw = 0;
          double sumwcum = 0;
          int ind = 0;
          double *speeds;

          for (i = 0; i < p; i++)
          {
              sumd += mlimits[i];
          }

          if (sumd < n)
          {
             printf(
               "This problem size %d cannot be solved, "
               "sum of upper bounds exceeded\n",
               n
             );

             return HMPI_ERR_MLIMITS;
          }

          if (sumd == n)
          {
             np[0] = 0;

             for (i = 1; i <= p; i++)
             {
                np[i] = np[i-1] + mlimits[i-1];
             }

             return HMPI_OK;
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Homogeneous_distribution_with_mlimits_and_weights_ordered_sets(
                   p,
                   n,
                   mlimits,
                   w,
                   allocations
          );

          if (rc == HMPI_ERR_PARTITION_SET)
          {
             rc = __HMPI_Homogeneous_distribution_with_mlimits(
                     p,
                     n,
                     mlimits,
                     allocations
             );

             if (rc != HMPI_OK)
             {
                printf(
                  "Problems with homogeneous partitioning of set "
                  "with memory bounds on the number of elements "
                  "that can be stored by each processor\n"
                );

                return HMPI_ERR_PARTITION_SET;
             }
          }

          if (rc != HMPI_OK)
          {
             return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          if (metric == NULL)
          {
             free(allocations);
             return HMPI_OK;
          }

          Size_of_bin  = (int*)malloc(
                               sizeof(int)
                               *
                               p
          );

          if (Size_of_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          Current_bin_capacity = (int*)malloc(
                                       sizeof(int)
                                       *
                                       p
          );

          if (Current_bin_capacity == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumw += w[i];
          }

          for (i = 0; i < p; i++)
          {
             Current_bin_capacity[i] = 0;
             Size_of_bin[i] = sumw/p;
             sumwcum += Size_of_bin[i];
          }

          Size_of_bin[0] += sumw - sumwcum;

          for (i = 0; i < p; i++)
          {
             for (j = 0; j < allocations[i]; j++)
             {
                Current_bin_capacity[i] += w[ind++];
             }
          }

          speeds = (double*)malloc(
                            sizeof(double)
                            *
                            p
          );

          if (speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             speeds[i] = 1.0;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for array elements
           * of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
               {
                  *metric = (*umf)(
                             p,
                             speeds,
                             Current_bin_capacity,
                             Size_of_bin
                  );
               }
               break;
             case SYSTEM_DEFINED:
               {
                  *metric = __HMPI_System_defined_metric(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
               }
               break;
             default:
               {
                  return HMPI_ERR_METRIC;
               }
               break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(speeds);
          free(allocations);

          return HMPI_OK;
       }

       /*
        * This looks like a NP-hard problem.
        * Processors can be reordered.
        */
       if ((w != NULL) 
           && (mlimits != NULL) 
           && (ordering == 1)
           && (processor_ordering == 1
          )
       )
       {
          int *Size_of_bin;
          int *Current_bin_capacity;
          int *allocations;
          int ind = 0;
          double sumw = 0;
          double sumwcum = 0;
          int *rearranged_mlimits, *rearrangedp;
          int temp;
          double *speeds;

          for (i = 0; i < p; i++)
          {
              sumd += mlimits[i];
          }

          if (sumd < n)
          {
             printf(
               "This problem size %d cannot be solved, "
               "sum of upper bounds exceeded\n",
               n
             );

             return HMPI_ERR_MLIMITS;
          }

          if (sumd == n)
          {
             for (i = 0; i < p; i++)
             {
                np[2*i] = i;
             }

             for (i = 0; i < p; i++)
             {
                np[2*i+1] = mlimits[i];
             }

             return HMPI_OK;
          }

          rearranged_mlimits = (int*)malloc(
                                     sizeof(int)
                                     *
                                     p
          );

          if (rearranged_mlimits == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
	         	      *
			      p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             rearrangedp[i] = i;
             rearranged_mlimits[i] = mlimits[i];
          }

          for (i = 0; i < p; i++)
          {
             for (j = 1; j < p; j++)
             {
                 if (rearranged_mlimits[j-1] > rearranged_mlimits[j])
                 {
	            temp = rearrangedp[j-1];
	            rearrangedp[j-1] = rearrangedp[j];
		    rearrangedp[j] = temp;

                    temp = rearranged_mlimits[j-1];
                    rearranged_mlimits[j-1] = rearranged_mlimits[j];
                    rearranged_mlimits[j] = temp;
                 }
             }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Homogeneous_distribution_with_mlimits_and_weights_ordered_sets(
                   p,
                   n,
                   rearranged_mlimits,
                   w,
                   allocations
          );

          if (rc == HMPI_ERR_PARTITION_SET)
          {
             rc = __HMPI_Homogeneous_distribution_with_mlimits(
                      p,
                      n,
                      rearranged_mlimits,
                      allocations
             );

             if (rc != HMPI_OK)
             {
                printf(
                  "Problems with homogeneous partitioning of set "
                  "with memory bounds on the number of elements "
                  "that can be stored by each processor\n"
                );

                return HMPI_ERR_PARTITION_SET;
             }
          }

          if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (ind = 0, i = 0; i < p; i++)
          {
             np[ind++] = rearrangedp[i];
             np[ind++] = allocations[i];
          }

          if (metric == NULL)
          {
             free(allocations);
	     free(rearranged_mlimits);
	     free(rearrangedp);

             return HMPI_OK;
          }

          Size_of_bin  = (int*)malloc(
                               sizeof(int)
                               *
                               p
          );

          if (Size_of_bin == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          Current_bin_capacity = (int*)malloc(
                                       sizeof(int)
                                       *
                                       p
          );

          if (Current_bin_capacity == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumw += w[i];
          }

          for (i = 0; i < p; i++)
          {
             Current_bin_capacity[i] = 0;
             Size_of_bin[rearrangedp[i]] = sumw/p;
             sumwcum += Size_of_bin[i];
          }

          Size_of_bin[rearrangedp[0]] += sumw - sumwcum;

          for (i = 0; i < p; i++)
          {
             for (j = 0; j < allocations[i]; j++)
             {
                Current_bin_capacity[i] += w[ind++];
             }
          }

          speeds = (double*)malloc(
                            sizeof(double)
                            *
                            p
          );

          if (speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             speeds[i] = 1.0;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for array elements
           * of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
               {
                  *metric = (*umf)(
                             p,
                             speeds,
                             Current_bin_capacity,
                             Size_of_bin
                  );
               }
               break;
             case SYSTEM_DEFINED:
               {
                  *metric = __HMPI_System_defined_metric(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
               }
               break;
             default:
               {
                  return HMPI_ERR_METRIC;
               }
               break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(speeds);
          free(allocations);
	  free(rearranged_mlimits);
	  free(rearrangedp);

          return HMPI_OK;
       }

       printf("Parameters provided are invalid\n");
       return HMPI_ERR_PARTITION_SET;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Speeds_are_single_numbers_with_mlimits
   (
       int p,
       const double *speeds,
       const int *bounds,
       int n,
       int *np
   )
   {
       int i, j, rc;
       int bound_exceeded = 0;
       int sum = 0;
       double sumd = 0;

       for (i = 0; i < p; i++)
       {
          sumd += bounds[i];
       }

       if (sumd < n)
       {	        
          printf(
            "The problem size %d cannot be solved because "
            "memory bounds on the number of elements "
            "that can be stored by each processor are exceeded\n"
          );

          return HMPI_ERR_PARTITION_SET;
       }

       if (sumd == n)
       {	        
          for (i = 0; i < p; i++)
          {
             np[i] = bounds[i];
          }

	  return HMPI_OK;
       }

       rc = __HMPI_Number_of_elements_proportional_to_speed(
                p,
	        n,
	        speeds,
	        np
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           if (np[i] > bounds[i])
           {
              np[i] = bounds[i];
              bound_exceeded = 1;
              break;
           }
       }

       if (bound_exceeded == 1)
       {
          int k;
          int ind = 0;
          int sind = 0;
          double *speedsm;
          int *npm;
          int *boundsm;
          int nm = n - bounds[i];

          speedsm = (double*)malloc(
                             sizeof(double)
                             *
                             (p-1)
          );        

          if (speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          npm = (int*)malloc(
                      sizeof(int)
                      *
                      (p-1)
          );        

          if (npm == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          boundsm = (int*)malloc(
                          sizeof(int)
                          *
                          (p-1)
          );        

          if (boundsm == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (j = 0; j < p; j++)
          {
             if (j == i)
             {
                continue;
             }

             npm[ind] = np[j];
             boundsm[ind] = bounds[j];
             speedsm[sind++] = speeds[j];
             ind++;
          }

          rc = __HMPI_Speeds_are_single_numbers_with_mlimits(
                   p-1,
                   speedsm,
                   boundsm,
                   nm,
                   npm
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          for (ind = 0, j = 0; j < p; j++)
          {
             if (j == i)
             {
                continue;
             }

             np[j] = npm[ind];
             ind++;
          }

          free(speedsm);
          free(boundsm);
          free(npm);
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_set
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int n,
       const int *w,
       int ordering,
       int processor_ordering,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, rc;

       /*
	* Check the parameters provided by the user
        * If w is not NULL and the set is well ordered, then the
        * user has to specify if the implementations can reorder
        * the processors before partitioning.
	*/
       if ((w != NULL) 
           && (ordering == 1
          )
       )
       {
          if ((speeds == NULL)
              && (mlimits != NULL
             )
          )
          {
             if ((processor_ordering != 0) 
                 && (processor_ordering != 1
                 )
             )
             {
                printf(
                  "If weights of the elements are not NULL and the set is well ordered, "
                  "processor reordering has to be 0 or 1\n"
                );
                return HMPI_ERR_PARTITION_SET;
             }
          }

          if (speeds != NULL)
          {
             if ((processor_ordering != 0) 
                 && (processor_ordering != 1
                 )
             )
             {
                printf(
                  "If weights of the elements are not NULL and the set is well ordered, "
                  "processor reordering has to be 0 or 1\n"
                );
                return HMPI_ERR_PARTITION_SET;
             }
          }
       }

       /*
	* Distribution of the set amongst processors
        * that are homogeneous
	*/
       if (speeds == NULL)
       {
          return __HMPI_Partition_set_homogeneous(
                   p,
	           mlimits,
	           n,
	           w,
	           ordering,
	           processor_ordering,
                   type_of_metric,
                   umf,
                   metric,
	           np
          );
       }

       /*
        * Heterogeneous distribution for non-ordered sets.
        *
	* The following criterion and restriction should 
	* be satisfied:
	* The number of elements in each partition should be 
	* proportional to the speed of the processor 
	* owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
        *
        * Speeds are single numbers, Set elements has 
        * no weights and no bounds on the number of 
        * elements that can be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 0)
           && (pn == 1
          )
       )
       {
          return __HMPI_Number_of_elements_proportional_to_speed(
                   p,
	           n,
		   speeds,
		   np
          );
       }

       /*
        * Speeds are single numbers, set elements have no 
        * weights and there is a limit on the number of 
        * elements that can be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits !=  NULL)
           && (ordering == 0)
           && (pn == 1
          )
       )
       {
          return __HMPI_Speeds_are_single_numbers_with_mlimits(
                 p,
                 speeds,
                 mlimits,
                 n,
                 np
          );
       }

       /*
        * Speeds are functions of problem size, set elements 
        * have no weights and no bounds on the number of 
        * elements that can be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 0)
           && (pn > 1
          )
       )
       {
          double *speeds_opt = (double*)malloc(
                                        sizeof(double)
                                        *
                                        p
          );

          if (speeds_opt == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Speed_function_of_problem_size(
                 p,
                 pn,
                 speeds,
                 psizes,
                 n,
                 speeds_opt,
                 np
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          free(speeds_opt);

          return HMPI_OK;
       }

       /*
        * Speeds are functions of problem size, no weights and there
        * is a limit on the number of elements that can 
        * be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 0)
           && (pn > 1
          )
       )
       {
          double *speeds_opt = (double*)malloc(
                                        sizeof(double)
                                        *
                                        p
          );

          if (speeds_opt == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Speed_function_of_problem_size_with_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 n,
                 speeds_opt,
                 np
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          free(speeds_opt);

          return HMPI_OK;
       }

       /*
	* The following criterion and restriction should 
	* be satisfied:
	* The sum of weights of the elements in each 
	* partition should be proportional to the speeds
	* of the processor owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
        *
        * Speeds are single numbers, set has weighted weights 
        * and no bounds on the number of elements
        * This is a NP-hard problem.
        * A naive implementation is provided here.
        * This is of complexity O(n*n)
        */
       if ((w != NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 0)
           && (pn == 1
          )
       )
       {
          double *rearranged_speeds;
          int *rearranged_weights;
	  int *rearrangedp;
	  int *rearrangedw;
          double temp;
	  int temp_number;
          int *allocations;

          rearranged_speeds = (double*)malloc(
                                       sizeof(double)
		   		       *
				       p
          );

          if (rearranged_speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
		   	      *
			      p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_speeds[i] = speeds[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_speeds[j-1] < rearranged_speeds[j])
                  {
                     temp = rearranged_speeds[j-1];
		     rearranged_speeds[j-1] = rearranged_speeds[j];
		     rearranged_speeds[j] = temp;

		     temp_number = rearrangedp[j-1];
		     rearrangedp[j-1] = rearrangedp[j];
		     rearrangedp[j] = temp_number;
                  }
              }
          }

          /*
	   * We rearrange the element weights 
	   * in descending order.
	   */
          {
             rearranged_weights = (int*)malloc(
                                        sizeof(int)
		      		        *
				        n
             );

             if (rearranged_weights == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rearrangedw = (int*)malloc(
                                 sizeof(int)
		   	         *
			         n
             );

             if (rearrangedw == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < n; i++)
             {
                 rearrangedw[i] = i;
                 rearranged_weights[i] = w[i];
             }

             for (i = 0; i < n; i++)
             {
                 for (j = 1; j < n; j++)
                 {
                     if (rearranged_weights[j-1] < rearranged_weights[j])
                     {
                        temp_number = rearranged_weights[j-1];
		        rearranged_weights[j-1] = rearranged_weights[j];
		        rearranged_weights[j] = temp_number;

		        temp_number = rearrangedw[j-1];
		        rearrangedw[j-1] = rearrangedw[j];
		        rearrangedw[j] = temp_number;
                     }
                 }
             }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              n
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Sum_of_weights_for_nonordered_set(
                   p,
		   n,
	           rearranged_speeds,
	           rearranged_weights,
                   type_of_metric,
                   umf,
                   metric,
	           allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (i = 0; i < n; i++)
          {
              np[rearrangedw[i]] = rearrangedp[allocations[i]];
          }

	  free(rearranged_speeds);
	  free(rearrangedp);
	  free(rearrangedw);
	  free(rearranged_weights);
	  free(allocations);

	  return HMPI_OK;
       }

       /*
        * Speeds are single numbers, set has weighted elements 
        * and there is a limit on the number of elements that can 
        * be stored by each processor.
        * This is a NP-hard problem.
        * A naive implementation is provided here.
        */
       if ((w != NULL) 
           && (mlimits !=  NULL)
           && (ordering == 0)
           && (pn == 1
          )
       )
       {
          double *rearranged_speeds;
          int *rearranged_weights;
	  int *rearrangedp;
	  int *rearrangedw;
          int *rearranged_mlimits;
          double temp;
	  int temp_number;
          int *allocations;
          int temp_mlimit;

          rearranged_mlimits = (int*)malloc(
                                     sizeof(int)
                                     *
                                     p
          );

          if (rearranged_mlimits == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearranged_speeds = (double*)malloc(
                                       sizeof(double)
		   		       *
				       p
          );

          if (rearranged_speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
		   	      *
			      p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_speeds[i] = speeds[i];
              rearranged_mlimits[i] = mlimits[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_speeds[j-1] < rearranged_speeds[j])
                  {
                     temp = rearranged_speeds[j-1];
		     rearranged_speeds[j-1] = rearranged_speeds[j];
		     rearranged_speeds[j] = temp;

		     temp_number = rearrangedp[j-1];
		     rearrangedp[j-1] = rearrangedp[j];
		     rearrangedp[j] = temp_number;

                     temp_mlimit = rearranged_mlimits[j-1];
                     rearranged_mlimits[j-1] = rearranged_mlimits[j];
                     rearranged_mlimits[j] = temp_mlimit;
                  }
              }
          }

          /*
	   * We rearrange the element weights 
	   * in descending order.
	   */
          {
             rearranged_weights = (int*)malloc(
                                        sizeof(int)
		      		        *
				        n
             );

             if (rearranged_weights == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rearrangedw = (int*)malloc(
                                 sizeof(int)
		   	         *
			         n
             );

             if (rearrangedw == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < n; i++)
             {
                 rearrangedw[i] = i;
                 rearranged_weights[i] = w[i];
             }

             for (i = 0; i < n; i++)
             {
                 for (j = 1; j < n; j++)
                 {
                     if (rearranged_weights[j-1] < rearranged_weights[j])
                     {
                        temp = rearranged_weights[j-1];
		        rearranged_weights[j-1] = rearranged_weights[j];
		        rearranged_weights[j] = temp;

		        temp_number = rearrangedw[j-1];
		        rearrangedw[j-1] = rearrangedw[j];
		        rearrangedw[j] = temp_number;
                     }
                 }
             }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              n
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Apply_mlimits_to_unordered_sum_of_weights (
                   p,
                   n,
                   rearranged_speeds,
                   rearranged_mlimits,
                   rearranged_weights,
                   type_of_metric,
                   umf,
                   metric,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (i = 0; i < n; i++)
          {
             np[rearrangedw[i]] = rearrangedp[allocations[i]];
          }

	  free(rearranged_speeds);
	  free(rearranged_mlimits);
	  free(rearranged_weights);
	  free(rearrangedp);
	  free(rearrangedw);
	  free(allocations);

	  return HMPI_OK;
       }

       /*
        * Speeds are functions of problem size
        * set with weighted elements and 
        * no bounds on the number of elements
        * No known results
        */
       if ((w != NULL)
           && (mlimits ==  NULL)
           && (ordering == 0)
           && (pn > 1
          )
       )
       {
          /*
           * Naive implementation
           */
          return __HMPI_Sum_of_weights_for_nonordered_set_speed_functions(
                   p,
                   pn,
                   speeds,
                   psizes,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   np
          );
       }

       /*
        * Speeds are functions of problem size, 
        * set with weighted elements and there
        * is a limit on the number of elements that can 
        * be stored by each processor.
        * No known results.
        */
       if ((w != NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 0)
           && (pn > 1
          )
       )
       {
          /*
           * Naive implementation
           */
          return __HMPI_Sum_of_weights_for_nonordered_set_speed_functions_with_mlimits(
                   p,
                   pn,
                   speeds,
                   psizes,
                   mlimits,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   np
          );
       }

       /*
        * Heterogeneous distribution for ordered sets.
        *
	* The following criterion and restriction should 
	* be satisfied:
	* The number of elements in each partition should be 
	* proportional to the speed of the processor 
	* owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
        *
        * Speeds are single numbers, no weights and no
        * bounds on the number of elements
        */
       if ((w == NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 1)
           && (pn == 1
          )
       )
       {
          int *allocations;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Number_of_elements_proportional_to_speed(
                   p,
	           n,
		   speeds,
		   allocations
          );

          if (rc != HMPI_OK)
          {
	     return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);

	  return HMPI_OK;
       }

       /*
        * Speeds are single numbers, no weights and there
        * is a limit on the number of elements that can 
        * be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits !=  NULL)
           && (ordering == 1)
           && (pn == 1
          )
       )
       {
          int *allocations;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Speeds_are_single_numbers_with_mlimits(
                 p,
                 speeds,
                 mlimits,
                 n,
                 allocations
          );

          if (rc != HMPI_OK)
          {
	     return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);

          return HMPI_OK;
       }

       /*
        * Speeds are functions of problem size
        * no weights and no bounds on the number of elements
        */
       if ((w == NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 1)
           && (pn > 1
          )
       )
       {
          double *speeds_opt;
          int *allocations;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          speeds_opt = (double*)malloc(
                                sizeof(double)
                                *
                                p
          );

          if (speeds_opt == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Speed_function_of_problem_size(
                 p,
                 pn,
                 speeds,
                 psizes,
                 n,
                 speeds_opt,
                 allocations
          );

          if (rc != HMPI_OK)
          {
	     return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(speeds_opt);
          free(allocations);

          return HMPI_OK;
       }

       /*
        * Speeds are functions of problem size, no weights and there
        * is a limit on the number of elements that can 
        * be stored by each processor
        */
       if ((w == NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 1)
           && (pn > 1
          )
       )
       {
          int *allocations;
          double *speeds_opt;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          speeds_opt = (double*)malloc(
                                sizeof(double)
                                *
                                p
          );

          if (speeds_opt == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Speed_function_of_problem_size_with_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 n,
                 speeds_opt,
                 allocations
          );

          if (rc != HMPI_OK)
          {
	     return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(speeds_opt);
          free(allocations);

          return HMPI_OK;
       }

       /*
	* The following criterion and restriction should 
	* be satisfied:
	* The sum of weights of the elements in each 
	* partition should be proportional to the speeda
	* of the processor owning that partition.
	*
	* The number of elements in each partition must 
	* be less than the maximum number of elements a 
	* processor can hold.
	*
        * Speeds are single numbers, set has weighted weights 
        * and no bounds on the number of elements
        * Processors cannot be reordered.
        * No known results.
        */
       if ((w != NULL) 
           && (mlimits ==  NULL)
           && (ordering == 1)
           && (pn == 1)
           && (processor_ordering == 0
          )
       )
       {
          int *allocations;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Sum_of_weights_for_ordered_set(
                   p,
		   n,
	           speeds,
	           w,
                   type_of_metric,
                   umf,
                   metric,
	           allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);

	  return HMPI_OK;
       }

       /*
        * Processors can be reordered.
        * We rearrange the processors in decreasing order
        * of speeds
        * No known results.
        * A naive implementation is provided here.
        */
       if ((w != NULL) 
           && (mlimits ==  NULL) 
           && (ordering == 1)
           && (pn == 1)
           && (processor_ordering == 1
          )
       )
       {
          double *rearranged_speeds;
	  int *rearrangedp;
	  double temp;
	  int temp_number;
          int *allocations;
          int ind;

          rearranged_speeds = (double*)malloc(
                                       sizeof(double)
		   		       *
				       p
          );

          if (rearranged_speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
		   	      *
			      p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_speeds[i] = speeds[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_speeds[j-1] < rearranged_speeds[j])
                  {
                     temp = rearranged_speeds[j-1];
		     rearranged_speeds[j-1] = rearranged_speeds[j];
		     rearranged_speeds[j] = temp;

		     temp_number = rearrangedp[j-1];
		     rearrangedp[j-1] = rearrangedp[j];
		     rearrangedp[j] = temp_number;
                  }
              }
          }

          allocations = (int*)malloc(
		              sizeof(int)
			      *
			      p
          );

	  if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Sum_of_weights_for_ordered_set(
                   p,
		   n,
	           rearranged_speeds,
	           w,
                   type_of_metric,
                   umf,
                   metric,
	           allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (ind = 0, i = 0; i < p; i++)
          {
             np[ind++] = rearrangedp[i];
             np[ind++] = allocations[i];
          }

	  free(rearranged_speeds);
	  free(rearrangedp);
	  free(allocations);

	  return HMPI_OK;
       }

       /*
        * Speeds are single numbers, set has weighted elements and there
        * is a limit on the number of elements that can 
        * be stored by each processor
        * Processors cannot be reordered.
        * No known results.
        * A naive implementation is provided here
        */
       if ((w != NULL) 
           && (mlimits !=  NULL)
           && (ordering == 1)
           && (pn == 1)
           && (processor_ordering == 0
          )
       )
       {
          int *allocations;
          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Apply_mlimits_to_ordered_sum_of_weights(
                   p,
                   n,
                   speeds,
                   mlimits,
                   w,
                   -1,
                   NULL,
                   NULL,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          /*
           * Metric is not calculated before.
           * Do it now.
           */
          if (metric != NULL)
          {
             int i, ind, j, sumw;
             int *Size_of_bin, *Current_bin_capacity;

             Size_of_bin  = (int*)malloc(
                                  sizeof(int)
                                  *
                                  p
             );

             if (Size_of_bin == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rc = __HMPI_Size_of_bins(
                      p,
                      n,
                      speeds,
                      w,
                      Size_of_bin,
                      &sumw
             );

             if (rc != HMPI_OK)
             {
                return rc;
             }

             Current_bin_capacity = (int*)malloc(
                                          sizeof(int)
                                          *
                                          p
             );

             if (Current_bin_capacity == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < p; i++)
             {
                Current_bin_capacity[i] = 0;
             }

             for (i = 0, ind = 0; i < p; i++)
             {
                 for (j = 0; j < allocations[i]; j++)
                 {
                     Current_bin_capacity[i] += w[ind++];
                 }
             }

             /*
              * The ideal sum of weights is given by
              * elements of array Size_of_bin and the
              * actual sum of weights is calculated for array elements
              * of Current_bin_capacity.
              */
             switch (type_of_metric)
             {
                case USER_SPECIFIED:
                  {
                     *metric = (*umf)(
                                p,
                                speeds,
                                Current_bin_capacity,
                                Size_of_bin
                     );
                  }
                  break;
                case SYSTEM_DEFINED:
                  {
                     *metric = __HMPI_System_defined_metric(
                               p,
                               speeds,
                               Current_bin_capacity,
                               Size_of_bin
                     );
                  }
                  break;
                default:
                  {
                     return HMPI_ERR_METRIC;
                  }
                  break;
             }

             free(Size_of_bin);
             free(Current_bin_capacity);
          }

          np[0] = 0;
          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations);

	  return HMPI_OK;
       }

       /*
        * Processors can be reordered.
        * No known results.
        * A naive implementation is provided here.
        */
       if ((w != NULL) 
           && (mlimits !=  NULL)
           && (ordering == 1)
           && (pn == 1)
           && (processor_ordering == 1
          )
       )
       {
          double *rearranged_speeds;
	  int *rearrangedp;
          int *rearranged_mlimits;
	  double temp;
	  int temp_number, temp_mlimit;
          int *allocations;
          int ind;

          rearranged_mlimits = (int*)malloc(
                                     sizeof(int)
                                     *
                                     p
          );

          if (rearranged_mlimits == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearranged_speeds = (double*)malloc(
                                       sizeof(double)
		   		       *
				       p
          );

          if (rearranged_speeds == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
		   	      *
			      p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_speeds[i] = speeds[i];
              rearranged_mlimits[i] = mlimits[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_speeds[j-1] < rearranged_speeds[j])
                  {
                     temp = rearranged_speeds[j-1];
		     rearranged_speeds[j-1] = rearranged_speeds[j];
		     rearranged_speeds[j] = temp;

		     temp_number = rearrangedp[j-1];
		     rearrangedp[j-1] = rearrangedp[j];
		     rearrangedp[j] = temp_number;

                     temp_mlimit = rearranged_mlimits[j-1];
                     rearranged_mlimits[j-1] = rearranged_mlimits[j];
                     rearranged_mlimits[j] = temp_mlimit;
                  }
              }
          }

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = __HMPI_Apply_mlimits_to_ordered_sum_of_weights (
                   p,
                   n,
                   rearranged_speeds,
                   rearranged_mlimits,
                   w,
                   -1,
                   NULL,
                   NULL,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          /*
           * Metric is not calculated before.
           * Do it now.
           */
          if (metric != NULL)
          {
             int i, ind, j, sumw;
             int *Size_of_bin, *Current_bin_capacity;

             Size_of_bin  = (int*)malloc(
                                  sizeof(int)
                                  *
                                  p
             );

             if (Size_of_bin == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rc = __HMPI_Size_of_bins(
                      p,
                      n,
                      rearranged_speeds,
                      w,
                      Size_of_bin,
                      &sumw
             );

             if (rc != HMPI_OK)
             {
                return rc;
             }

             Current_bin_capacity = (int*)malloc(
                                          sizeof(int)
                                          *
                                          p
             );

             if (Current_bin_capacity == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < p; i++)
             {
                Current_bin_capacity[i] = 0;
             }

             for (i = 0, ind = 0; i < p; i++)
             {
                 for (j = 0; j < allocations[i]; j++)
                 {
                     Current_bin_capacity[i] += w[ind++];
                 }
             }

             /*
              * The ideal sum of weights is given by
              * elements of array Size_of_bin and the
              * actual sum of weights is calculated for array elements
              * of Current_bin_capacity.
              */
             switch (type_of_metric)
             {
                case USER_SPECIFIED:
                  {
                     *metric = (*umf)(
                                p,
                                rearranged_speeds,
                                Current_bin_capacity,
                                Size_of_bin
                     );
                  }
                  break;
                case SYSTEM_DEFINED:
                  {
                     *metric = __HMPI_System_defined_metric(
                               p,
                               rearranged_speeds,
                               Current_bin_capacity,
                               Size_of_bin
                     );
                  }
                  break;
                default:
                  {
                     return HMPI_ERR_METRIC;
                  }
                  break;
             }

             free(Size_of_bin);
             free(Current_bin_capacity);
          }

	  for (ind = 0, i = 0; i < p; i++)
          {
             np[ind++] = rearrangedp[i];
             np[ind++] = allocations[i];
          }

	  free(rearranged_speeds);
	  free(rearranged_mlimits);
	  free(rearrangedp);
	  free(allocations);

	  return HMPI_OK;
       }

       /*
        * Speeds are functions of problem size
        * set with weighted elements and 
        * no bounds on the number of elements
        * Processors cannot be reordered.
        * No known results.
        */
       if ((w != NULL)
           && (mlimits ==  NULL) 
           && (ordering == 1)
           && (pn > 1)
           && (processor_ordering == 0
          )
       )
       {
          int *allocations;
          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }
 
          /*
           * Naive implementation
           */
          rc = __HMPI_Sum_of_weights_for_ordered_set_speed_functions(
                   p,
                   pn,
                   speeds,
                   psizes,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          np[0] = 0;

          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations); 

          return HMPI_OK;
       }

       /*
        * Processors can be reordered.
        * No known results.
        */
       if ((w != NULL)
           && (mlimits ==  NULL) 
           && (ordering == 1)
           && (pn > 1)
           && (processor_ordering == 1
          )
       )
       {
          /*
           * Naive implementation
           */
          return __HMPI_Sum_of_weights_for_ordered_set_speed_functions_processor_reordering(
                   p,
                   pn,
                   speeds,
                   psizes,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   np
          );
       }

       /*
        * Speeds are functions of problem size, 
        * set with weighted elements and there
        * is a limit on the number of elements that can 
        * be stored by each processor
        * Processors cannot be reordered.
        * No known results.
        */
       if ((w != NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 1)
           && (pn > 1)
           && (processor_ordering == 0
          )
       )
       {
          int *allocations;

          allocations = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (allocations == NULL)
          {
             return MPC_ERR_NOMEM;
          }
 
          /*
           * Naive implementation
           */
          rc = __HMPI_Sum_of_weights_for_ordered_set_speed_functions_with_mlimits(
                   p,
                   pn,
                   speeds,
                   psizes,
                   mlimits,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   allocations
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          /*
           * Metric is not calculated before.
           * Do it now.
           */
          if (metric != NULL)
          {
             int i, ind, j, sumw;
             int *Size_of_bin, *Current_bin_capacity;

             Size_of_bin  = (int*)malloc(
                                  sizeof(int)
                                  *
                                  p
             );

             if (Size_of_bin == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             rc = __HMPI_Size_of_bins(
                      p,
                      n,
                      speeds,
                      w,
                      Size_of_bin,
                      &sumw
             );

             if (rc != HMPI_OK)
             {
                return rc;
             }

             Current_bin_capacity = (int*)malloc(
                                          sizeof(int)
                                          *
                                          p
             );

             if (Current_bin_capacity == NULL)
             {
                return MPC_ERR_NOMEM;
             }

             for (i = 0; i < p; i++)
             {
                Current_bin_capacity[i] = 0;
             }

             for (i = 0, ind = 0; i < p; i++)
             {
                 for (j = 0; j < allocations[i]; j++)
                 {
                     Current_bin_capacity[i] += w[ind++];
                 }
             }

             /*
              * The ideal sum of weights is given by
              * elements of array Size_of_bin and the
              * actual sum of weights is calculated for array elements
              * of Current_bin_capacity.
              */
             switch (type_of_metric)
             {
                case USER_SPECIFIED:
                  {
                     *metric = (*umf)(
                                p,
                                speeds,
                                Current_bin_capacity,
                                Size_of_bin
                     );
                  }
                  break;
                case SYSTEM_DEFINED:
                  {
                     *metric = __HMPI_System_defined_metric(
                               p,
                               speeds,
                               Current_bin_capacity,
                               Size_of_bin
                     );
                  }
                  break;
                default:
                  {
                     return HMPI_ERR_METRIC;
                  }
                  break;
             }

             free(Size_of_bin);
             free(Current_bin_capacity);
          }

          np[0] = 0;

          for (i = 1; i <= p; i++)
          {
             np[i] = np[i-1] + allocations[i-1];
          }

          free(allocations); 

          return HMPI_OK;
       }

       /*
        * Processors can be reordered.
        * No known results.
        */
       if ((w != NULL) 
           && (mlimits !=  NULL) 
           && (ordering == 1)
           && (pn > 1)
           && (processor_ordering == 1
          )
       )
       {
          /*
           * Naive implementation
           */
          return __HMPI_Sum_of_weights_for_ordered_set_speed_functions_processor_reordering_with_mlimits(
                   p,
                   pn,
                   speeds,
                   psizes,
                   mlimits,
                   n,
                   w,
                   type_of_metric,
                   umf,
                   metric,
                   np
          );
       }

       printf("Parameters provided are invalid\n");
       return HMPI_ERR_PARTITION_SET;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_set_processor(
           int pos,
           int n,
           int p,
           int processor_ordering,
           const int *np
   )
   {
       int i, j;

       if (processor_ordering == 1)
       {
          int *cumnp = (int*)malloc(
                             sizeof(int)
                              *
                              (p+1)
          ); 

          if (cumnp == NULL)
          {
             printf("Can't allocate cumnp in Function HMPI_Get_set_processor\n");
             return MPC_ERR_NOMEM;
          }

          cumnp[0] = 0;

          for (i = 1; i <= p; i++)
          {
             cumnp[i] = np[2*i - 1] + cumnp[i - 1];
          }  

          for (i = 0; i < p; i++)
          {
             if ((pos >= cumnp[i])
                 && (pos < cumnp[i+1]
                )
             )
             {
                free(cumnp);
                return np[i];
             }
          }

          free(cumnp);
          return -1;
       }

       {
          int *cumnp = (int*)malloc(
                             sizeof(int)
                             *
                             (p+1)
          );

          if (cumnp == NULL)
          {
             printf("Can't allocate cumnp in Function HMPI_Get_set_processor\n");
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i <= p; i++)
          {
              cumnp[i] = 0;
              for (j = 0; j < i; j++)
              {
                  cumnp[i] += np[j];
              }
          }

          for (i = 0; i < p; i++)
          {
             if ((pos >= cumnp[i])
                 && (pos < cumnp[i+1]
                )
             )
             {
                free(cumnp);
                return i;
             }
          }

          free(cumnp);
       }

       return -1;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_unordered_set (
           int p,
           int pn,
           const double *speeds,
           const int *psizes,
           const int *mlimits,
           int n,
           const int *w,
           int type_of_metric,
           User_defined_metric umf,
           double *metric,
           int *np
   )
   {
       return HMPI_Partition_set(
              p,
              pn,
              speeds,
              psizes,
              mlimits,
              n,
              w,
              0,
              -1,
              type_of_metric,
              umf,
              metric,
              np
       );
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_ordered_set (
           int p,
           int pn,
           const double *speeds,
           const int *psizes,
           const int *mlimits,
           int n,
           const int *w,
           int processor_reordering,
           int type_of_metric,
           User_defined_metric umf,
           double *metric,
           int *np
   )
   {
       return HMPI_Partition_set(
              p,
              pn,
              speeds,
              psizes,
              mlimits,
              n,
              w,
              1,
              processor_reordering,
              type_of_metric,
              umf,
              metric,
              np
       );
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Get_my_partition
   (
       int i,
       int p,
       const int *speeds,
       int n
   )
   {
        int ind;
        int rc, myd;
        double *perf;
        int *d = (int*)malloc(
                       sizeof(int)
                       *
                       p
        );

        if (d == NULL)
        {
           printf("Can't allocate cumnp in Function HMPI_Get_my_partition\n");
           return MPC_ERR_NOMEM;
        }

        perf = (double*)malloc(
                       sizeof(double)
                       *
                       p
        );

        if (perf == NULL)
        {
           printf("Can't allocate perf in Function HMPI_Get_my_partition\n");
           return MPC_ERR_NOMEM;
        }

        for (ind = 0; ind < p; ind++)
        {
           perf[ind] = speeds[ind];
        }

        rc = HMPI_Partition_set(
                 p,
                 1,
                 perf,
                 NULL,
                 NULL,
                 n,
                 NULL,
                 0,
                 0,
                 -1,
                 NULL,
                 NULL,
                 d
        );

        if (rc != HMPI_OK)
        {
           printf("Problems partitioning\n");
           return -1;
        }

        myd = d[i];

        free(d);
        free(perf);
 
        return myd;
   }

   /*-----------------------------------------------------*/
