
   /************************************************/
   /* Implementation of Partitioning interfaces    */
   /* for matrices                                 */
   /*                                              */
   /* Revision history                             */
   /* 23-05-2003  --      Initial version          */
   /************************************************/

   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>

   #include <math.h>
   #include <hmpi.h>

   #ifndef min
   #define min(x, y) ((x < y) ? x : y)
   #endif

   /*-----------------------------------------------------*/

   int HMPI_Create_rectangles_1d_recursive(
       int p,
       const int *row_np,
       const int *column_np,
       int *w,
       int *h,
       int *trow,
       int *tcolumn
   )
   {
   }

   /*-----------------------------------------------------*/

   int HMPI_Create_rectangles_1d(
       int p,
       int q,
       int m, 
       int n,
       const int *row_np,
       const int *column_np,
       int *w,
       int *h,
       int *trow,
       int *tcolumn,
       int *ci
   )
   {
       int i, j, k, x, y;

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               w[HMPI_RECT_INDEX(i, j, i, j, p, q)] = column_np[i*q+j];
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               h[HMPI_RECT_INDEX(i, j, i, j, p, q)] = row_np[i*q+j];
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               trow[i*q+j] = 0;
               for (k = 0; k < i; k++)
               {
                   trow[i*q+j] += h[HMPI_RECT_INDEX(k, j, k, j, p, q)];
               }
           }
       }

       for (j = 0; j < q; j++)
       {
           for (i = 0; i < p; i++)
           {
               tcolumn[i*q+j] = 0;
               for (k = 0; k < j; k++)
               {
                   tcolumn[i*q+j] += w[HMPI_RECT_INDEX(i, k, i, k, p, q)];
               }
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (x = 0; x < p; x++)
               {
                   for (y = 0; y < q; y++)
                   {
                       int width = HMPI_Common_height(
                                   tcolumn[i*q+j],
				   (tcolumn[i*q+j] 
				    + 
				    w[HMPI_RECT_INDEX(i, j, i, j, p, q)]
                                   ),
				   tcolumn[x*q+y],
				   (tcolumn[x*q+y] 
				    + 
				    w[HMPI_RECT_INDEX(x, y, x, y, p, q)]
                                   )
                       );

		       w[HMPI_RECT_INDEX(i, j, x, y, p, q)] = width;
                   }
               }
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (x = 0; x < p; x++)
               {
                   for (y = 0; y < q; y++)
                   {
                       int height = HMPI_Common_height(
                                    trow[i*q+j],
				    (trow[i*q+j] 
				     + 
				     h[HMPI_RECT_INDEX(i, j, i, j, p, q)]
                                    ),
				    trow[x*q+y],
				    (trow[x*q+y] 
				     + 
				     h[HMPI_RECT_INDEX(x, y, x, y, p, q)]
                                    )
                       );

		       h[HMPI_RECT_INDEX(i, j, x, y, p, q)] = height;
                   }
               }
           }
       }

       if (ci == NULL)
       {
          return HMPI_OK;
       }

       for (x = 0; x < p; x++)
       {
           for (y = 0; y < q; y++)
           {
               int hi = h[HMPI_RECT_INDEX(x, y, x, y, p, q)];
               int wi = w[HMPI_RECT_INDEX(x, y, x, y, p, q)];
	       int tr = trow[x*q+y];
               int tc = tcolumn[x*q+y];

	       for (i = 0; i < hi; i++)
	       {
		   for (j = 0; j < wi; j++) 
		   {
                      ci[(tr+i)*n+tc+j] = x*q+y;
                   }
               }
           }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Create_rectangles_2d
   (
       int p,
       int q,
       int m, 
       int n,
       const int *row_np,
       const int *column_np,
       int *w,
       int *h,
       int *trow,
       int *tcolumn,
       int *ci,
       int *cj
   )
   {
       int i, j, k, x, y;

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               w[HMPI_RECT_INDEX(i, j, i, j, p, q)] = column_np[i*q+j];
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               h[HMPI_RECT_INDEX(i, j, i, j, p, q)] = row_np[i*q+j];
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               trow[i*q+j] = 0;
               for (k = 0; k < i; k++)
               {
                   trow[i*q+j] += h[HMPI_RECT_INDEX(k, j, k, j, p, q)];
               }
           }
       }

       for (j = 0; j < q; j++)
       {
           for (i = 0; i < p; i++)
           {
               tcolumn[i*q+j] = 0;
               for (k = 0; k < j; k++)
               {
                   tcolumn[i*q+j] += w[HMPI_RECT_INDEX(i, k, i, k, p, q)];
               }
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (x = 0; x < p; x++)
               {
                   for (y = 0; y < q; y++)
                   {
                       int width = HMPI_Common_height(
                                   tcolumn[i*q+j],
				   (tcolumn[i*q+j] 
				    + 
				    w[HMPI_RECT_INDEX(i, j, i, j, p, q)]
                                   ),
				   tcolumn[x*q+y],
				   (tcolumn[x*q+y] 
				    + 
				    w[HMPI_RECT_INDEX(x, y, x, y, p, q)]
                                   )
                       );

		       w[HMPI_RECT_INDEX(i, j, x, y, p, q)] = width;
                   }
               }
           }
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (x = 0; x < p; x++)
               {
                   for (y = 0; y < q; y++)
                   {
                       int height = HMPI_Common_height(
                                    trow[i*q+j],
				    (trow[i*q+j] 
				     + 
				     h[HMPI_RECT_INDEX(i, j, i, j, p, q)]
                                    ),
				    trow[x*q+y],
				    (trow[x*q+y] 
				     + 
				     h[HMPI_RECT_INDEX(x, y, x, y, p, q)]
                                    )
                       );

		       h[HMPI_RECT_INDEX(i, j, x, y, p, q)] = height;
                   }
               }
           }
       }

       if ((ci == NULL)
           && (cj == NULL
          )
       )
       {
          return HMPI_OK;
       }

       for (x = 0; x < p; x++)
       {
           for (y = 0; y < q; y++)
           {
               int hi = h[HMPI_RECT_INDEX(x, y, x, y, p, q)];
               int wi = w[HMPI_RECT_INDEX(x, y, x, y, p, q)];
	       int tr = trow[x*q+y];
               int tc = tcolumn[x*q+y];

	       for (i = 0; i < hi; i++)
	       {
		   for (j = 0; j < wi; j++) 
		   {
                      ci[(tr+i)*n+tc+j] = x;
                      cj[(tr+i)*n+tc+j] = y;
                   }
               }
           }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int _HMPI_Factor(
       int divisor,
       int* quotient,
       int **factors,
       int *numf,
       int *max_n_of_factors
   )
   {
      int i;

      while (((*quotient)%divisor) == 0)
      {
         (*factors)[(*numf)++] = divisor;

         /*
          * Copy and enlarge the array
          */
         if ((*numf) >= (*max_n_of_factors))
         {
            int tempmax;
            int *tempf = (int*)malloc(
                               sizeof(int)
                               *
                               (*max_n_of_factors)
            );

            if (tempf == NULL)
            {
               return MPC_ERR_NOMEM;
            }

            for (i = 0; i < (*max_n_of_factors); i++)
            {
               tempf[i] = (*factors)[i];
            }

            free(factors[0]);

            tempmax = (*max_n_of_factors);
            (*max_n_of_factors) *= 2;

            factors[0] = (int*)malloc(
                               sizeof(int)
                               *
                               (*max_n_of_factors)
            );

            if (factors[0] == NULL)
            {
               return MPC_ERR_NOMEM;
            }

            for (i = 0; i < tempmax; i++)
            {
               (*factors)[i] = tempf[i];
            }

            free(tempf);
         }

         (*quotient) /= divisor;
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int _HMPI_Get_factors(
       int n, 
       int *numf, 
       int **result
   )
   {
      int rc;
      int quotient = n;
      int divisor, maxDivisor;
      int max_n_of_factors = 2;

      *numf = 0;

      result[0] = (int*)malloc(
                        sizeof(int)
                        *
                        max_n_of_factors
      );

      if (result[0] == NULL)
      {
         return MPC_ERR_NOMEM;
      }
   
      //
      // Try special cases of 2 and 3
      rc = _HMPI_Factor(
           2, 
           &quotient, 
           result, 
           numf, 
           &max_n_of_factors
      );

      if (rc != HMPI_OK)
      {
         return rc;
      }

      rc = _HMPI_Factor(
           3, 
           &quotient, 
           result, 
           numf, 
           &max_n_of_factors
      );

      if (rc != HMPI_OK)
      {
         return rc;
      }

      //
      // Try pairs of the form 6m-1 and 6m+1
      // (i.e. 5, 7, 11, 13, 17, 19, . .)
      maxDivisor = sqrt(quotient);
      for (divisor = 5; divisor <= maxDivisor; divisor+=6)
      {
         rc = _HMPI_Factor(
              divisor, 
              &quotient, 
              result, 
              numf, 
              &max_n_of_factors
         );

         if (rc != HMPI_OK)
         {
            return rc;
         }

         rc = _HMPI_Factor(
              divisor+2, 
              &quotient, 
              result, 
              numf, 
              &max_n_of_factors
         );

         if (rc != HMPI_OK)
         {
            return rc;
         }
      }

      // store final factor
      if (quotient > 1)
      {
         (*result)[(*numf)++] = quotient;
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Homogeneous_matrix_1d_no_mlimits
   (
       int p,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci
   )
   {
       int *row_np, *column_np;

       /*
        * This could be a open problem.
        * The first question is can we use a 
        * 1D array as a logical 2D array of 1*p
        * processes or p*1 processes.
        * If not, what grid to use. Does it matter.
        * We find all factors of p and form a 
        * grid. May not be a optimal one.
        */
       int i, j;
       int f1 = 1;
       int f2 = 1;

       f1 = sqrt(p);
       f2 = sqrt(p);

       if ((f1*f2) != p)
       {
          int numf, rc;
          int **result = (int**)malloc(
                                sizeof(int*)
          );

          if (result == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rc = _HMPI_Get_factors(
                        p, 
                        &numf, 
                        result
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          for (f1 = 1, j = 0; j < numf; j+=2)
          {
             f1 *= (*result)[j];
          }

          for (f2 = 1, j = 1; j < numf; j+=2)
          {
             f2 *= (*result)[j];
          }

          free(result[0]);
          free(result);
       }

       if (HMPI_Debug_flag)
       {
          printf(
            "HMPI===> HMPI_Homogeneous_matrix_1d_no_mlimits: 1d array %d converted to 2d grid"
            " (%d,%d)\n",
            p,
            f1,
            f2
          );
       }

       row_np = (int*)malloc(
                      sizeof(int)
		      *
		      (f1*f2)
       );

       if (row_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np = (int*)malloc(
                         sizeof(int)
		         *
		         (f1*f2)
       );

       if (column_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       if ((m < f1)
           && (n < f2
          )
       )
       {
          for (i = 0; i < m; i++)
          {
             for (j = 0; j < f2; j++)
             {
                row_np[i*f2+ j] = 1;
             }
          }

          for (i = m; i < f1; i++)
          {
             for (j = 0; j < f2; j++)
             {
                row_np[i*f2+ j] = 0;
             }
          }

          for (i = 0; i < n; i++)
          {
             for (j = 0; j < f1; j++)
             {
                column_np[i*f1 + j] = 1;
             }
          }

          for (i = n; i < f2; i++)
          {
             for (j = 0; j < f1; j++)
             {
                column_np[i*f1 + j] = 0;
             }
          }
       }

       if ((m < f1)
           && (n >= f2
          )
       )
       {
          for (i = 0; i < m; i++)
          {
             for (j = 0; j < f2; j++)
             {
                row_np[i*f2+ j] = 1;
             }
          }

          for (i = m; i < f1; i++)
          {
             for (j = 0; j < f2; j++)
             {
                row_np[i*f2+ j] = 0;
             }
          }

          for (i = 0; i < f2; i++)
          {
             for (j = 0; j < m; j++)
             {
                column_np[i + j*f2] = n/f2;
             }

             for (j = m; j < f1; j++)
             {
                column_np[i + j*f2] = 0;
             }
          }

          for (i = 0; i < m; i++)
          {
             column_np[i*f2] += n%f2;
          }
       }
       
       if ((m >= f1)
           && (n >= f2
          )
       )
       {
          for (i = 0; i < f1; i++)
          {
             for (j = 0; j < f2; j++)
             {
                row_np[i*f2+ j] = m/f1;
             }
          }

          for (i = 0; i < f2; i++)
          {
             row_np[i] += m%f1;
          }

          for (i = 0; i < f2; i++)
          {
             for (j = 0; j < f1; j++)
             {
                column_np[i + j*f2] = n/f2;
             }
          }

          for (i = 0; i < f1; i++)
          {
             column_np[i*f2] += n%f2;
          }
       }

       if ((m >= f1)
           && (n < f2
          )
       )
       {
          for (i = 0; i < n; i++)
          {
             for (j = 0; j < f1; j++)
             {
                column_np[i*f1 + j] = 1;
             }
          }

          for (i = n; i < f2; i++)
          {
             for (j = 0; j < f1; j++)
             {
                column_np[i*f1 + j] = 0;
             }
          }

          for (i = 0; i < f1; i++)
          {
             for (j = 0; j < n; j++)
             {
                row_np[i*f2+ j] = m/f1;
             }

             for (j = n; j < f2; j++)
             {
                row_np[i*f2+ j] = 0;
             }
          }

          for (i = 0; i < n; i++)
          {
             row_np[i] += m%f1;
          }
       }

       HMPI_Create_rectangles_1d(
           f1,
	   f2,
           m,
           n,
           row_np,
	   column_np,
	   w,
	   h,
           trow,
           tcol,
           ci
       );

       free(row_np);
       free(column_np);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Homogeneous_matrix_2d_no_mlimits
   (
       int p,
       int q,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       int i, j, rc;
       int *row_np, *column_np;

       row_np = (int*)malloc(
                      sizeof(int)
		      *
		      (p*q)
       );

       if (row_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np = (int*)malloc(
                         sizeof(int)
		         *
		         (p*q)
       );

       if (column_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       if ((m < p)
           && (n < q
          )
       )
       {
          for (i = 0; i < m; i++)
          {
             for (j = 0; j < q; j++)
             {
                row_np[i*q+ j] = 1;
             }
          }

          for (i = m; i < p; i++)
          {
             for (j = 0; j < q; j++)
             {
                row_np[i*q+ j] = 0;
             }
          }

          for (i = 0; i < n; i++)
          {
             for (j = 0; j < p; j++)
             {
                column_np[i*p + j] = 1;
             }
          }

          for (i = n; i < q; i++)
          {
             for (j = 0; j < p; j++)
             {
                column_np[i*p + j] = 0;
             }
          }
       }

       if ((m < p)
           && (n >= q
          )
       )
       {
          for (i = 0; i < m; i++)
          {
             for (j = 0; j < q; j++)
             {
                row_np[i*q+ j] = 1;
             }
          }

          for (i = m; i < p; i++)
          {
             for (j = 0; j < q; j++)
             {
                row_np[i*q+ j] = 0;
             }
          }

          for (i = 0; i < q; i++)
          {
             for (j = 0; j < m; j++)
             {
                column_np[i + j*q] = n/q;
             }

             for (j = m; j < p; j++)
             {
                column_np[i + j*q] = 0;
             }
          }

          for (i = 0; i < m; i++)
          {
             column_np[i*q] += n%q;
          }
       }
       
       if ((m >= p)
           && (n >= q
          )
       )
       {
          for (i = 0; i < p; i++)
          {
             for (j = 0; j < q; j++)
             {
                row_np[i*q+ j] = m/p;
             }
          }

          for (i = 0; i < q; i++)
          {
             row_np[i] += m%p;
          }

          for (i = 0; i < q; i++)
          {
             for (j = 0; j < p; j++)
             {
                column_np[i + j*q] = n/q;
             }
          }

          for (i = 0; i < p; i++)
          {
             column_np[i*q] += n%q;
          }
       }

       if ((m >= p)
           && (n < q
          )
       )
       {
          for (i = 0; i < n; i++)
          {
             for (j = 0; j < p; j++)
             {
                column_np[i*p + j] = 1;
             }
          }

          for (i = n; i < q; i++)
          {
             for (j = 0; j < p; j++)
             {
                column_np[i*p + j] = 0;
             }
          }

          for (i = 0; i < p; i++)
          {
             for (j = 0; j < n; j++)
             {
                row_np[i*q+ j] = m/p;
             }

             for (j = n; j < q; j++)
             {
                row_np[i*q+ j] = 0;
             }
          }

          for (i = 0; i < n; i++)
          {
             row_np[i] += m%p;
          }
       }

       HMPI_Create_rectangles_2d(
           p,
	   q,
           m,
           n,
           row_np,
	   column_np,
	   w,
	   h,
           trow,
           tcol,
           ci,
           cj
       );

       free(row_np);
       free(column_np);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Matrix_cq_recursive_bisection
   (
       int p,
       const double *speeds,
       const int *mlimits,
       int dimension,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       int i, j, rc, dimension_size;
       int total_limits = 0;
       double tspeeds = 0.0;
       double first_half_speeds = 0.0;
       int tmlimits = 0;
       int first_half_mlimits = 0.0;
       int second_half_mlimits = 0.0;
       double first_half_allocation;
       double second_half_allocation;
       int H = 0;
       int V = 1;

       if (mlimits != NULL)
       {
          for (i = 0; i < p; i++)
          {
              total_limits += mlimits[i];
          }

          if (total_limits < (m*n))
          {
             printf(
	       "==>HMPI: Problems applying the limits during "
	       " matrix partitioning using RECURSIVE "
	       "one-dimensional distribution\n"
             );

             return HMPI_ERR_MLIMITS;
          }
       }

       if (p == 1)
       {
          w[0] = n;
	  h[0] = m;
	  trow[0] = 0;
	  tcol[0] = 0;

	  return HMPI_OK;
       }

       for (i = 0; i < p; i++)
       {
           if (i < (p/2))
           {
              first_half_speeds += speeds[i];
	      first_half_mlimits += mlimits[i];
           }

           tspeeds += speeds[i];
	   tmlimits += mlimits[i];
       }

       if (dimension == H)
       {
          dimension_size = m;
	  first_half_mlimits = (first_half_mlimits)/n;
	  second_half_mlimits = (tmlimits/n) - first_half_mlimits;
       }

       if (dimension == V)
       {
          dimension_size = n;
	  first_half_mlimits = (first_half_mlimits)/m;
	  second_half_mlimits = (tmlimits/m) - first_half_mlimits;
       }

       {
          int allocations[2];
	  double speedsr[] = {
              first_half_speeds,
	      (tspeeds - first_half_speeds)
          };
	  int mlimitsr[] = {
              first_half_mlimits,
	      second_half_mlimits   
          };

	  rc = HMPI_Partition_set(
                   2,
                   1,
                   speedsr,
                   NULL,
		   mlimitsr,
		   dimension_size,
		   NULL,
		   1,
		   0,
		   -1,
		   NULL,
		   NULL,
		   allocations
          );

          if (rc == HMPI_ERR_PARTITION_SET)
          {
             printf(
	       "==>HMPI: Problems partitioning the matrix "
	       " using RECURSIVE one-dimensional distribution\n"
             );
             return HMPI_ERR_PARTITION_MATRIX;
          }

          if (rc == HMPI_ERR_MLIMITS)
          {
             printf(
	       "==>HMPI: Problems applying the limits during "
	       " matrix partitioning using RECURSIVE "
	       "one-dimensional distribution\n"
             );
             return HMPI_ERR_MLIMITS;
          }

          first_half_allocation = allocations[0];
	  second_half_allocation = allocations[1];
       }

       if (p == 2)
       {
          if (dimension == H)
          {
             w[0] = n;
	     h[0] = first_half_allocation;
	     trow[0] = 0;
	     tcol[0] = 0;

             w[1] = n;
	     h[1] = second_half_allocation;
	     trow[0] = first_half_allocation;
	     tcol[0] = 0;
          }

          if (dimension == V)
          {
             w[0] = first_half_allocation;
	     h[0] = m;
	     trow[0] = 0;
	     tcol[0] = 0;

             w[1] = second_half_allocation;
	     h[1] = m;
	     trow[0] = 0;
	     tcol[0] = first_half_allocation;

          }

	  return HMPI_OK;
       }

       {
          int dimensionr;
	  int partitionm;
	  int partitionn;

          if (dimension == H)
          {
             dimensionr = V;
          }

          if (dimension == V)
          {
             dimensionr = H;
          }

          if (dimension == H)
          {
	     partitionm = first_half_allocation;
	     partitionn = n;
          }

          if (dimension == V)
          {
	     partitionm = m;
	     partitionn = first_half_allocation;
          }

          rc = HMPI_Matrix_cq_recursive_bisection(
                   (p/2),
		   speeds,
		   mlimits,
		   dimension,
		   partitionm,
		   partitionn,
		   w,
		   h,
		   trow,
		   tcol
          );
          
	  if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (i = 0; i < (p/2); i++)
          {
	      trow[i] = 0;
              tcol[i] = 0;

	      for (j = 0; j < i; j++)
              {
                  trow[i] += h[j];
                  tcol[i] += w[j];
              }
          }

          if (dimension == H)
          {
	     partitionm = second_half_allocation;
	     partitionn = n;
          }

          if (dimension == V)
          {
	     partitionm = m;
	     partitionn = second_half_allocation;
          }

          rc = HMPI_Matrix_cq_recursive_bisection(
                   p - (p/2),
		   speeds + (p/2),
		   mlimits + (p/2),
		   dimension,
		   partitionm,
		   partitionn,
		   w + (p/2),
		   h + (p/2),
		   trow + (p/2),
		   tcol + (p/2)
          );

	  if (rc != HMPI_OK)
          {
             return rc;
          }

	  for (i = 0; i < (p - (p/2)); i++)
          {
              if (dimension == H)
              {
                 trow[(p/2) + i] = first_half_allocation;
                 tcol[(p/2) + i] = 0;
              }

              if (dimension == V)
              {
                 trow[(p/2) + i] = 0;
                 tcol[(p/2) + i] = first_half_allocation;
              }

	      for (j = 0; j < i; j++)
              {
                  trow[(p/2) + i] += h[j];
                  tcol[(p/2) + i] += w[j];
              }
          }
       }

       return HMPI_OK;
   }
       
   /*-----------------------------------------------------*/

   int HMPI_Matrix_one_dimensional_recursive
   (
       int p,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   { 
       int rc;
       int H = 0;
       int V = 1;

       /*
	* Use the Orthogonal recursive bisection scheme of
	* Crandall and Quinn.
	*/
       int *wp, *hp;

       wp = (int*)malloc(
                  sizeof(int)
		  *
		  p
       );

       hp = (int*)malloc(
                  sizeof(int)
		  *
		  p
       );

       rc = HMPI_Matrix_cq_recursive_bisection(
                p,
	        speeds,
	        mlimits,
	        V,
		m,
		n,
		wp,
		hp,
		trow,
		tcol
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       HMPI_Create_rectangles_1d_recursive(
           p,
           wp,
	   hp,
	   w,
	   h,
	   trow,
	   tcol
       );

       free(wp);
       free(hp);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Common_height
   (
       int top_row_1,
       int bottom_row_1,
       int top_row_2,
       int bottom_row_2
   )
   {
       /*
        * One area contains the other
        */
       if ((top_row_1 >= top_row_2)
           && (bottom_row_1 <= bottom_row_2)
       )
       {
          return (bottom_row_1 - top_row_1);
       }

       if ((top_row_1 <= top_row_2)
           && (bottom_row_1 >= bottom_row_2)
       )
       {
          return (bottom_row_2 - top_row_2);
       }

       /*
        * One area is followed or preceded by another
        * with an overlap
        */
       if ((top_row_1 <= top_row_2)
           && (bottom_row_1 >= top_row_2)
           && (bottom_row_1 <= bottom_row_2)
       )
       {
          return (bottom_row_1 - top_row_2);
       }

       if ((top_row_1 >= top_row_2)
           && (top_row_1 <= bottom_row_2)
           && (bottom_row_1 >= bottom_row_2)
       )
       {
          return (bottom_row_2 - top_row_1);
       }

       /*
        * There is no overlap
        */
       if ((bottom_row_1 < top_row_2)
           || (top_row_1 > bottom_row_2)
       )
       {
          return 0;
       }

       if ((top_row_1 < top_row_2)
           && (bottom_row_1 < bottom_row_2)
       )
       {
          return 0;
       }

       if ((top_row_1 > top_row_2)
           && (bottom_row_1 > bottom_row_2)
       )
       {
          return 0;
       }

       return 0;
   }

   /*-----------------------------------------------------*/

   void HMPI_Create_rectangles_1d_column_based
   (
       int p,
       int number_of_columns,
       int *rectangles_in_each_column,
       const double *areas,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcolumn
   )
   {
       int i, j, ix = 0;

       for (i = 0; i < number_of_columns; i++)
       {
          int width = 0;

          for (j = 0; j < rectangles_in_each_column[i]; j++)
          {
             width += areas[p - j - 1 - ix];
          }

          for (j = 0; j < rectangles_in_each_column[i]; j++)
          {
             w[p - j - 1 - ix] = width*n;
             h[p - j - 1 - ix] = (areas[p - j - 1 - ix]/width)*m;
          }

          ix += rectangles_in_each_column[i];
       }

       /*
        * Adjust the parameters w & h
        * TBD
        */
       return;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_column_based_speeds_single_numbers_with_mlimits
   (
       int p,
       int q,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       /*
        * TBD
        */
       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_row_based_speeds_single_numbers_no_mlimits
   (
       int p,
       int q,
       const double *speeds,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       int i, j, rc;
       double *row_speed_sums;
       int *row_np, *row_np_sub;
       int *column_np;

       row_speed_sums = (double*)malloc(
                        sizeof(double)
	                *
			p
       );

       if (row_speed_sums == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       row_np = (int*)malloc(
                      sizeof(int)
		      *
		      (p*q)
       );

       if (row_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np = (int*)malloc(
                         sizeof(int)
		         *
		         (p*q)
       );

       if (column_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       row_np_sub = (int*)malloc(
                          sizeof(int)
	                  *
		          p
       );

       if (row_np_sub == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           row_speed_sums[i] = 0.0;
           for (j = 0; j < q; j++)
           {
               row_speed_sums[i] += speeds[i*q+j];
           }
       }

       /*
	* Partition of the row dimension among
	* p processors
	*/
       rc = HMPI_Partition_set(
                p,
                1,
	        row_speed_sums,
                NULL,
		NULL,
		m,
		NULL,
		0,
		0,
		-1,
		NULL,
		NULL,
		row_np_sub
       );

       if (rc == HMPI_ERR_PARTITION_SET)
       {
          return HMPI_ERR_PARTITION_MATRIX;
       }

       if (rc == HMPI_ERR_MLIMITS)
       {
          return HMPI_ERR_MLIMITS;
       }

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               row_np[i*q+j] = row_np_sub[i];
           }
       }

       free(row_speed_sums);
       free(row_np_sub);

       for (i = 0; i < p; i++)
       {
           double *column_speed_sums;
	   int *column_np_sub;

	   column_speed_sums = (double*)malloc(
			                sizeof(double)
				        *
				        q
           );

           if (column_speed_sums == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   column_np_sub = (int*)malloc(
			         sizeof(int)
			         *
			         q
           );

           if (column_np_sub == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   for (j = 0; j < q; j++)
           {
               column_speed_sums[j] = speeds[i*q + j];
           }

           /*
	    * Partition of the column dimension among
	    * q processors
	    */
           rc = HMPI_Partition_set(
                    q,
                    1,
	            column_speed_sums,
	    	    NULL,
                    NULL,
		    n,
		    NULL,
		    0,
		    0,
		    -1,
		    NULL,
		    NULL,
		    column_np_sub
           );

           if (rc == HMPI_ERR_PARTITION_SET)
           {
              return HMPI_ERR_PARTITION_MATRIX;
           }

           if (rc == HMPI_ERR_MLIMITS)
           {
              return HMPI_ERR_MLIMITS;
           }

           if (rc != HMPI_OK)
           {
              return rc;
           }

	   for (j = 0; j < q; j++)
           {
               column_np[i*q + j] = column_np_sub[j];
           }

	   free(column_np_sub);
	   free(column_speed_sums);
       }

       HMPI_Create_rectangles_2d(
           p,
	   q,
           m,
           n,
           row_np,
	   column_np,
	   w,
	   h,
           trow,
           tcol,
           ci,
           cj
       );

       free(row_np);
       free(column_np);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_column_based_speeds_single_numbers_no_mlimits
   (
       int p,
       int q,
       const double *speeds,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       int i, j, rc;
       double *column_speed_sums;
       int *column_np, *column_np_sub;
       int *row_np;

       column_speed_sums = (double*)malloc(
                           sizeof(double)
			   *
			   q
       );

       if (column_speed_sums == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       row_np = (int*)malloc(
                      sizeof(int)
		      *
		      (p*q)
       );

       if (row_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np = (int*)malloc(
                         sizeof(int)
		         *
		         (p*q)
       );

       if (column_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np_sub = (int*)malloc(
                             sizeof(int)
		             *
		             q
       );

       if (column_np_sub == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < q; i++)
       {
           column_speed_sums[i] = 0.0;
           for (j = 0; j < p; j++)
           {
               column_speed_sums[i] += speeds[j*q+i];
           }
       }

       /*
	* Partition of the column dimension among
	* q processors
	*/
       rc = HMPI_Partition_set(
                q,
                1,
	        column_speed_sums,
                NULL,
		NULL,
		n,
		NULL,
		0,
		0,
		-1,
		NULL,
		NULL,
		column_np_sub
       );

       if (rc == HMPI_ERR_PARTITION_SET)
       {
          return HMPI_ERR_PARTITION_MATRIX;
       }

       if (rc == HMPI_ERR_MLIMITS)
       {
          return HMPI_ERR_MLIMITS;
       }

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               column_np[i*q+j] = column_np_sub[j];
           }
       }

       free(column_speed_sums);
       free(column_np_sub);

       for (i = 0; i < q; i++)
       {
           double *row_speed_sums;
	   int *row_np_sub;

	   row_speed_sums = (double*)malloc(
			             sizeof(double)
				     *
				     p
           );

           if (row_speed_sums == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   row_np_sub = (int*)malloc(
			      sizeof(int)
			      *
			      p
           );

           if (row_np_sub == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   for (j = 0; j < p; j++)
           {
               row_speed_sums[j] = speeds[j*q + i];
           }

           /*
	    * Partition of the row dimension among
	    * p processors
	    */
           rc = HMPI_Partition_set(
                    p,
                    1,
	            row_speed_sums,
	    	    NULL,
                    NULL,
		    m,
		    NULL,
		    0,
		    0,
		    -1,
		    NULL,
		    NULL,
		    row_np_sub
           );

           if (rc == HMPI_ERR_PARTITION_SET)
           {
              return HMPI_ERR_PARTITION_MATRIX;
           }

           if (rc == HMPI_ERR_MLIMITS)
           {
              return HMPI_ERR_MLIMITS;
           }

           if (rc != HMPI_OK)
           {
              return rc;
           }

	   for (j = 0; j < p; j++)
           {
               row_np[j*q + i] = row_np_sub[j];
           }

	   free(row_np_sub);
	   free(row_speed_sums);
       }

       HMPI_Create_rectangles_2d(
           p,
	   q,
           m,
           n,
           row_np,
	   column_np,
	   w,
	   h,
           trow,
           tcol,
           ci,
           cj
       );

       free(row_np);
       free(column_np);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_column_based_speed_functions_no_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_column_based_speed_functions_with_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Matrix_two_dimensional_column_based
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       /*
        * Homogeneous distribution
        */
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_2d_no_mlimits(
                 p,
                 q,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * TBD: Meaning of mlimits
        */
       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_2d_column_based_speeds_single_numbers_no_mlimits(
                 p,
                 q,
                 speeds,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          /*
           * TBD: Meaning of mlimits
           * No known results for this case
           */
          return HMPI_2d_column_based_speeds_single_numbers_with_mlimits(
                 p,
                 q,
                 speeds,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          /*
           * No known results for this case
           */
          return HMPI_2d_column_based_speed_functions_no_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
                 psizes,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          /*
           * No known results for this case
           */
          return HMPI_2d_column_based_speed_functions_with_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_row_based_speed_functions_with_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_row_based_speed_functions_no_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_row_based_speeds_single_numbers_with_mlimits
   (
       int p,
       int q,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Matrix_two_dimensional_row_based
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_2d_no_mlimits(
                 p,
                 q,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_2d_row_based_speeds_single_numbers_no_mlimits(
                 p,
                 q,
                 speeds,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_2d_row_based_speeds_single_numbers_with_mlimits(
                 p,
                 q,
                 speeds,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_2d_row_based_speed_functions_no_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
		 psizes,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_2d_row_based_speed_functions_with_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
		 psizes,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_cartesian_speeds_single_numbers_no_mlimits
   (
       int p,
       int q,
       const double *speeds,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       int i, j, rc;
       double *row_speed_sums;
       int *row_np, *row_np_sub;
       double *column_speed_sums;
       int *column_np, *column_np_sub;

       row_np = (int*)malloc(
                      sizeof(int)
		      *
		      (p*q)
       );

       if (row_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       row_np_sub = (int*)malloc(
                          sizeof(int)
		          *
		          p
       );

       if (row_np_sub == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np = (int*)malloc(
                         sizeof(int)
		         *
		         (p*q)
       );

       if (column_np == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_np_sub = (int*)malloc(
                             sizeof(int)
		             *
		             q
       );

       if (column_np_sub == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       row_speed_sums = (double*)malloc(
			         sizeof(double)
			         *
				 p
       );

       if (row_speed_sums == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       column_speed_sums = (double*)malloc(
                           sizeof(double)
			   *
			   q
       );

       if (column_speed_sums == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           row_speed_sums[i] = 0.0;
           for (j = 0; j < q; j++)
           {
               row_speed_sums[i] += speeds[i*q+j];
           }
       }

       for (i = 0; i < q; i++)
       {
           column_speed_sums[i] = 0.0;
           for (j = 0; j < p; j++)
           {
               column_speed_sums[i] += speeds[j*q+i];
           }
       }

       /*
	* Partition of the row dimension among
	* p processors
	*/
       rc = HMPI_Partition_set(
                p,
                1,
		row_speed_sums,
		NULL,
		NULL,
		m,
		NULL,
		0,
		0,
		-1,
		NULL,
		NULL,
		row_np_sub
       );

       if (rc == HMPI_ERR_PARTITION_SET)
       {
          printf(
	    "Problems partitioning the matrix "
	    " using HMPI_CARTESIAN two-dimensional distribution\n"
          );

          return HMPI_ERR_PARTITION_MATRIX;
       }

       if (rc == HMPI_ERR_MLIMITS)
       {
          printf(
	    "Problems applying the limits during "
	    " matrix partitioning using HMPI_CARTESIAN "
	    "two-dimensional distribution\n"
          );

          return HMPI_ERR_MLIMITS;
       }

       if (rc != HMPI_OK)
       {
          return rc;
       }

       /*
	* Partition of the column dimension among
	* q processors
	*/
       rc = HMPI_Partition_set(
                q,
                1,
	        column_speed_sums,
		NULL,
		NULL,
		n,
		NULL,
		0,
		0,
		-1,
		NULL,
		NULL,
		column_np_sub
       );

       if (rc == HMPI_ERR_PARTITION_SET)
       {
          printf(
	    "Problems partitioning the matrix"
	    " using HMPI_CARTESIAN two-dimensional distribution\n"
          );
          return HMPI_ERR_PARTITION_MATRIX;
       }

       if (rc == HMPI_ERR_MLIMITS)
       {
          printf(
	    "Problems applying the limits during"
	    " matrix partitioning using HMPI_CARTESIAN "
	    "two-dimensional distribution\n"
          );

          return HMPI_ERR_MLIMITS;
       }

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               row_np[i*q+j] = row_np_sub[i];
               column_np[i*q+j] = column_np_sub[j];
           }
       }

       HMPI_Create_rectangles_2d(
           p,
	   q,
           m,
           n,
           row_np,
	   column_np,
	   w,
	   h,
	   trow,
	   tcol,
           ci,
           cj
       );

       free(row_np);
       free(column_np);
       free(row_np_sub);
       free(column_np_sub);
       free(row_speed_sums);
       free(column_speed_sums);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_cartesian_speeds_single_numbers_with_mlimits
   (
       int p,
       int q,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_cartesian_speed_functions_with_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_2d_cartesian_speed_functions_no_mlimits
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Matrix_two_dimensional_cartesian
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_2d_no_mlimits(
                 p,
                 q,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_2d_cartesian_speeds_single_numbers_no_mlimits(
                 p,
                 q,
                 speeds,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_2d_cartesian_speeds_single_numbers_with_mlimits(
                 p,
                 q,
                 speeds,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_2d_cartesian_speed_functions_no_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
		 psizes,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_2d_cartesian_speed_functions_with_mlimits(
                 p,
                 q,
                 pn,
                 speeds,
		 psizes,
                 mlimits,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci,
                 cj
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_matrix_2d
   (
       int p,
       int q,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci,
       int *cj
   )
   {
       switch (type_of_distribution) 
       {
          case HMPI_COLUMN_BASED:
            {
               return HMPI_Matrix_two_dimensional_column_based(
                      p,
		      q,
                      pn, 
		      speeds,
		      psizes,
		      mlimits,
		      m,
		      n,
		      w,
		      h,
                      trow,
                      tcol,
                      ci,
                      cj
               );
            }
	    break;
	  case HMPI_ROW_BASED:
            {
               return HMPI_Matrix_two_dimensional_row_based(
                      p,
		      q,
                      pn,
		      speeds,
		      psizes,
		      mlimits,
		      m,
		      n,
		      w,
		      h,
                      trow,
                      tcol,
                      ci,
                      cj
               );
            }
	    break;
	  case HMPI_CARTESIAN:
            {
               return HMPI_Matrix_two_dimensional_cartesian(
                      p,
		      q,
                      pn,
		      speeds,
		      psizes,
		      mlimits,
		      m,
		      n,
		      w,
		      h,
                      trow,
                      tcol,
                      ci,
                      cj
               );
            }
	    break;
          default:
	    {
               printf(
	         "Invalid type of distribution provided"
		 " for two-dimensional processor arrangement\n"
	       );
               return HMPI_ERR_PARTITION_MATRIX;
	       break;
            }
       }

       printf(
         "Invalid type of distribution provided"
	 " for two-dimensional processor arrangement\n"
       );

       return HMPI_ERR_PARTITION_MATRIX;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_dynamic_row_based_speeds_single_numbers_no_mlimits
   (
       int p,
       const double *speeds,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       return HMPI_1d_dynamic_column_based_speeds_single_numbers_no_mlimits(
              p,
              speeds,
              n,
              m,
              lb,
              dpf,
              w,
              h,
              trow,
              tcol
       );
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_dynamic_column_based_speeds_single_numbers_no_mlimits
   (
       int p,
       const double *speeds,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       int i, j, q, C, c_opt;
       double S = 0.0, tspeed = 0.0, tarea = 0.0;
       double **perimeter;
       int **cumulative_r;
       double *one_d_p;
       int *one_d_r, *optimal_d;
       double *areas, *rearranged_speeds;
       int *rearrangedp;
       double temp;
       int temp_number;

       /*
        * Sort the speeds in ascending order
        */
       rearranged_speeds = (double*)malloc(
                                    sizeof(double)
                 		    *
				    p
       );

       if (rearranged_speeds == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rearrangedp = (int*)malloc(
                           sizeof(int)
                  	   *
			   p
       );

       if (rearrangedp == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          rearrangedp[i] = i;
       }

       for (i = 0; i < p; i++)
       {
           rearranged_speeds[i] = speeds[i];
       }

       for (i = 0; i < p; i++)
       {
          for (j = 1; j < p; j++)
          {
             if (rearranged_speeds[j-1] > rearranged_speeds[j])
             {
                temp = rearranged_speeds[j-1];
                rearranged_speeds[j-1] = rearranged_speeds[j];
	        rearranged_speeds[j] = temp;

	        temp_number = rearrangedp[j-1];
	        rearrangedp[j-1] = rearrangedp[j];
		rearrangedp[j] = temp_number;
              }
          }
       }

       /*
        * Normalise to two decimal places
        * Sum of the areas should be 1.
        */
       areas = (double*)malloc(
                        sizeof(double)
                        *
                        p
       );

       if (areas == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          tspeed += rearranged_speeds[i];
       }

       for (i = 0; i < p; i++)
       {
          int to_2_decimals;
          areas[i] = (rearranged_speeds[i]/tspeed)*100;
          to_2_decimals = areas[i];
          areas[i] = to_2_decimals;
          areas[i] = (areas[i]/100);
          tarea += areas[i];
       }

       areas[0] = areas[0] + (1 - tarea);

       /*
        * perimeter and cumulative_r are Upper Triangular arrangements
        * Study the paper 'Matrix Multiplication on Heterogeneous
        * Platforms' by Beaumont et al to see the layout of the 
        * arrays perimeter and cumulative_r
        */
       one_d_p = (double*)malloc(
                       sizeof(double)
                       *
                       (p*(p+1)/2)
       );

       if (one_d_p == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       one_d_r = (int*)malloc(
                    sizeof(int)
                    *
                    (p*(p+1)/2)
       );

       if (one_d_r == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       perimeter = (double**)malloc(
                             sizeof(double*)
                             *
                             p
       );

       if (perimeter == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       cumulative_r = (int**)malloc(
                             sizeof(int*)
                             *
                             p
       );

       if (cumulative_r == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       optimal_d = (int*)malloc(
                         sizeof(int)
                         *
                         p
       );

       if (optimal_d == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          perimeter[i] = one_d_p - i;
          one_d_p+=(p-i);

          cumulative_r[i] = one_d_r - i;
          one_d_r+=(p-i);
       }

       for (q = 0; q < p; q++)
       {
          S += speeds[q];
          perimeter[0][q] = 1 + S*q;
          cumulative_r[0][q] = 0;
       }

       for (C = 1; C < p; C++)
       {
          for (q = C; q < p; q++)
          {
             int r_opt;

             perimeter[C][q] = (*dpf)(
                               C,
                               q,
                               p,
                               rearranged_speeds,
                               perimeter,
                               &r_opt
             );

             cumulative_r[C][q] = r_opt;
          }
       }

       q = p;
       c_opt = p;
       i = 0;

       while (c_opt >= 2)
       {
          int temp, c_optimal = 1;

          temp = perimeter[0][q];

          for (C = 1; C < p; C++)
          {
             if ((perimeter[C][q-1]) < temp)
             {
                c_optimal = C+1;
                temp = perimeter[C][q-1];
             }
          }

          optimal_d[i++] = q - cumulative_r[c_optimal-1][q-1];
          q = cumulative_r[c_optimal-1][q-1];
          c_opt = c_optimal;
       }

       optimal_d[i] = q;

       /*
        * Fill the output parameters
        */
       HMPI_Create_rectangles_1d_column_based(
           p,
	   i+1,
           optimal_d,
           areas,
           m,
           n,
	   w,
	   h,
           trow,
           tcol
       );

       free(one_d_p);
       free(one_d_r);
       free(perimeter);
       free(optimal_d);
       free(cumulative_r);
       free(rearranged_speeds);
       free(rearrangedp);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * column-based partitioning/row-based partitioning 
    * of the matrix using the column-based heuristic approach 
    * proposed by Beaumont et al
    */
   int HMPI_1d_dynamic_speeds_single_numbers_no_mlimits
   (
       int p,
       const double *speeds,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       switch (type_of_distribution)
       {
          case HMPI_ROW_BASED:
            {
               return 
               HMPI_1d_dynamic_row_based_speeds_single_numbers_no_mlimits(
                    p,
                    speeds,
                    m,
                    n,
                    lb,
                    dpf,
                    w,
                    h,
                    trow,
                    tcol
               );
            }
	    break;
          case HMPI_COLUMN_BASED:
            {
               return 
               HMPI_1d_dynamic_column_based_speeds_single_numbers_no_mlimits(
                    p,
                    speeds,
                    m,
                    n,
                    lb,
                    dpf,
                    w,
                    h,
                    trow,
                    tcol
               );
            }
            break;
          case HMPI_GENERAL:
            {
            }
            break;
          default:
            {
               printf(
	         "Invalid type of distribution provided"
		 " for one-dimensional processor arrangement "
                 " with DYNAMIC formulation\n"
	       );

               return HMPI_ERR_PARTITION_MATRIX;

	       break;
            }
       }

       printf(
         "Invalid type of distribution provided for one-dimensional "
         "processor arrangement with DYNAMIC formulation\n"
       );

       return HMPI_ERR_PARTITION_MATRIX;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_dynamic_speeds_single_numbers_with_mlimits
   (
       int p,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_dynamic_speed_functions_no_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_dynamic_speed_functions_with_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_matrix_1d_dp
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_DP_function dpf,
       int type_of_distribution,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci
   )
   {
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_1d_no_mlimits(
                 p,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci
          );
       }

       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_dynamic_speeds_single_numbers_no_mlimits(
                 p,
                 speeds,
                 m,
                 n,
                 lb,
                 dpf,
                 type_of_distribution,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_dynamic_speeds_single_numbers_with_mlimits(
                 p,
                 speeds,
                 mlimits,
                 m,
                 n,
                 lb,
                 dpf,
                 type_of_distribution,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_dynamic_speed_functions_no_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 m,
                 n,
                 lb,
		 dpf,
                 type_of_distribution,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_dynamic_speed_functions_with_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 m,
                 n,
                 lb,
                 dpf,
                 type_of_distribution,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/


   int  HMPI_Next_best_matrix_partition(
            int m,
            int n,
            int* oldw,
            int* oldh,
            int* oldtrow,
            int* oldtcol
   )
   {
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_iterative_speeds_single_numbers_no_mlimits
   (
       int p,
       const double *speeds,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Iterative_function cf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       int i, rc;
       int Lower_bound = (*lb)(
                           p,
			   speeds,
			   m,
			   n
       );
       int *oldw, *oldh, *oldtrow, *oldtcol;
       double cost;

       oldw = (int*)malloc(
	            sizeof(int)
		    *
		    p
       );

       if (oldw == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldh = (int*)malloc(
                    sizeof(int)
                    *
                    (p*p)
       );

       if (oldh == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldtrow = (int*)malloc(
		       sizeof(int)
		       *
		       p
       );
       
       if (oldtrow == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldtcol = (int*)malloc(
		       sizeof(int)
		       *
		       p
       );
       
       if (oldtcol == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           oldw[i] = w[i];
       }

       for (i = 0; i < (p*p); i++)
       {
           oldh[i] = h[i];
       }

       for (i = 0; i < p; i++)
       {
           oldtrow[i] = trow[i];
       }

       for (i = 0; i < p; i++)
       {
           oldtcol[i] = tcol[i];
       }

       do
       {
           cost = (*cf)(
                   p,
		   oldw,
		   oldh,
		   oldtrow,
		   oldtcol
           );

	   if ((cost < 0)
               || (cost <= Lower_bound
               )
           )
           {
              for (i = 0; i < p; i++)
              {
                  w[i] = oldw[i];
              }

              for (i = 0; i < (p*p); i++)
              {
                  h[i] = oldh[i];
              }

              for (i = 0; i < p; i++)
              {
                  trow[i] = oldtrow[i];
              }

              for (i = 0; i < p; i++)
              {
                  tcol[i] = oldtcol[i];
              }
           }
	   else
           {
              /*
               * TBD
               */
              HMPI_Next_best_matrix_partition(
                  m,
		  n,
                  oldw,
		  oldh,
		  oldtrow,
		  oldtcol
              );
           }
       } while ((cost > 0) && (cost > Lower_bound));

       free(oldw);
       free(oldh);
       free(oldtrow);
       free(oldtcol);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int  HMPI_1d_iterative_speeds_single_numbers_with_mlimits
   (
       int p,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Iterative_function cf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_iterative_speed_functions_no_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Iterative_function cf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_iterative_speed_functions_with_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Iterative_function cf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_matrix_1d_iterative
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Iterative_function cf,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci
   )
   {
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_1d_no_mlimits(
                 p,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci
          );
       }

       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_iterative_speeds_single_numbers_no_mlimits(
                 p,
                 speeds,
                 m,
                 n,
                 lb,
                 cf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_iterative_speeds_single_numbers_with_mlimits(
                 p,
                 speeds,
                 mlimits,
                 m,
                 n,
                 lb,
                 cf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_iterative_speed_functions_no_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 m,
                 n,
                 lb,
                 cf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_iterative_speed_functions_with_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 m,
                 n,
                 lb,
                 cf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_refining_speeds_single_numbers_no_mlimits
   (
       int p,
       const double *speeds,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Refining_function rf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       int i, rc;
       int *oldw, *oldh, *oldtrow, *oldtcol;

       oldw = (int*)malloc(
		    sizeof(int)
		    *
		    p
       );
       
       if (oldw == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldh = (int*)malloc(
                    sizeof(int)
                    *
                    (p*p)
       );

       if (oldh == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldtrow = (int*)malloc(
		       sizeof(int)
		       *
		       p
       );
       
       if (oldtrow == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       oldtcol = (int*)malloc(
		       sizeof(int)
		       *
		       p
       );
       
       if (oldtcol == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           oldw[i] = w[i];
       }

       for (i = 0; i < (p*p); i++)
       {
           oldh[i] = h[i];
       }

       for (i = 0; i < p; i++)
       {
           oldtrow[i] = trow[i];
       }

       for (i = 0; i < p; i++)
       {
           oldtcol[i] = tcol[i];
       }

       do
       {
           rc = (*rf)(
                   p,
		   speeds,
		   m,
		   n,
		   oldw,
		   oldh,
		   oldtrow,
		   oldtcol,
		   w,
		   h,
		   trow,
		   tcol
           );

	   if (rc > 0)
	   {
              for (i = 0; i < p; i++)
              {
                  oldw[i] = w[i];
              }

              for (i = 0; i < (p*p); i++)
              {
                  oldh[i] = h[i];
              }

              for (i = 0; i < p; i++)
              {
                  oldtrow[i] = trow[i];
              }

              for (i = 0; i < p; i++)
              {
                  oldtcol[i] = tcol[i];
              }
	   }
           else
	   {
              for (i = 0; i < p; i++)
              {
                  w[i] = oldw[i];
              }

              for (i = 0; i < (p*p); i++)
              {
                  h[i] = oldh[i];
              }

              for (i = 0; i < p; i++)
              {
                  trow[i] = oldtrow[i];
              }

              for (i = 0; i < p; i++)
              {
                  tcol[i] = oldtcol[i];
              }
	   }
       } while (rc > 0);

       free(oldw);
       free(oldh);
       free(oldtrow);
       free(oldtcol);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_refining_speeds_single_numbers_with_mlimits
   (
       int p,
       const double *speeds,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Refining_function rf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_refining_speed_functions_no_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Refining_function rf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_1d_refining_speed_functions_with_mlimits
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Refining_function rf,
       int *w,
       int *h,
       int *trow,
       int *tcol
   )
   {
       printf("Implementation currently not available\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_matrix_1d_refining
   (
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int m,
       int n,
       HMPI_Lower_bound lb,
       HMPI_Refining_function rf,
       int *w,
       int *h,
       int *trow,
       int *tcol,
       int *ci
   )
   {
       if ((speeds == NULL)
           && (mlimits == NULL
          )
       )
       {
          return HMPI_Homogeneous_matrix_1d_no_mlimits(
                 p,
                 m,
                 n,
                 w,
                 h,
                 trow,
                 tcol,
                 ci
          );
       }

       /*
        * Meaning of mlimits
        */
       if ((speeds == NULL)
           && (mlimits != NULL
          )
       )
       {
       }

       if ((mlimits == NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_refining_speeds_single_numbers_no_mlimits(
                 p,
                 speeds,
                 m,
                 n,
                 lb,
                 rf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn == 1
          )
       )
       {
          return HMPI_1d_refining_speeds_single_numbers_with_mlimits(
                 p,
                 speeds,
                 mlimits,
                 m,
                 n,
                 lb,
                 rf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits == NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_refining_speed_functions_no_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 m,
                 n,
                 lb,
                 rf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       /*
        * No known results for this case
        */
       if ((mlimits != NULL)
           && (pn > 1
          )
       )
       {
          return HMPI_1d_refining_speed_functions_with_mlimits(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 m,
                 n,
                 lb,
                 rf,
                 w,
                 h,
                 trow,
                 tcol
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_processor_2d
   (
       int row,
       int column,
       int p,
       int q,
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int *I,
       int *J
   )
   {
       int x, y, i, j;

       for (x = 0; x < p; x++)
       {
           for (y = 0; y < q; y++)
           {
               int hi = h[HMPI_RECT_INDEX(x, y, x, y, p, q)];
               int wi = w[HMPI_RECT_INDEX(x, y, x, y, p, q)];
	       int toprow = trow[x*q+y];
               int topcol = tcol[x*q+y];

	       for (i = 0; i < hi; i++)
	       {
		   for (j = 0; j < wi; j++) 
		   {
                       if (((row >= (toprow + i)) 
	 	            && (row < (toprow + hi)
			    )
			   ) 
                           &&
			   ((column >= (topcol + j))
			     && (column < (topcol + wi)
			    )
			   )
                       )
                       {
                          *I = i;
		          *J = j;

		          return HMPI_OK;
                       }
                   }
               }
           }
       }

       return HMPI_ERR_PARTITION_NOT_EXISTS;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_processor_1d
   (
       int row,
       int column,
       int p,
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int *I
   )
   {
       int x, y, i, j;

       for (x = 0; x < p; x++)
       {
           int hi = h[HMPI_RECT_INDEX(x, 0, x, 0, p, 1)];
           int wi = w[HMPI_RECT_INDEX(x, 0, x, 0, p, 1)];
           int toprow = trow[x];
           int topcol = tcol[x];

	   for (i = 0; i < hi; i++)
	   {
	       for (j = 0; j < wi; j++) 
	       {
                   if (((row >= (toprow + i)) 
	                && (row < (toprow + hi)
		        )
		       ) 
                       &&
		       ((column >= (topcol + j))
		        && (column < (topcol + wi)
		        )
		       )
                   )
                   {
                      *I = i;
		      return HMPI_OK;
                   }
               }
           }
       }

       return HMPI_ERR_PARTITION_NOT_EXISTS;
   }

   /*-----------------------------------------------------*/

   int HMPI_Print_rectangle_1d(
        int p,
        int m,
        int n,
        const int *w,
        const int *h,
        const int *trow,
        const int *tcol,
        const int *ci
   )
   {
       int i, j, k, l;
       int q = 1;

       if (ci != NULL)
       {
           printf("The processor allocation is :\n");

           for (i = 0; i < m; i++)
           {
               for (j = 0; j < n; j++)
               {
                  printf("(%d) ", ci[i*n+j]);
               }
               printf("\n");
           }

           printf("\n");
       }

       printf("The top row coordinates are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
              printf("%d ", trow[i*q+j]);
           }
           printf("\n");
       }

       printf("\n");

       printf("The top column coordinates are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
              printf("%d ", tcol[i*q+j]);
           }
           printf("\n");
       }

       printf("\n");

       printf("The common widths of rectangles are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (k = 0; k < p; k++)
               {
                   for (l = 0; l < q; l++)
                   {
                      printf("%d ", w[HMPI_RECT_INDEX(i, j, k, l, p, q)]);
                   }
               }
               printf("\n");
           }
       }

       printf("\n");

       printf("The common heights of rectangles are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (k = 0; k < p; k++)
               {
                   for (l = 0; l < q; l++)
                   {
                      printf("%d ", h[HMPI_RECT_INDEX(i, j, k, l, p, q)]);
                   }
               }
               printf("\n");
           }
       }

       printf("\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Print_rectangle_2d(
        int p,
        int q,
        int m,
        int n,
        const int *w,
        const int *h,
        const int *trow,
        const int *tcol,
        const int *ci,
        const int *cj
   )
   {
       int i, j, k, l;

       if ((ci != NULL)
           && (cj != NULL
          )
       )
       {
           printf("The processor allocation is :\n");

           for (i = 0; i < m; i++)
           {
               for (j = 0; j < n; j++)
               {
                  printf("(%d,%d) ", ci[i*n+j], cj[i*n+j]);
               }
               printf("\n");
           }

           printf("\n");
       }

       printf("The top row coordinates are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
              printf("%d ", trow[i*q+j]);
           }
           printf("\n");
       }

       printf("\n");

       printf("The top column coordinates are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
              printf("%d ", tcol[i*q+j]);
           }
           printf("\n");
       }

       printf("\n");

       printf("The common widths of rectangles are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (k = 0; k < p; k++)
               {
                   for (l = 0; l < q; l++)
                   {
                      printf("%d ", w[HMPI_RECT_INDEX(i, j, k, l, p, q)]);
                   }
               }
               printf("\n");
           }
       }

       printf("\n");

       printf("The common heights of rectangles are:\n");

       for (i = 0; i < p; i++)
       {
           for (j = 0; j < q; j++)
           {
               for (k = 0; k < p; k++)
               {
                   for (l = 0; l < q; l++)
                   {
                      printf("%d ", h[HMPI_RECT_INDEX(i, j, k, l, p, q)]);
                   }
               }
               printf("\n");
           }
       }

       printf("\n");

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/
   
   /*
    * If the user set trow and tcol as NULL, then the 
    * type_of_distribution has to be two-dimensional.
    */
   int HMPI_Get_matrix_processor (
           int r,
           int c,
           int p,
           int q, 
           const int *w,
           const int *h,
           const int *trow,
           const int *tcol,
           HMPI_Processor *root
   )
   {
      int i, j, rc;

      if ((p==1) || (q==1))
      {
         if (p == 1)
         {
            rc = HMPI_Get_processor_1d(
                     r,
                     c,
                     q,
                     w,
                     h,
                     trow,
                     tcol,
                     &c
            );

            if (rc != HMPI_OK)
            {
               return rc;
            }

            root->I = 0;
            root->J = c;

            return HMPI_OK;
         }

         rc = HMPI_Get_processor_1d(
                  r,
                  c,
                  p,
                  w,
                  h,
                  trow,
                  tcol,
                  &c
         );

         if (rc != HMPI_OK)
         {
            return rc;
         }

         root->I = c;
         root->J = 0;

         return HMPI_OK;
      }

      rc = HMPI_Get_processor_2d(
                  r,
                  c,
                  p,
                  q,
                  w,
                  h,
                  trow,
                  tcol,
                  &i,
                  &j
      );

      if (rc != HMPI_OK)
      {
         return rc;
      }

      root->I = i;
      root->J = j;

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_my_width
   (
       int i,
       int j,
       int p,
       int q,
       const double *speeds,
       int type_of_distribution,
       int m,
       int n
   )
   {
       int rc;
       int *w, *h, *tcol, *trow;
       int width;

       w = (int*)malloc(
                 sizeof(int)
                 *
                 (p*q)
       );

       if (w == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       h = (int*)malloc(
                 sizeof(int)
                 *
                 (p*q*p*q)
       );

       if (h == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       trow = (int*)malloc(
                    sizeof(int)
                    *
                    (p*q)
       );

       if (trow == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       tcol = (int*)malloc(
                    sizeof(int)
                    *
                    (p*q)
       );

       if (tcol == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = HMPI_Partition_matrix_2d(
                    p,
                    q,
                    1,
                    speeds,
                    NULL,
                    NULL,
                    m,
                    n,
                    type_of_distribution,
                    w,
                    h,
                    trow,
                    tcol,
                    NULL,
                    NULL
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       width = w[i*q+j];

       free(w);
       free(h);
       free(trow);
       free(tcol);

       return width;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_my_height
   (
       int i,
       int j,
       int p,
       int q,
       const double *speeds,
       int type_of_distribution,
       int m,
       int n
   )
   {
       int rc;
       int *w, *h, *tcol, *trow;
       int height;

       w = (int*)malloc(
                 sizeof(int)
                 *
                 (p*q)
       );

       if (w == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       h = (int*)malloc(
                 sizeof(int)
                 *
                 (p*q*p*q)
       );

       if (h == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       trow = (int*)malloc(
                    sizeof(int)
                    *
                    (p*q)
       );

       if (trow == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       tcol = (int*)malloc(
                    sizeof(int)
                    *
                    (p*q)
       );

       if (tcol == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = HMPI_Partition_matrix_2d(
                    p,
                    q,
                    1,
                    speeds,
                    NULL,
                    NULL,
                    m,
                    n,
                    type_of_distribution,
                    w,
                    h,
                    trow,
                    tcol,
                    NULL,
                    NULL
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       height = HMPI_RECT_INDEX(i, j, i, j, p, q);

       free(w);
       free(h);
       free(trow);
       free(tcol);

       return height;
   }

   /*-----------------------------------------------------*/

   int HMPI_Get_diagonal
   (
       int i,
       int j,
       int p,
       int q,
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol
   )
   {
       return min(w[i*q+j], HMPI_RECT_INDEX(i, j, i, j, p, q));
   }

   /*-----------------------------------------------------*/

   int _HMPI_a22elements
   (
       int k,
       int n,
       int l,
       int w,
       int h,
       int trow,
       int tcol
   )
   {
       if (((k+1)%l) >= (tcol+w))
       {
          return (_HMPI_myelements(n, l, w, h, trow, tcol) -
                 (1/2)*((k+1)/l+1)*(2*(n/l) - (k+1)/l)*_HMPI_myelements_g(n, l, w, h, trow, tcol));
       }

       if ((((k+1)%l) < trow)
           && (((k+1)%l) < tcol
          )
       )
       {
          return (_HMPI_myelements(n, l, w, h, trow, tcol) -
                 ((1/2)*(n/l)*(n/l+1)- (1/2)*(n/l - (k+1)/l)*(n/l - (k+1)/l + 1))*_HMPI_myelements_g(n, l, w, h, trow, tcol));
       }

       if (((((k+1)%l) >= trow))
           && ((((k+1)%l) < (trow+h))
          )
          &&
          ((((k+1)%l) >= (tcol))
           && (((k+1)%l) < (tcol+w))
          )
       )
       {
          return (_HMPI_myelements(n, l, w, h, trow, tcol) -
                 ((1/2)*(n/l)*(n/l+1)- (1/2)*(n/l - (k+1)/l)*(n/l - (k+1)/l + 1))*_HMPI_myelements_g(n, l, w, h, trow, tcol) -
                 ((n/l - (k+1)/l)*_HMPI_myelements_g(n, l, w, h, trow, tcol) - (1/2)*(tcol+w - (k+1)%l)*(tcol+w - (k+1)%l + 1))
          );
       }

       if (((((k+1)%l) >= trow))
           && ((((k+1)%l) < (trow+h))
          )
          &&
          (((k+1)%l) < (tcol)
          )
       )
       {
          return ((1/2)*(k+1/l)*(2*(n/l) - (k+1)/l - 1)*_HMPI_myelements_g(n, l, w, h, trow, tcol));
       }

       if ((((k+1)%l) > (trow+h))
          &&
           (((k+1)%l) < tcol
          )
       )
       {
          return ((1/2)*(k+1/l)*(2*(n/l) - (k+1)/l - 1)*_HMPI_myelements_g(n, l, w, h, trow, tcol));
       }

       if ((((k+1)%l) > (trow+h))
          &&
          ((((k+1)%l) >= (tcol))
           && (((k+1)%l) < (tcol+w))
          )
       )
       {
          return (_HMPI_myelements(n, l, w, h, trow, tcol) -
                 (1/2)*(k+1/l)*(2*(n/l) - (k+1)/l - 1)*_HMPI_myelements_g(n, l, w, h, trow, tcol) -
                 (n/l - (k+1)/l)*h*((k+1)%l- tcol));
       }

       if (((((k+1)%l) < trow))
          &&
          ((((k+1)%l) >= (tcol))
           && (((k+1)%l) < (tcol+w))
          )
       )
       {
          return (_HMPI_myelements(n, l, w, h, trow, tcol) -
                 ((1/2)*(n/l)*(n/l+1)- (1/2)*(n/l - (k+1)/l)*(n/l - (k+1)/l + 1))*_HMPI_myelements_g(n, l, w, h, trow, tcol) -
                 (n/l - (k+1)/l)*h*((k+1)%l- tcol));
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int _HMPI_myelements_g
   (
       int n,
       int l,
       int w,
       int h,
       int trow,
       int tcol
   )
   {
       if ((w == h)
           && ((trow+h) == (tcol+w)
          )
       )
       {
          return ((1/2)*w*(w+1));
       }

       if (trow >= (tcol+w))
       {
          return w*h;
       }

       if ((trow+h) <= tcol)
       {
          return 0;
       }

       if ((trow == 0)
          && (tcol == 0
          )
       )
       {
          if ((trow < (tcol+w))
              && ((trow+h) > tcol
             )
          )
          {
             return 1/2*h*(h+1);
          }
       }

       if ((trow < (tcol+w))
           && ((trow+h) > tcol
          )
       )
       {
          if ((trow == tcol)
              && ((trow+h) == (tcol+w)
             )   
          )
          {
             return 1/2*w*(w+1);
          }
          if ((trow == tcol)
              && ((trow+h) < (tcol+w)
             )   
          )
          {
             return (1/2*((trow+h)-tcol)*((trow+h)-tcol+1));
          }
          if ((trow == tcol)
              && ((trow+h) > (tcol+w)
             )   
          )
          {
             return (w*h - 1/2*((tcol+w)-trow)*((tcol+w)-trow+1));
          }
          if (((trow+h) == (tcol+w))
              && (trow > tcol
             )   
          )
          {
             return (w*h - 1/2*((tcol+w)-trow)*((tcol+w)-trow+1));
          }
          if (((trow+h) == (tcol+w))
              && (trow < tcol
             )   
          )
          {
             return (1/2*((trow+h)-tcol)*((trow+h)-tcol+1));
          }
          if ((trow > tcol)
              && ((trow+h) < (tcol+w)
             )
          )
          {
             return (1/2*h*(trow-tcol + trow+h-tcol));
          }
          if ((trow < tcol)
              && ((trow+h) > (tcol+w)
             )
          )
          {
             return (1/2*w*(tcol-trow + tcol+w-trow));
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int _HMPI_myelements
   (
       int n,
       int l,
       int w,
       int h,
       int trow,
       int tcol
   )
   {
       if ((w == h)
           && ((trow+h) == (tcol+w)
          )
       )
       {
          return (((((n/l)*(n/l+1))/2) - (n/l))*w*h + (n/l)*(1/2)*w*(w+1));
       }

       if (trow >= (tcol+w))
       {
          return (((n/l)*(n/l+1))/2)*w*h;
       }

       if ((trow+h) <= tcol)
       {
          return (((((n/l)*(n/l+1))/2) - (n/l))*w*h);
       }

       if ((trow == 0)
          && (tcol == 0
          )
       )
       {
          if ((trow < (tcol+w))
              && ((trow+h) > tcol
             )
          )
          {
             return (((((n/l)*(n/l+1))/2) - (n/l))*w*h + 1/2*(n/l)*h*(h+1));
          }
       }

       if ((trow < (tcol+w))
           && ((trow+h) > tcol
          )
       )
       {
          if ((trow == tcol)
              && ((trow+h) == (tcol+w)
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + 1/2*w*(w+1));
          }
          if ((trow == tcol)
              && ((trow+h) < (tcol+w)
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + (1/2*((trow+h)-tcol)*((trow+h)-tcol+1)));
          }
          if ((trow == tcol)
              && ((trow+h) > (tcol+w)
             )
          )
          {
             return ((((n/l)*(n/l+1)/2 - n/l)*w*h + w*h - 1/2*((tcol+w)-trow)*((tcol+w)-trow+1)));
          }
          if (((trow+h) == (tcol+w))
              && (trow > tcol
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + (w*h - 1/2*((tcol+w)-trow)*((tcol+w)-trow+1)));
          }
          if (((trow+h) == (tcol+w))
              && (trow < tcol
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + 1/2*(trow+h-tcol)*(trow+h-tcol+1));
          }
          if ((trow > tcol)
              && ((trow+h) < (tcol+w)
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + 1/2*h*(trow-tcol + trow+h-tcol));
          }
          if ((trow < tcol)
              && ((trow+h) > (tcol+w)
             )
          )
          {
             return (((n/l)*(n/l+1)/2 - n/l)*w*h + 1/2*w*(tcol-trow + tcol+w-trow));
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * Currently only square dense matrices are assumed.
    */
   int HMPI_Get_my_elements
   (
       int m,
       int n,
       int gm,
       int gn,
       int i,
       int j,
       int p,
       int q,
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int type_of_distribution,
       char upper_or_lower
   )
   {
       int lower  = _HMPI_myelements(
                         m,
                         gm,
                         w[i*q+j],
                         HMPI_RECT_INDEX(i, j, i, j, p, q),
                         trow[i*q+j],
                         tcol[i*q+j]
       );

       if (upper_or_lower == 'L')
       {
          return lower;
       }
   
       return (n*n - lower);
   }

   /*-----------------------------------------------------*/

   /*
    * Currently only square dense matrices are assumed.
    */
   int HMPI_Get_my_kk_elements
   (
       int k,
       int m,
       int n,
       int gm,
       int gn,
       int i,
       int j,
       int p,
       int q,
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int type_of_distribution,
       char upper_or_lower
   )
   {
       int lower= _HMPI_a22elements(
                       k,
                       m,
                       gm,
                       w[i*q+j],
                       HMPI_RECT_INDEX(i, j, i, j, p, q),
                       trow[i*q+j],
                       tcol[i*q+j]
       );

       if (upper_or_lower == 'L')
       {
          return lower;
       }
   
       return ((m-k)*(m-k) - lower);
   }

   /*-----------------------------------------------------*/
