
   /************************************************/
   /* Partitioning interfaces for graphs           */
   /*                                              */
   /* Revision history                             */
   /* 22-04-2004  --      Initial version          */
   /************************************************/

   #include <hmpi.h>
   #include <metis.h>
   #include <hmetis.h>

   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>
   
   #define  HMPI_UBFACTOR   8

   /*
    * Third Party Software METIS and hMETIS is used
    * for some of the partitioning problems.
    * In future, METIS and hMETIS and HMPI can be integrated
    * so that application programmers can provide the 
    * various options for multi-level partitioning of graphs.
    *
    * In METIS, there are two partitioning objectives:
    * (a) Minimizing the number of edges that straddle 
    *     partitions (edgecut)
    * (b) Minimizing the total communication volume.
    *
    * We use the objective b.
    *
    * Please refer to METIS & hMETIS documentation for 
    * more information on the usage of the graph partitioning
    * interfaces.
    */ 

   /*-----------------------------------------------------*/

   int HMPI_Partition_graph(
       int p,
       int pn,
       const double *speeds,
       const int *psizes,
       const int *mlimits,
       int n,
       int m,
       const int *vwgt,
       const int *xadj,
       const int *adjacency,
       const int *adjwgt,
       int *vp,
       int *edgecut
   )
   {
       int i, j, rc;

       /*
        * Use the partitioning interface for Set when 
        * edges have no weights.
        */
       if ((vwgt == NULL)
           && (xadj == NULL)
           && (adjacency == NULL)
           && (adjwgt == NULL
          )
       )
       {
          rc = HMPI_Partition_set(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 n,
                 NULL,
                 0,
                 1,
                 -1,
                 NULL,
                 NULL,
                 vp
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          *edgecut = 0;

          for (i = 0; i < n; i++)
          {
             int owner_processor = vp[i];

             for (j = xadj[i]; j < xadj[i+1]; j++)
             {
                int neighbor = adjacency[j];
 
                if (vp[neighbor] != owner_processor)
                {
                   (*edgecut)++;
                }
             }
          }

          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 0;
          int numflag = 0;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 8)
          {
             METIS_PartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  NULL,
                  NULL,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_PartGraphKway_w(
                  &n,
                  xadj,
                  adjacency,
                  NULL,
                  NULL,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 1;
          int numflag = 0;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 8)
          {
             METIS_PartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  NULL,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_PartGraphKway_w(
                  &n,
                  xadj,
                  adjacency,
                  NULL,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * Ideally all the partitions should be equally weighted.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 2;
          int numflag = 0;
          float *tpwgts;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          tpwgts = (float*)malloc(
                           sizeof(float)
                           *
                           p
          );

          if (tpwgts == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             tpwgts[i] = 1/p;
          }

          if (p <= 8)
          {
             METIS_WPartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  NULL,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_WPartGraphVKway_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  NULL,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);
          free(tpwgts);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * Ideally all the partitions should be equally weighted.
        * There is a upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 3;
          int numflag = 0;
          float *tpwgts;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          tpwgts = (float*)malloc(
                           sizeof(float)
                           *
                           p
          );

          if (tpwgts == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
             tpwgts[i] = 1/p;
          }

          if (p <= 8)
          {
             METIS_WPartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_WPartGraphVKway_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);
          free(tpwgts);

          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * Ideally all the partitions should be equally weighted.
        * There is a upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * Speeds of processors are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 2;
          int numflag = 0;
          float *tpwgts;
          float sumf = 0.0;
          double tspeed= 0.0;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          tpwgts = (float*)malloc(
                           sizeof(float)
                           *
                           p
          );

          if (tpwgts == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumf += vwgt[i];
          }

          for (i = 0; i < p; i++)
          {
             tspeed = speeds[i];
          }

          for (i = 0; i < p; i++)
          {
             tpwgts[i] = (speeds[i]/tspeed)*sumf;
          }

          if (p <= 8)
          {
             METIS_WPartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_WPartGraphVKway_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);
          free(tpwgts);

          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
      
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          int nparts = p;
          int wgtflag = 3;
          int numflag = 0;
          float *tpwgts;
          float sumf = 0.0;
          double tspeed= 0.0;
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               5
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          tpwgts = (float*)malloc(
                           sizeof(float)
                           *
                           p
          );

          if (tpwgts == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < n; i++)
          {
             sumf += vwgt[i];
          }

          for (i = 0; i < p; i++)
          {
             tspeed = speeds[i];
          }

          for (i = 0; i < p; i++)
          {
             tpwgts[i] = (speeds[i]/tspeed)*sumf;
          }

          if (p <= 8)
          {
             METIS_WPartGraphRecursive_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }
          else
          {
             METIS_WPartGraphVKway_w(
                  &n,
                  xadj,
                  adjacency,
                  vwgt,
                  adjwgt,
                  &wgtflag,
                  &numflag,                 
                  &nparts,
                  tpwgts,
                  options,
                  edgecut,
                  vp
             );
          }

          free(options);
          free(tpwgts);

          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       printf("Parameters are erroneous\n");
       return HMPI_ERR_PARTITION_GRAPH;
   }

   /*-----------------------------------------------------*/

   int HMPI_Partition_bipartite_graph(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int n,
	   int m,
	   const int *vtype,
           const int *vwgt,
	   const int *xadj,
	   const int *adjacency,
	   const int *adjwgt,
           int type_of_partitioning,
	   int *vp,
	   int *edgecut
   )
   {
       int i, j, rc;

       /*
        * Use the partitioning interface for Set when 
        * edges have no weights. And there is no 
        * adjacency matrix. The two disjoint subsets are 
        * not considered separately.
        */
       if ((vwgt == NULL)
           && (xadj == NULL)
           && (adjacency == NULL)
           && (adjwgt == NULL)
           && (type_of_partitioning == PARTITION_OTHER
          )
       )
       {
          rc = HMPI_Partition_set(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 n,
                 NULL,
                 0,
                 1,
                 -1,
                 NULL,
                 NULL,
                 vp
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          *edgecut = 0;

          for (i = 0; i < n; i++)
          {
             int owner_processor = vp[i];

             for (j = xadj[i]; j < xadj[i+1]; j++)
             {
                int neighbor = adjacency[j];
 
                if (vp[neighbor] != owner_processor)
                {
                   (*edgecut)++;
                }
             }
          }

          return HMPI_OK;
       }
       
       /*
        * Use the partitioning interface for Set when 
        * edges have no weights. And there is no 
        * adjacency matrix. The two disjoint subsets are 
        * considered separately.
        */
       if ((vwgt == NULL)
           && (xadj == NULL)
           && (adjacency == NULL)
           && (adjwgt == NULL)
           && (type_of_partitioning == PARTITION_SUBSET
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       if (type_of_partitioning == PARTITION_OTHER)
       {
          return HMPI_Partition_graph(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 n,
                 m,
                 vwgt,
                 xadj,
                 adjacency,
                 adjwgt,
                 vp,
                 edgecut
          );
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * The number of vertices in each partition in each subset
        * should be the same.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * The number of vertices in each partition in each subset
        * should be the same.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * The number of vertices in each partition in each subset
        * should be the same.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * The number of vertices in each partition in each subset
        * should be the same.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * Ideally all the partitions in each subset should 
        * be equally weighted.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is a upper bound on the number of elements
        * stored by each processor.
        * Ideally all the partitions in each subset should 
        * be equally weighted.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Ideally all the partitions in each subset should 
        * be equally weighted.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is a upper bound on the number of elements
        * stored by each processor.
        * Ideally all the partitions in each subset should 
        * be equally weighted.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each subset 
        *     is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each subset 
        *     is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each subset 
        *     is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each subset 
        *     is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * Speeds of processors are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each subset 
        *     is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     in each subset is proportional to the speed 
        *     of the processor owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each
        *     subset is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition in each
        *     subset is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition in each 
        *     subset is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition 
        *     in each subset is proportional to the speed of 
        *     the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition in each
        *     subset is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition in each
        *     subset is proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
      
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning 
        *     that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor
        *     owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (adjwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       printf("Parameters are erroneous\n");
       return HMPI_ERR_PARTITION_GRAPH;
   }

   /*-----------------------------------------------------*/
   
   /*
    * There are two methods provided by hMETIS to partition
    * a hypergraph.
    * One using multilevel recursive bisection and the other
    * using multilevel k-way partitioning
    * hMETIS provides options to define the quality criteria 
    * that can be used for partitioning.
    * We use default options for the present but however when
    * hMETIS is integrated with HMPI, application programmers 
    * will be allowed to choose the options.
    * 
    */
   int HMPI_Partition_hypergraph(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int nv,
	   int nedges,
	   const int *vwgt,
	   const int *hptr,
	   const int *hind,
	   const int *hwgt,
	   int *vp,
	   int *edgecut
   )
   {
       int i, j, rc;

       /*
        * Use the partitioning interface for Set when 
        * edges have no weights.
        */
       if ((vwgt == NULL)
           && (hptr == NULL)
           && (hind == NULL)
           && (hwgt == NULL
          )
       )
       {
          rc = HMPI_Partition_set(
                 p,
                 pn,
                 speeds,
                 psizes,
                 mlimits,
                 nv,
                 NULL,
                 0,
                 1,
                 -1,
                 NULL,
                 NULL,
                 vp
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          *edgecut = 0;

          for (i = 0; i < nedges; i++)
          {
             int owner_processor, ix = 0;
             int *vertex = (int*)malloc(
                                 sizeof(int)
                                 *
                                 (hptr[i+1] - hptr[i])
             );

             if (vertex == NULL)
             {
                return MPC_ERR_NOMEM;
             }
 
             for (j = hptr[i]; j < hptr[i+1]; j++)
             {
                vertex[ix++] = hind[j];
             }

             owner_processor = vp[vertex[0]];
 
             for (j = 1; j < (hptr[i+1] - hptr[i]); j++)
             {
                if (vp[vertex[j]] != owner_processor)
                {
                   (*edgecut)++;
                }
             }

             free(vertex);
          }

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               9
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 16)
          {
             HMETIS_PartRecursive_w(
                  nv,
                  nedges,
                  NULL,
                  hptr,
                  hind,
                  NULL,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }
          else
          {
             HMETIS_PartKway_w(
                  nv,
                  nedges,
                  NULL,
                  hptr,
                  hind,
                  NULL,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }

          free(options);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               9
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 16)
          {
             HMETIS_PartRecursive_w(
                  nv,
                  nedges,
                  NULL,
                  hptr,
                  hind,
                  hwgt,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }
          else
          {
             HMETIS_PartKway_w(
                  nv,
                  nedges,
                  NULL,
                  hptr,
                  hind,
                  NULL,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }

          free(options);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * Ideally all the partitions should be equally weighted.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               9
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 16)
          {
             HMETIS_PartRecursive_w(
                  nv,
                  nedges,
                  vwgt,
                  hptr,
                  hind,
                  NULL,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }
          else
          {
             HMETIS_PartKway_w(
                  nv,
                  nedges,
                  vwgt,
                  hptr,
                  hind,
                  NULL,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }

          free(options);

          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the total number of edges that straddle
        * partitions.
        * Ideally all the partitions should be equally weighted.
        * There is a upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          int *options = (int*)malloc(
                               sizeof(int)
                               *
                               9
          );

          if (options == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          options[0] = 0;

          if (p <= 16)
          {
             HMETIS_PartRecursive_w(
                  nv,
                  nedges,
                  vwgt,
                  hptr,
                  hind,
                  hwgt,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }
          else
          {
             HMETIS_PartKway_w(
                  nv,
                  nedges,
                  vwgt,
                  hptr,
                  hind,
                  hwgt,
                  p,
                  HMPI_UBFACTOR,
                  options,
                  vp,
                  edgecut
             );
          }

          free(options);

          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that the edgecut is minimal
        * Edgecut is the sum of the weight of the straddling edges.
        * Ideally all the partitions should be equally weighted.
        * There is a upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds == NULL)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal. Edgecut is the total number 
        *     of edges that straddle partitions.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is proportional
        *     to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * Speeds of processors are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The number of vertices in each partition is 
        *     proportional to the speed of the processor 
        *     owning that partition.
        * (b) The edgecut is minimal. Edgecut is the sum 
        *     of the weights of the straddling edges.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt == NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        * Speeds of processors are functions of problem size.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt == NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
      
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }
       
       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn == 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is no upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits == NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       /*
        * Partition the graph such that 
        * (a) The sum of weights of vertices in each partition is 
        *     proportional to the speed of the processor owning that partition.
        * (b) The edgecut is minimal.
        * speeds are functions of problem size.
        * There is an upper bound on the number of elements
        * stored by each processor.
        */
       if ((speeds != NULL)
           && (pn > 1)
           && (mlimits != NULL)
           && (vwgt != NULL)
           && (hwgt != NULL
          )
       )
       {
          printf("Implementation currently not available\n");
          return HMPI_OK;
       }

       printf("Parameters are erroneous\n");
       return HMPI_ERR_PARTITION_GRAPH;
   }

   /*-----------------------------------------------------*/
