
   /************************************************/
   /* hmpi - Implementation of helper functions of */
   /*        the HMPI library                      */
   /*                                              */
   /* Revision history                             */
   /* 01-02-2002  --      Initial version          */
   /* 04-10-2007  --      Added additional optimal */
   /*                     parameters to network    */
   /************************************************/

   #include <hmpi.h>
   #include <stdio.h>
   #include <stdlib.h>

   /*-----------------------------------------------------*/

   /*
    * Fetch the Net structure from the map.
    * The key is the network identifier.
    */
   HMPI_Net*
   HMPI_Get_net_from_map
   (
       const HMPI_Group* pgroup_t,
       int* index
   )
   {
       int i;
       unsigned char is_entry_found = 0;

       for (i = 0; i < _hmpi_num_of_nets; i++)
       {
           if (_nids[i] == *pgroup_t)
           {
              is_entry_found = 1;
              break;
           }
       }

       if (!is_entry_found)
       {
          return NULL;
       }

       *index  = i;
       return _hmpi_nets[i];
   }

   /*-----------------------------------------------------*/

   /*
    * Basic checks on the network identifier.
    */
   int
   HMPI_Test_net_id
   (
       const HMPI_Group* pgroup_t
   )
   {
       if (pgroup_t == NULL)
       {
          return -1;
       }

       if (*pgroup_t == HMPI_UNDEFINED_NID
           || *pgroup_t == 0
       )
       {
          return -1;
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * Free nodes wait for commands from Dispatcher here which
    * is the waiting point. Following is the implementation of 
    * MPC_Offer:
    *
    * If the command is MPC_HIRED, the node is hired in a 
    * network region given by the name of the network region 
    * and as a result they become busy. In the implementation,
    * the name of a network region is constant given by
    * HMPI_STATIC_NET_NAME.
    *
    * If the command is MPC_SKIP, they are not hired and all 
    * the nodes that get this command become the new free_comm. 
    *
    * If the command is MPC_FIRED, that means the node has been 
    * fired as a result of MPC_Net_free operation and all the 
    * nodes including dispatcher become the new free_comm.
    *
    * If the command is MPC_OUT, the nodes leave MPC_Offer and 
    * reach the epilogue of the waiting point.
    *
    * This should be probably provided as a synchronisation
    * function to the User.
    */
   int
   HMPI_Offer_free_processes_to_net
   (
       MPC_Name name,
       MPC_Net* net
   )
   {
       if (HMPI_Debug_flag)
       {
          HMPI_Printf(
                     "HMPI===> HMPI_OFFER: Free processes for offer:"
                     " Global rank is %d\n",
                     HMPI_Group_rank(HMPI_COMM_WORLD_GROUP)
          );
       }

       {
           MPC_Name MPC_names[1];
           MPC_Net* MPC_nets[1];

           {
               MPC_names[0] = name;
           }

           {
               MPC_nets[0] = net;
           }

           {
               int result = MPC_Offer(
                                &MPC_command,
                                MPC_names,
                                MPC_nets,
                                1
               );

               if (result != MPC_OK)
               {
                  return result;
               }
           }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/
   /*
    * Initialise an abstract network structure.
    * TBD: The attribute MPC_BENCH/MPC_NOT_BENCH will 
    * probably come from Net type declaration through a 
    * topological function.
    */
   int
   HMPI_Init_net
   (
       HMPI_Net *pnet
   )
   {
       MPC_Net* net = &(pnet->_net);

       pnet->numcoord = -1;
       pnet->coord = NULL;
       pnet->time = -1.;

       net->pweb = MPC_NULL_WEB;
       net->type = NULL;
       net->count = 0;
       net->params = NULL;
       net->is_static = MPC_STATIC_NET;
       net->bench = MPC_BENCH;
       net->rank = MPC_UNINITED_RANK;
       net->power = MPC_INIT_POWER;
       net->leader = 0;
       net->oldroot = MPC_MULTI_ROOT;
       net->num_of_links = 0;
       net->coord = NULL;
       net->supernet = NULL;
       net->mapping = NULL;
       net->temp_topo = NULL;
       net->subnet_list = NULL;
       net->burst_list = NULL;
       net->subnet_next = NULL;
       net->burst_next = NULL;
       net->next = NULL;
       net->nodes = NULL;
       net->links = NULL;

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * Get the rank of the process given its coordinates.
    * This is called by HMPI_Rank interface function and 
    * should be made part of the implementation of HMPI_Rank.
    */
   int
   HMPI_Process_rank_given_coordinates
   (
       MPC_Net* net,
       const int* coord
   )
   {
       if ((net->supernet) == NULL)
       {
          return (net->type->coord2num)
                 (
                    coord,
                    net->params,
                    net->power,
                    &(net->nodes),
                    &(net->links)
          );

       }
       
       {
          int supernumber = HMPI_Process_rank_given_coordinates(
                                net->supernet,
                                coord
          );

          int i;
          for (i = 0; i<net->power; i++)
          {
              if ((net->mapping[i]) == supernumber)
              {
                 return i;
              }
          }
       }

       return MPC_ERR_INVALIDMAPPING;
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Translate_ranks
   (
       const HMPI_Group* nid1,
       int number,
       int *ranks1,
       const HMPI_Group* nid2,
       int *ranks2
   )
   {
       int rc;
       MPI_Group group1;
       MPI_Group group2;

       MPI_Comm* comm1 = (MPI_Comm*)HMPI_Get_comm(
                                        nid1
       );

       MPI_Comm* comm2 = (MPI_Comm*)HMPI_Get_comm(
                                        nid2
       );

       rc = MPI_Comm_group(
               *comm1,
               &group1
       );

       if (rc != MPI_SUCCESS)
       { 
          return rc;
       }

       rc = MPI_Comm_group(
               *comm2,
               &group2
       );

       if (rc != MPI_SUCCESS)
       {
          return rc;
       }

       rc =  MPI_Group_translate_ranks(
                group1,
                number,
                ranks1,
                group2,
                ranks2
       );

       if (rc != MPI_SUCCESS)
       {
          return rc;
       }

       rc = MPI_Group_free(&group1);

       if (rc != MPI_SUCCESS)
       {
          return rc;
       }

       rc = MPI_Group_free(&group2);

       if (rc != MPI_SUCCESS)
       {
          return rc;
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/


