
   /************************************************/
   /* hmpi - Implementation of Interfaces of       */
   /*        the HMPI library                      */
   /*                                              */
   /* Revision history                             */
   /* 01-02-2002  --      Initial version          */
   /* 04-10-2007  --      Added HMPI_Group_timeof  */
   /************************************************/

   #include <hmpi.h>
   #include <hmpi_internal_funcs.h>

   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>

   int _hmpi_net_base;
   unsigned int _hmpi_num_of_nets;

   HMPI_Group HMPI_global;
   HMPI_Group HMPI_proc;
   HMPI_Group HMPI_host;
   HMPI_Group HMPI_free;

   HMPI_Net HMPI_Net_global;
   HMPI_Net HMPI_Net_proc;
   HMPI_Net HMPI_Net_host;
   HMPI_Net HMPI_Net_free;

   HMPI_Group* _nids;
   HMPI_Net** _hmpi_nets;

   MPI_Comm HMPI_COMM_WORLD;
   double *HMPI_Comm_world_performances = NULL; 
   double *HMPI_Comp_world_performances = NULL; 

   int HMPI_Debug_flag = 0;

   /*----------------------------------------------------*/ 
   /*
    * Error messages. A neat way to initialize is to move 
    * it to HMPI_Init.
    */
   static const char*
   HMPI_Error_messages[HMPI_LAST_ERROR - MPC_ERR_LAST] =
          {
                "HMPI_OK",

                "Process is not member of the "
                "group",

                "Null group represented "
                "by the group identifier; the group"
                " must have been destroyed or not "
                "created",

                "Internal error; the group "     
                "represented by the group"
                " identifier does not exist",

                "Invalid group identifier",

                "Invalid input parameters",

                "Invalid operand type specified in "
                "creation of sub-group; "
                "Has to be VECTOR or SCALAR or EXPR",

                "Invalid operator specified in "
                "creation of sub-group",

                "error in the conditions passed",

                "Process calling this function "
                "not the host ",                    

                "Processes calling this function "
                "are either not host or not free ",

                "HMPI has encountered an internal"
                " error; Please report the issue to "
                " HMPI helpdesk",

                "Problems partitioning the set",

                "Bounds on the number of elements "
                "that can be stored by the processors"
                " exceeded", 

                "Invalid dimensions", 

                "Problems partitioning the matrix", 

                "Partition does not exist",

                "Problems partitioning the graph",

                "Problems partitioning the tree",

                "Type of metric unknown",

                "Last error code"
   };

   /*-----------------------------------------------------*/

   /*
    * HMPI_Init() -- Initialize HMPI runtime.
    *
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int 
   HMPI_Init
   (
       int *pargc,
       char*** pargv
   )
   {
       int i, gsize, rc;

       rc = MPC_Init(
               pargc,
               pargv,
               MPC_The_init_strategy,
               MPC_Trivial_error_print,
               MPC_The_get_env,
               MPC_The_strategy,
               MPC_The_topology_data,
               MPC_The_processors_info
       );

       if (rc != MPC_OK)
       {
          return rc;
       }

       HMPI_Printf = MPC_Printf;

       /*
        * Allocate the map.
        */
       _nids =  NULL;
       _hmpi_nets = NULL;
       _hmpi_num_of_nets = 0;

       _nids = (HMPI_Group*)malloc(
                            sizeof(HMPI_Group)
                            *
                            HMPI_NUM_OF_NETS
       );

       if (_nids == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       /*
        * Probably this initialization is not necessary.
        */
       {
          int i;
          for (i = 0; i < HMPI_NUM_OF_NETS; i++)
          {
              _nids[i] = HMPI_UNDEFINED_NID;
          }
       }

       _hmpi_nets = (HMPI_Net**)malloc(
                         sizeof(HMPI_Net*)
                         *
                         HMPI_NUM_OF_NETS
       );

       if (_hmpi_nets == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       /*
        * Meaningful allocation or names to
        * be given for pre-defined groups.
        *
        * Host+free nodes are part of global abstract
        * network HMPI_GLOBAL.
        *
        * Dispatcher processes requests and quits in 
        * MPC_Init.
        */
       HMPI_global = HMPI_GLOBAL_NET_NAME;
       HMPI_Net_global._name = HMPI_global;
       HMPI_Net_global._net = MPC_Net_global;

       _nids[0] = HMPI_global;

       _hmpi_nets[0] = (HMPI_Net*)malloc(
                            sizeof(HMPI_Net)
       );

       if (_hmpi_nets[0] == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       *(_hmpi_nets[0]) = HMPI_Net_global;

       HMPI_COMM_WORLD = *(MPI_Comm*)(MPC_Net_global.pweb);

       ++_hmpi_num_of_nets;

       HMPI_proc = HMPI_UNDEFINED_NID;
       HMPI_host = HMPI_UNDEFINED_NID;
       HMPI_free = HMPI_UNDEFINED_NID;
        
       /*
        * Only Host is part of abstract
        * network HMPI_HOST.
        * All other processes need not know about
        * this abstract network.
        */
       if (MPC_Is_host())
       {
          HMPI_host = HMPI_HOST_NET_NAME;

          HMPI_Net_host._name = HMPI_host;
          HMPI_Net_host._net = MPC_Net_host;

          _nids[_hmpi_num_of_nets] = HMPI_host;

          _hmpi_nets[_hmpi_num_of_nets] = (HMPI_Net*)malloc(
                              sizeof(HMPI_Net)
          );

          if (_hmpi_nets[_hmpi_num_of_nets] == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          *(_hmpi_nets[_hmpi_num_of_nets]) = HMPI_Net_host;

          ++_hmpi_num_of_nets;
       }

       /*
        * All free nodes are part of abstract
        * network HMPI_FREE.
        * Host is used as a parent for the creation
        * of first abstract network and it is also not
        * a free process.
        */
       if (HMPI_Is_free())
       {
          HMPI_free = HMPI_FREE_NET_NAME;

          HMPI_Net_free._name = HMPI_free;
          HMPI_Net_free._net = MPC_Net_free_procs;

          _nids[_hmpi_num_of_nets] = HMPI_free;

          _hmpi_nets[_hmpi_num_of_nets] = (HMPI_Net*)malloc(
                                                     sizeof(HMPI_Net)
          );

          if (_hmpi_nets[_hmpi_num_of_nets] == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          *(_hmpi_nets[_hmpi_num_of_nets]) = HMPI_Net_free;

          ++_hmpi_num_of_nets;
       }

       /*
        * Only one process per processor is part of 
        * Group HMPI_PROC_WORLD_GROUP.
        */
       if (MPC_Is_member(&MPC_Net_recon))
       {
          HMPI_proc = HMPI_PROC_NET_NAME;

          HMPI_Net_proc._name = HMPI_proc;
          HMPI_Net_proc._net = MPC_Net_recon;

          _nids[_hmpi_num_of_nets] = HMPI_proc;

          _hmpi_nets[_hmpi_num_of_nets] = (HMPI_Net*)malloc(
                                                     sizeof(HMPI_Net)
          );

          if (_hmpi_nets[_hmpi_num_of_nets] == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          *(_hmpi_nets[_hmpi_num_of_nets]) = HMPI_Net_proc;

          ++_hmpi_num_of_nets;
       }

       /*
        * The base for assigning network IDs.
        * A meaningful initial number or allocation 
        * needs to be used.
        */
       _hmpi_net_base = HMPI_GROUP_BASE_ID;

       /*
        * Save the performances of the processes
        */
       gsize = HMPI_Group_size(
                   HMPI_COMM_WORLD_GROUP
       );

       HMPI_Comm_world_performances = (double*)malloc(
                                               sizeof(double)
                                               *
                                               gsize
       );

       if (HMPI_Comm_world_performances == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = MPC_Processes_static_info(
               HMPI_Comm_world_performances
       );

       if (rc != MPC_OK)
       {
          return HMPI_ERR_INTERNAL;
       }

       if (HMPI_Is_member(HMPI_COMP_WORLD_GROUP))
       {
          int csize = HMPI_Group_size(
                          HMPI_COMP_WORLD_GROUP
          );

          HMPI_Comp_world_performances = (double*)malloc(
                                                  sizeof(double)
                                                  *
                                                  csize
          );

          if (HMPI_Comp_world_performances == NULL)
          {
             return MPC_ERR_NOMEM;
          }
       }

       if (HMPI_Debug_flag)
       {
          int grank;
          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> HMPI_INIT: MPI_COMM_WORLD rank(%d)-> HMPI runtime initialized\n",
              grank
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Finalize() -- Finalize HMPI runtime.
    *
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Finalize
   (
       int exitcode
   )
   {
       if (HMPI_Debug_flag)
       {
          int grank;
          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> HMPI_FINALIZE: MPI_COMM_WORLD rank(%d)-> Finalizing HMPI runtime\n",
              grank
          );
       }

       if (HMPI_Is_member(HMPI_COMP_WORLD_GROUP))
       {
          if (HMPI_Comp_world_performances != NULL)
          {
             free(HMPI_Comp_world_performances);
          }
       }

       /*
        * Free the abstract networks HMPI_FREE, HMPI_GLOBAL
        * HMPI_HOST.
        * For all free nodes, the 2 pre-defined static abstract
        * networks are HMPI_GLOBAL and HMPI_FREE.
        * For host, the 2 pre-defined static abstract
        * networks are HMPI_GLOBAL and HMPI_HOST.
        *
        * For processes which are members of HMPI_PROC_WORLD_GROUP,
        * there is a third network.
        */
       free(_hmpi_nets[0]);
       free(_hmpi_nets[1]);

       if (MPC_Is_member(&MPC_Net_recon))
       {
           free(_hmpi_nets[2]);
       }

       /*
        * Free the map.
        */
       free(_hmpi_nets);
       free(_nids);

       /*
        * Free the performances
        */
       free(HMPI_Comm_world_performances);

       if (HMPI_Debug_flag)
       {
          int grank;
          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> MPC_Exit: MPI_COMM_WORLD rank(%d)-> Finalizing MPC runtime\n",
              grank
          );
       }

       {
          int result = MPC_Exit(exitcode);      
            
          if (result != MPC_OK)
          {
             return result;
          }
       }

       return HMPI_OK;
   }   

   /*-----------------------------------------------------*/

   /*
    * HMPI_Abort() -- Aborts all the tasks.
    *
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Abort
   (
       int exitcode
   )
   {
       if (HMPI_Debug_flag)
       {
          int grank;

          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf("HMPI===> HMPI_ABORT: MPI_COMM_WORLD rank(%d)-> Aborting HMPI runtime\n");
       }

       {
          int result = MPC_Abort(exitcode);      
            
          if (result != MPC_OK)
          {
             return result;
          }
       }
       
       return HMPI_OK;
   }   

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_create() -- Create a HMPI group of processes.
    *
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_create
   (
       HMPI_Group* pgroup_t,
       const HMPI_Model* net_type,
       int* net_parameters,
       int param_count
   )
   {
       int i, rc;

       HMPI_Net* pnet;
       MPC_Net* net;
       
       /* 
        * The name of the abstract network is a constant.
        * Net identifier pgroup_t calculated will be unique for
        * each process and is only process specific 
        * but the name of the network region can be constant.
        * The name is used to create a sub-group representing
        * this abstract network using MPI_Comm_split, the name 
        * being used as value to parameter 'color'
        */
       MPC_Name name = HMPI_STATIC_NET_NAME;

       {
          pnet = (HMPI_Net*)malloc(
                              sizeof(HMPI_Net)
          );

          if (pnet == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          {
             int result = HMPI_Init_net(
                              pnet
             );

             if (result != HMPI_OK)
             {
                return result;
             }
          }
       }

       /*
        * Somehow, the cordinates and the network type need 
        * be filled before offering the free processes.
        * This is because topological functions are called in
        * MPC_Node_hire when nodes are hired.
        */
       net = &(pnet->_net);
       net->type = (HMPI_Model*)net_type;
       net->coord = (int*)malloc(
                          sizeof(int)
                          *
                          net->type->numcoord
       );

       if (net->coord == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       if (HMPI_Is_free())
       {
          int result = HMPI_Offer_free_processes_to_net(
                           name,
                           net
          );

          if (result != HMPI_OK)
          {
             return result;
          }

          /*
           * For processes that are hired, we fill the 
           * abstract network map.
           */
          if (!HMPI_Is_free())
          {
             if (net_parameters != NULL)
             {
                for (i = 0; i < net->count; i++)
                {
                   net_parameters[i] = net->params[i];
                }
             }

             /*
              * Increment the number of nets.
              * TBD:
              * Some meaningful allocation of network
              * identifiers should be used.
              */
             _nids[_hmpi_num_of_nets] = _hmpi_net_base
                                        + 
                                        _hmpi_num_of_nets;

             _hmpi_nets[_hmpi_num_of_nets] = pnet;
             pnet->_name = _nids[_hmpi_num_of_nets];
             ++_hmpi_num_of_nets;

             *pgroup_t = pnet->_name;

             if (HMPI_Debug_flag)
             {
                int grank;

                MPI_Comm_rank(
                   MPI_COMM_WORLD,
                   &grank
                );

                HMPI_Printf(
                    "HMPI===> HMPI_GROUP_CREATE: Group member MPI_COMM_WORLD rank(%d)-> Group(%d) created\n",
                    grank,
                    *pgroup_t
                );
             }

             return HMPI_OK;
          }

          /*
           * For processes that are not hired, we need to
           * deallocate the net structure allocated.
           */
          if (HMPI_Is_free())
          {
             free(net->coord);
             free(pnet);
             *pgroup_t = HMPI_UNDEFINED_NID;
          }

          return HMPI_OK;
       }

       /*
        * Processing for the parent of the abstract network
        * Host or member of any pre-existing abstract network
        * can become the parent.
        */
       if (HMPI_Debug_flag)
       {
          int grank;

          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
               "HMPI===> HMPI_GROUP_CREATE: Parent with MPI_COMM_WORLD rank(%d)-> Creating group\n",
               grank
          );
       }

       /*
        * Fill the network parameters and the parameter count.
        * Copy the parameters.
        */
       {
          net->params = (int*)malloc(
                              sizeof(int)
                              *
                              param_count
          );

          if (net->params == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < param_count; i++)
          {
             net->params[i] = net_parameters[i];
          }

          net->count = param_count;
       }

       {
          int result = MPC_Net_create(
                           name,
                           net
          );

          if (result != MPC_OK)
          {
             return result;
          }
       }

       /*
        * Update the map with the unique network identifier.
        * and the network identifier.
        */
       {
          _nids[_hmpi_num_of_nets] = _hmpi_net_base
                                     +
                                     _hmpi_num_of_nets;
          _hmpi_nets[_hmpi_num_of_nets] = pnet;

          pnet->_name = _nids[_hmpi_num_of_nets];
          *pgroup_t = _nids[_hmpi_num_of_nets];
          ++_hmpi_num_of_nets;
       }

       if (HMPI_Debug_flag)
       {
          int grank;

          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> HMPI_GROUP_CREATE: Parent with MPI_COMM_WORLD rank(%d)-> Group(%d) successfully created\n",
              grank,
              *pgroup_t
          );
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int 
   HMPI_Test_one_proc_i
   (
       int numcoord,
       const int *dim
   )
   {
       int i;

       for (i = 0; i < numcoord; i++)
       {
          if (dim[i] != 1)
          {
             return 0;
          }
       }

       return 1;
   }

   /*-----------------------------------------------------*/

   int 
   HMPI_Test_size_i
   (
       int gsize,
       int numcoord,
       const int *dim
   )
   {
       int i;
       int product = 1;

       for (i = 0; i < numcoord; i++)
       {
           product = product*dim[i];
       }

       if (product > gsize)
       {
          return 1;
       }

       return 0;
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Evaluate_model_recursively_i
   (
       int d,
       const HMPI_Model* model,
       HMPI_Heuristic_function hfunc,
       const int* model_parameters,
       int model_param_count,
       int gsize,
       int numcoord,
       double *mintime,
       int *dim,
       int *optimal_dim,
       int **optimal_speeds
   )
   {
       int i, y, rc;
       double time;

       int totalp;
       int *modelp, paramc;

       if (d > numcoord)
       {
          return HMPI_OK;
       }

       for (y = 1; y <= gsize; y++)
       {
           dim[d] = y;
 
           if ((d+1) == numcoord)
           {
              int global_size;
              int only_one_proc;
              int multiple_of_g, g;

              /*
               * We have a combination here
               * Test if the combination against
               * the number of processes available.
               */
              int greater = HMPI_Test_size_i(
                                gsize,
                                numcoord,
                                dim
              );

              if (greater)
              {
                 continue;
              }

              /*
               * Set the model parameters.
               * Evaluate model for the combination
               */
              for (totalp = 1, i = 0; i < numcoord; i++)
              {
                 totalp *= dim[i];
              }

              paramc = model_param_count + numcoord + totalp + totalp*totalp + totalp;

              modelp = (int*)malloc(
                             sizeof(int)
                             *
                             paramc
              );

              if (modelp == NULL)
              {
                 return MPC_ERR_NOMEM;
              }

              for (i = 0; i < model_param_count; i++)
              {
                 modelp[i] = model_parameters[i];
              }

              for (i = 0; i < numcoord; i++)
              {
                 modelp[model_param_count+i] = dim[i];
              }

              /*
               * Get the speeds
               */
              for (i = 0; i < totalp; i++)
              {
                 modelp[model_param_count+numcoord+i] = (int)HMPI_Comm_world_performances[i];
              }

              /* 
               * Initialize the communication link data
               */
              for (i = 0; i < totalp*totalp; i++)
              {
                 modelp[model_param_count+numcoord+totalp+i] = 0;
              }

              /* 
               * Initialize the communication link data flags
               */
              for (i = 0; i < totalp; i++)
              {
                 modelp[model_param_count+numcoord+totalp+totalp*totalp+i] = 0;
              }

              /*
               * User may have provided heuristic 
               * function to reduce the 
               * number of processor arrangements 
               * evaluated.
               */
              if (hfunc != NULL)
              {
                 rc = (*hfunc)(
                      numcoord,
                      dim,
                      modelp,
                      paramc
                 );

                 if (rc == 0)
                 {
                    free(modelp);
                    continue;
                 }
              }

              /*
               * Evaluate the combination
               */
              time = HMPI_Timeof(
                         model,
                         modelp,
                         paramc
              );

              if (HMPI_Debug_flag)
              {
                 printf("HMPI===> HMPI_GROUP_AUTO_CREATE: Process arrangement ");

                 for (i = 0; i < numcoord; i++)
                 {
                    printf("[%d]=%d ", i, dim[i]);
                 }

                 printf(" TIME=%0.9f\n", time);
              }

              free(modelp);

              //
              // We evaluate in order of increasing process arrangements
              // with increasing number of processes
              // We stick with the least possible
              if (time < (*mintime))
              {
                 /*
                  * Copy the combination to 
                  * optimal dimension
                  */
                 for (i = 0; i < numcoord; i++)
                 {
                     optimal_dim[i] = dim[i];
                 }

                 free(optimal_speeds[0]);

                 optimal_speeds[0] = (int*)malloc(
                                           sizeof(int)
                                           *
                                           totalp
                 );

                 if (optimal_speeds[0] == NULL)
                 {
                    return MPC_ERR_NOMEM;
                 }

                 for (i = 0; i < totalp; i++)
                 {
                    (*optimal_speeds)[i] = (int)HMPI_Comm_world_performances[i];
                 }
                 
                 *mintime = time;
              }
 
              continue;
           }

           rc = HMPI_Evaluate_model_recursively_i
                (
                    d+1,
                    model,
                    hfunc,
                    model_parameters,
                    model_param_count,
                    gsize,
                    numcoord,
                    mintime,
                    dim,
                    optimal_dim,
                    optimal_speeds
           );

           if (rc != HMPI_OK)
           {
              return rc;
           }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_auto_create() --
    * Create a HMPI group of processes with optimal number of 
    * processes. The HMPI runtime system detects the optimal 
    * number of processes.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_auto_create
   (
       HMPI_Group* gid,
       const HMPI_Model* model,
       const int* model_parameters,
       int model_param_count
   )
   {
       return HMPI_Group_auto_create_i(
              gid,
              model,
              NULL,
              model_parameters,
              model_param_count
       );
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_heuristic_auto_create() --
    * Create a HMPI group of processes with optimal number of 
    * processes. The HMPI runtime system detects the optimal 
    * number of processes. Application programmers can provide
    * heuristics to reduce the number of process arrangements 
    * evaluated.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_heuristic_auto_create
   (
       HMPI_Group* gid,
       const HMPI_Model* model,
       HMPI_Heuristic_function hfunc,
       const int* model_parameters,
       int model_param_count
   )
   {
       return HMPI_Group_auto_create_i(
              gid,
              model,
              hfunc,
              model_parameters,
              model_param_count
       );
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Group_auto_create_i
   (
       HMPI_Group* gid,
       const HMPI_Model* model,
       HMPI_Heuristic_function hfunc,
       const int* model_parameters,
       int model_param_count
   )
   {
       int i, rc;
       int numcoord, *dim;
       double mintime = DBL_MAX;
       int gsize;

       int totalp;
       int *modelp, paramc;
       int *optimal_proc_arrangement, **optimal_speeds;

       if (HMPI_Is_member(HMPI_FREE_GROUP))
       {
          rc =  HMPI_Group_create(
                    gid,
                    model,
                    NULL,
                    0
          );

          if (rc != HMPI_OK)
          {
             return rc;
          }

          return HMPI_OK;
       }

       /*
        * Number of processes that are FREE and the 
        * parent of the group
        */
       gsize = HMPI_Group_size(HMPI_COMM_WORLD_GROUP);

       numcoord = model->numcoord;

       dim = (int*)malloc(
                   sizeof(int)
                   *
                   numcoord
       );

       if (dim == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       optimal_proc_arrangement = (int*)calloc(
                                        numcoord,
                                        sizeof(int)
       );

       if (optimal_proc_arrangement == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       optimal_speeds = (int**)malloc(
                              sizeof(int*)
       );

       if (optimal_speeds == NULL)
       {
          return MPC_ERR_NOMEM;
       }                            

       optimal_speeds[0] = (int*)malloc(
                                 sizeof(int)
       );

       if (optimal_speeds[0] == NULL)
       {
          return MPC_ERR_NOMEM;
       }                            

       /*
        * Evaluate for all possible combinations of 
        * coordinates.
        *
        * For example if the total number of processes
        * in MPI_COMM_WORLD are 3 and the number of dimensions
        * in the topology is 3, then the possible 
        * combinations are
        * 1 1 1
        * 1 1 2
        * 1 1 3
        * 1 2 1
        * 1 3 1
        * 2 1 1
        * 3 1 1
        */
       rc = HMPI_Evaluate_model_recursively_i
            (
                0,
                model,
                hfunc,
                model_parameters,
                model_param_count,
                gsize,
                numcoord,
                &mintime,
                dim,
                optimal_proc_arrangement,
                optimal_speeds
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       free(dim);

       /*
        * Get the speeds
        */
       for (totalp = 1, i = 0; i < numcoord; i++)
       {
           totalp *= optimal_proc_arrangement[i];
       }

       paramc = model_param_count + numcoord + totalp + totalp*totalp + totalp;

       modelp = (int*)malloc(
                      sizeof(int)
                      *
                      paramc
       );

       if (modelp == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < model_param_count; i++)
       {
           modelp[i] = model_parameters[i];
       }

       for (i = 0; i < numcoord; i++)
       {
           modelp[model_param_count+i] = optimal_proc_arrangement[i];
       }

       /*
        * Get the speeds
        */
       for (i = 0; i < totalp; i++)
       {
           modelp[model_param_count+numcoord+i] = (*optimal_speeds)[i];
       }

       /*
       *  Initilaize the communication link data
       */
       for (i = 0; i < totalp*totalp; i++)
       {
           modelp[model_param_count+numcoord+totalp+i] = 0;
       }

       /*
       *  Initilaize the communication link data flags
       */
       for (i = 0; i < totalp; i++)
       {
           modelp[model_param_count+numcoord+totalp+totalp*totalp+i] = 0;
       }

       /*
        * Now create the group with the optimal
        * number of processors.
        */
       rc = HMPI_Group_create(
                gid,
                model,
                modelp,
                paramc
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       /*
        * Fetch the Net structure from the map
        * using the network identifier.
        * Fill in the optimal parameters
        */
       {
          HMPI_Net* pnet;
          int index_of_net;

          pnet = HMPI_Get_net_from_map(
                     gid,
                     &index_of_net
          );

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }

          pnet->numcoord = numcoord;

          pnet->coord = (int*)malloc(
                              sizeof(int)
                              *
                              numcoord
          );

          if (pnet->coord == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < numcoord; i++)
          {
             pnet->coord[i] = optimal_proc_arrangement[i];
          }

          pnet->time = mintime;
       }

       free(optimal_proc_arrangement);
       free(modelp);
       free(optimal_speeds[0]);
       free(optimal_speeds);
       
       return HMPI_OK;
   }   

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_free() -- Free an HMPI group of processes.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_free
   (
       HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int is_subnet;
       int am_I_parent = HMPI_Is_parent(pgroup_t);
       int *net_parameters;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       if (HMPI_Debug_flag)
       {
          int grank;

          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> HMPI_GROUP_FREE: MPI_COMM_WORLD rank(%d)-> Destroying group(%d)\n",
              grank,
              *pgroup_t
          );
       } 

       /*
        * Fetch the Net structure from the map
        * using the network identifier.
        */
       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          );

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }

          net = &(pnet->_net);
       }

       {
          is_subnet = MPC_Is_subnet(net);
       }

       /*
        * Free the network region.
        */
       net_parameters = net->params;

       {
          int result = MPC_Net_free(net);

          if (result != MPC_OK)
          {
             return result;
          }
       }

       if (HMPI_Is_free())
       {
          int rc = MPC_Offer(
                      &MPC_command,
                      NULL,
                      NULL,
                      0
          );

          if (rc != MPC_OK)
          {
             return rc;
          }
       }

       /*
        * In the HMPI program model,
        * the host is part of all the  
        * HeteroMPI groups
        * Only the host process can notify
        * free processes to leave the waiting point
        */
       if (HMPI_Is_host())
       {
          int result = MPC_Notify_free_processes();

          if (result != MPC_OK)
          {
             return result;
          }
       }

       /*
       * Fixing Memory Leaks
       * rreddy BUG# 00003 MODULE MPC
       * Heterogeneous ScaLAPACK
       * Memory error due to reference of net params
       * in Net_free and so must be freed after
       */
       /*
        * Free the network parameters
        */
       if (!is_subnet)
       {
          if (am_I_parent)
          {
             if (net_parameters != NULL)
             {
                free(net_parameters);
             }
          }
       }

       if (HMPI_Debug_flag)
       {
          int grank;

          MPI_Comm_rank(
             MPI_COMM_WORLD,
             &grank
          );

          HMPI_Printf(
              "HMPI===> HMPI_GROUP_FREE: MPI_COMM_WORLD rank(%d)-> Group(%d) Destroyed\n",
              grank,
              *pgroup_t
          );
       }

       /*
        * Destroy the network structure.
        * For subnets, we don't allocate the coordinates.
        * Subnets use the parent coordinates.
        */
       {
          if (!is_subnet)
          {
             free(net->coord);
          }

          /*
          * Destroy the stored optimal paramaters
          */
          if (pnet->coord != NULL)
          {
             free(pnet->coord);
             pnet->coord = NULL;
          }

          free(pnet);
          _hmpi_nets[index_of_net] = NULL;
          _nids[index_of_net] = HMPI_UNDEFINED_NID;
          *pgroup_t = HMPI_UNDEFINED_NID;
       }

       /*
        * Now sending a message to dispatcher which would
        * send a message to host asking it to 
        * synchronize the arrival of free nodes
        * If the parent is host, do not do this.
        * In a basic HMPI program model, these 
        * synchronization messages are not required.
        */
       if (am_I_parent && (!HMPI_Is_host()))
       {
          int rc = MPC_Send_sync_message_to_dispatcher();
          if (rc != HMPI_OK)
          {
             return rc;
          }
       
          rc = MPC_Recv_sync_message_from_dispatcher();
          if (rc != HMPI_OK)
          {
             return rc;
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Subgroup_free() -- 
    * Free an HMPI group of processes. This function provides
    * functionality similar to the subnets of mpC.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Subgroup_create
   (
       HMPI_Group* subnet_id,
       const HMPI_Group* parent_net_id,
       int subnet_type,
       const char* conditions
   )
   {
       HMPI_Net* hmpi_parent_net;
       MPC_Net* supernet;
       HMPI_Net* hmpi_sub_net;
       MPC_Net* subnet;
       int index_of_net;

       if (!HMPI_Is_member(parent_net_id))
       {
          return HMPI_NOT_MEMBER;
       }

       hmpi_parent_net = HMPI_Get_net_from_map(
                             parent_net_id,
                             &index_of_net
       );

       if (hmpi_parent_net == NULL)
       {
          return HMPI_NULL_GROUP;
       }

       supernet = &(hmpi_parent_net->_net);

       {
          hmpi_sub_net = (HMPI_Net*)malloc(
                              sizeof(HMPI_Net)
          );

          if (hmpi_sub_net == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          {
             int result = HMPI_Init_net(
                              hmpi_sub_net
             );

             if (result != HMPI_OK)
             {
                return result;
             }
          }
       }

       subnet = &(hmpi_sub_net->_net);
       subnet->supernet = supernet;
 
       {
          MPC_Subnet_create_prologue((*supernet));

          for (; MPC_ind < supernet->power; MPC_ind++)
          {
              MPC_Coord_set(
                 supernet,
                 MPC_ind
              );

              {
                 int val = HMPI_Evaluate_conditions(
                               conditions,
                               supernet->coord
                 );

                 if ((val != 0) && (val != 1))
                 {
                    return val;
                 }

                 if (!val)
                 {
                    continue;
                 }
              }

              MPC_map[MPC_sub_pow] = HMPI_Process_rank_given_coordinates(
                                         supernet,
                                         supernet->coord
              );

              {
                 int i;
                 int result = 1;

                 for (i = 0; i < supernet->type->numcoord; i++)
                 {
                     if (MPC_co[i] != MPC_old_coord[i])
                     {
                        result = 0;
                        break;
                     }
                 }

                 if (result)
                 {
                    subnet->rank = MPC_sub_pow;
                 }
              }

              MPC_sub_pow++;
          }

          if (subnet_type == HMPI_FLEXIBLE)
          {
              MPC_Subnet_create_flexible_epilogue
              (
                  (*subnet),
                  (*supernet)
              );
          }
          else
          {
              MPC_Subnet_create_not_flexible_epilogue
              (
                  (*subnet),
                  (*supernet)
              );
          }
       }

       /*
        * Update the map with the unique network identifier.
        */
       {
          _nids[_hmpi_num_of_nets] = _hmpi_net_base
                                     +
                                     _hmpi_num_of_nets;
          _hmpi_nets[_hmpi_num_of_nets] = hmpi_sub_net;

          hmpi_sub_net->_name = _nids[_hmpi_num_of_nets];
          *subnet_id = _nids[_hmpi_num_of_nets];
          ++_hmpi_num_of_nets;
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_rank() -- Returns rank of the calling process.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_rank
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       pnet = HMPI_Get_net_from_map(
                  pgroup_t,
                  &index_of_net
       );

       if (pnet == NULL)
       {
          return HMPI_UNDEFINED;
       }

       net = &(pnet->_net);

       return net->rank;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_parent() -- Returns the rank of the parent 
    *                        of the group.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_parent
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return -1;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return -1;
          }
       }

       net = &(pnet->_net);

       return (net->type->parent)
              (
                 net->params,
                 net->power,
                 &(net->nodes),
                 &(net->links)
       );
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_size() -- Returns the number of processes 
    *                      in the group.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_size
   (
       const HMPI_Group* pgroup_t
   )
   {
       int size;
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;
       
       MPI_Comm *comm;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       /*
        * The groups  HMPI_COMM_WORLD_GROUP, HMPI_HOST_GROUP,
        * HMPI_PROC_WORLD_GROUP are static. That is the group structure 
        * does not change during the execution of the HMPI program.
        * So the net structure MPC_Net derived from the HMPI
        * group being static gives us current information.
        * Whereas the group HMPI_FREE_GROUP is dynamic.
        * The MPC specific information in the HMPI_FREE_GROUP is not 
        * updated as this group is specific to HMPI. The MPI_Comm 
        * specific part of the group is updated and so we get the
        * comm first to get the size.
        */
       if (pgroup_t == HMPI_FREE_GROUP)
       {
          comm = (MPI_Comm*)HMPI_Get_comm(
                                HMPI_FREE_GROUP
          );

          if (comm == NULL)
          {
             return HMPI_ERR_INTERNAL;
          }

          MPI_Comm_size(
             *comm,
             &size         
          );

          return size;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       if (net->power == MPC_INIT_POWER)
       {
          net->power = (net->type->power)
                       (
                          net->params,
                          net->power,
                          &(net->nodes),
                          &(net->links)
          );
       }

       return net->power;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Coordof() -- Returns the coordinates of the process 
    *                   with a specified rank.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Coordof
   (
       const HMPI_Group* pgroup_t,
       int node_rank,
       int* num_of_coordinates,
       int** coordinates
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       *num_of_coordinates = net->type->numcoord;

       if (coordinates == NULL)
       {
          return HMPI_OK;
       }

       coordinates[0] = (int*)malloc(
                              sizeof(int)
                              *
                              (*num_of_coordinates)
       );

       if (coordinates[0] == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       (net->type->num2coord)(
             node_rank,
             net->params,
             *coordinates,
             net->power, 
             &(net->nodes),
             &(net->links)
       );

       if (*coordinates == NULL)
       {
          return HMPI_INVALID_PARAMS;
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Group_process_ranks
   (
       const HMPI_Group* pgroup_t,
       int coord_count,
       const int* coordinates,
       int* rank_count,
       int** ranks
   )
   {
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       if (coordinates == NULL 
           || coord_count == 0
       )
       {
          return HMPI_INVALID_PARAMS;
       }

       *rank_count = 0;

       {
          int HMPI_ind = 0;
          int i;
          unsigned char is_member = 1;
          int num_of_coordinates; 
          int** process_coord = (int**)malloc(
                                      sizeof(int*)
          );

          if (process_coord == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          process_coord[0] = (int*)malloc(
                                   sizeof(int)
                                   * 
                                   HMPI_Group_topo_size(pgroup_t) 
          );

          if (process_coord[0] == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (HMPI_ind = 0; 
               HMPI_ind < HMPI_Group_size(pgroup_t); 
               HMPI_ind++
          )
          {
              int result = HMPI_Coordof(
                               pgroup_t,
                               HMPI_ind,
                               &num_of_coordinates,
                               process_coord
              );

              if (result != HMPI_OK)
              {
                 return result;
              }

              if (coord_count > num_of_coordinates)
              {
                 return HMPI_INVALID_PARAMS;
              }

              for (i = 0; i < coord_count; i++)
              {
                  if ((*process_coord)[i] != coordinates[i])
                  {
                     is_member = 0;
                     break;
                  }
              }

              if (is_member)
              {
                 (*ranks)[*rank_count] = HMPI_ind;
                 (*rank_count)++;
              }
          }

          free(process_coord[0]);
          free(process_coord);
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Coordof() -- Returns the number of coordinates that 
    *                   can specify a process in a group.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_topo_size
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return -1;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return -1;
          }
       }

       net = &(pnet->_net);

       return net->type->numcoord;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_topology() -- Returns the number of processes 
    * in the group in each dimension of the topology of the group.
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_topology
   (
       const HMPI_Group* pgroup_t,
       int* nd,
       int** d
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;
       int i;
       int gsize;
       int *coordinates;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       *nd = net->type->numcoord;

       if (d == NULL)
       {
          return HMPI_OK;
       }

       //
       // This is a nasty fix.
       // We get the last rank in the group.
       // We get the coordinates of this last rank.
       // To get the number of processes in each dimension,
       // we increment the coordinate by 1.
       gsize = HMPI_Group_size(pgroup_t);

       coordinates = (int*)malloc(
                           sizeof(int)
                           *
                           net->type->numcoord
       );

       if (coordinates == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       (net->type->num2coord)(
             gsize-1,
             net->params,
             coordinates,
             net->power, 
             &(net->nodes),
             &(net->links)
       );

       (*d) = (int*)malloc(
                    sizeof(int)
                    *
                    net->type->numcoord
       );

       if ((*d) == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < net->type->numcoord; i++)
       {
            (*d)[i] = coordinates[i] + 1;
       }

       free(coordinates);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Rank() -- Returns rank of the process with the 
    *                coordinates specified
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Rank
   (
       const HMPI_Group* pgroup_t,
       const int* coord
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       if (coord == NULL)
       {
          return HMPI_INVALID_PARAMS;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       return HMPI_Process_rank_given_coordinates(
                  net,
                  coord
       );
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_coordof() -- Returns the coordinates of the 
    *                         process
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   int
   HMPI_Group_coordof
   (
       const HMPI_Group* pgroup_t,
       int* num_of_coordinates,
       int** coordinates
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       *num_of_coordinates = net->type->numcoord;

       if (coordinates == NULL)
       {
          return HMPI_OK;
       }

       coordinates[0] = (int*)malloc(
                              sizeof(int)
                              *
                              (*num_of_coordinates)
       );

       if (coordinates[0] == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       (net->type->num2coord)(
             net->rank,
             net->params,
             *coordinates,
             net->power,
             &(net->nodes),
             &(net->links)
       );

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Is_host() -- Is the calling process the host?
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   unsigned char
   HMPI_Is_host()
   {
       return MPC_Is_host(); 
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Is_parent() -- Is the calling process the parent
    *                     process of the group?
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   unsigned char
   HMPI_Is_parent
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int parent_rank;
       int result;

       if (!HMPI_Is_member(pgroup_t))
       {
          return 0;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return 0;
       }

       pnet = HMPI_Get_net_from_map(
                  pgroup_t,
                  &index_of_net
       ); 

       if (pnet == NULL)
       {
          return 0;
       }

       net = &(pnet->_net);

       parent_rank = (net->type->parent)
                     (
                        net->params,
                        net->power,
                        &(net->nodes),
                        &(net->links)
       );

       return (parent_rank == net->rank);
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Is_member() -- Am I a member of the group?
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   unsigned char 
   HMPI_Is_member
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;

       int result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return 0;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return 0;
          }
       }

       net = &(pnet->_net);

       return MPC_Is_member(net); 
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Barrier() -- Barrier for the members of the group
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int HMPI_Barrier
   (
       const HMPI_Group* g
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;

       int result = HMPI_Test_net_id(g);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     g,
                     &index_of_net
          );

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       net = &(pnet->_net);

       return MPC_Local_barrier(net);

   }

   /*-----------------------------------------------------*/

   /*
    * Host is the only process that ensures the coordinated 
    * arrival of nodes at a particular waiting point. Only
    * host can signal the free nodes to exit the waiting point 
    * through dispatcher.
    * If a parent of any group wants to send a signal to the 
    * free nodes to exit the waiting point after the creation of the
    * group, then it can do so using the following:
    *
    * if (HMPI_Is_parent(...) || HMPI_Is_host())
    * {
    *   HMPI_Host_rendezvous(1);
    * }
    *
    * if (HMPI_Is_host())
    * {
    *    HMPI_Notify_free_processes();
    * }
    *
    * A whole group can rendezvous with the host
    *
    * if (HMPI_Is_member(&g) || HMPI_Is_host())
    * {
    *   HMPI_Host_rendezvous(HMPI_Group_size(&g));
    * }
    *
    * if (HMPI_Is_host())
    * {
    *    HMPI_Notify_free_processes();
    * }
    */
   int
   HMPI_Host_rendezvous(int count)
   {
       int rc;

       if (HMPI_Is_host())
       {
          rc = MPC_Recv_end(count);

          if (rc != MPC_OK)
          {
             return rc;
          }
       }

       if (!HMPI_Is_host())
       {
          rc = MPC_Send_end();

          if (rc != MPC_OK)
          {
             return rc;
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Notify_free_processes() -- 
    * Notify free processes to leave the waiting point
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   /*
    * Host sends a message to the dispatcher, which sends
    * a message to all unemployed nodes to leave the waiting
    * point.
    */
   int
   HMPI_Notify_free_processes()
   {
       if (!HMPI_Is_host())
       {
          return HMPI_NOT_HOST;
       }

       if (HMPI_Is_host())
       {
          int result = MPC_Host_out();

          if (result != MPC_OK)
          {
             return result;
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Sync_free_processes() -- 
    * Waiting point for free processes waiting for commands 
    * for group destruction
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Sync_free_processes()
   {
       int rc;

       if ((!HMPI_Is_host())
           && (!HMPI_Is_free()
          )
       )
       {
          return HMPI_NOT_HOST_AND_NOT_FREE;
       }

       if (HMPI_Is_free())
       {
          rc = MPC_Offer(
                  &MPC_command,
                  NULL,
                  NULL,
                  0
          );

          if (rc != MPC_OK)
          {
             return rc;
          }
       }

       if (HMPI_Is_host())
       {
          int rc = MPC_Recv_sync_message_from_dispatcher();

          if (rc != HMPI_OK)
          {
             return rc;
          }
       }

       /*
        * Once the host has received the message to go ahead
        * with coordinating the synchronization of free nodes,
        * it should sync the arrival of free nodes.
        */
       if (HMPI_Is_host())
       {
          int rc = HMPI_Notify_free_processes();

          if (rc != HMPI_OK)
          {
             return rc;
          }
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Is_free() -- 
    * Am I a member of the predefined group HMPI_FREE_GROUP?
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   unsigned char HMPI_Is_free()
   {
       return (!MPC_Is_busy());
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Get_comm() -- 
    * Returns an MPI communicator with communication group 
    * of MPI processes
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   const MPI_Comm*
   HMPI_Get_comm
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       int index_of_net;

       if (!HMPI_Is_member(pgroup_t))
       {
          return NULL;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return NULL;
          }
       }

       return (const MPI_Comm*)(pnet->_net.pweb);
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Recon() -- 
    * Updates the estimation of processor performances dynamically
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Recon
   (
       HMPI_Benchmark_function func,
       const void* input_p,
       int num_of_parameters,
       void* output_p
   )
   {
       int rc;
       double bench_time;

       if (!HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
       {
          return HMPI_NOT_MEMBER;
       }

       /* 
        * Execute the benchmark code
        */
       if (HMPI_Is_member(HMPI_PROC_WORLD_GROUP))
       {
          bench_time = MPC_Wtime();
      
          (*func)(
               input_p,
               num_of_parameters,
               output_p
          );

          bench_time = MPC_Wtime() - bench_time;
       }

       /*
        * HMPI_PROC_WORLD_GROUP or
        * MPC_Net_recon abstract network 
        * represents a process per computer/host.
        */
       if (HMPI_Is_member(HMPI_PROC_WORLD_GROUP))
       {
          if (HMPI_Debug_flag)
          {
             int grank;
             MPI_Comm_rank(
                MPI_COMM_WORLD,
                &grank
             );

             HMPI_Printf(
                "HMPI===> HMPI_RECON: MPI_COMM_WORLD rank [%d]-> "
                "Updating the processor runtime performance information\n",
                grank
             );
          }

          /*
          * What MPC_Update_computers_info does is it updates 
          * as well as returns the performances, which are 
          * computed in a MPC-specific manner by MPC
          */
          {
             int rc = MPC_Update_computers_info(
                         bench_time,
                         HMPI_Comp_world_performances
             );

             if (rc != MPC_OK)
             {
                return rc;
             }
          }

          if (HMPI_Debug_flag)
          {
             int grank;
             MPI_Comm_rank(
                MPI_COMM_WORLD,
                &grank
             );

             HMPI_Printf(
                 "HMPI===> HMPI_RECON: MPI_COMM_WORLD rank [%d]-> "
                 "Updated the processor runtime performance information\n",
                 grank
             );
          }
       }

       /*
        * The barrier is important here.
        * Only after the members of the group
        * HMPI_PROC_WORLD_GROUP update the performances,
        * we can query using the MPC_Processes_static_info
        */
       HMPI_Barrier(HMPI_COMM_WORLD_GROUP);
       
       rc = MPC_Processes_static_info(
               HMPI_Comm_world_performances
       );
       
       if (rc != MPC_OK)
       {
          return HMPI_ERR_INTERNAL;
       }

       MPC_Recon_epilogue();

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Get_number_of_free_processes() -- 
    * Returns the number of free processes available
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Get_number_of_free_processes()
   {
       int n, rc;

       if (!HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
       {
          return HMPI_NOT_MEMBER;
       }

       rc = MPC_Free_processes_info(&n);

       if (rc != HMPI_OK)
       {
          return -1;
       }

       return n;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Get_number_of_processors() -- 
    * Returns the number of physical processors of the underlying 
    * distributed memory machine
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Get_number_of_processors()
   {
       if (!HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
       {
          return HMPI_NOT_MEMBER;
       }

       return MPC_Get_number_of_processors();
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Get_processes_info() -- 
    * Returns the relative performances of the processes running 
    * on the physical processors of the underlying 
    * distributed memory machine
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Get_processes_info
   (
      double* relative_performances
   )
   {
      int rc, i, gsize;

      if (!HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
      {
         return HMPI_NOT_MEMBER;
      }

      rc = MPC_Processes_static_info(
              HMPI_Comm_world_performances
      );

      if (rc != MPC_OK)
      {
         return HMPI_ERR_INTERNAL;
      }

      gsize = HMPI_Group_size(HMPI_COMM_WORLD_GROUP);

      for (i = 0; i < gsize; i++)
      { 
         relative_performances[i] = HMPI_Comm_world_performances[i];
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Get_processors_info() -- 
    * Returns the relative performances of the physical 
    * processors of the underlying distributed memory machine
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Get_processors_info
   (
      double* relative_performances
   )
   {
      if (!HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
      {
         return HMPI_NOT_MEMBER;
      }

      MPC_Get_processors_info(
         NULL,
         relative_performances
      );

      if (relative_performances == NULL)
      {
         return HMPI_ERR_INTERNAL;
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_performances() -- 
    * Returns the relative performances of the processes in a group
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   int
   HMPI_Group_performances
   (
      const HMPI_Group* pgroup_t,
      double* relative_performances
   )
   {
      int rc;
      int My_global_rank;
      double My_processor_performance;
      MPI_Comm* pcomm;

      if (!HMPI_Is_member(pgroup_t))
      {
         return HMPI_NOT_MEMBER;
      }

      My_global_rank = HMPI_Group_rank(
                           HMPI_COMM_WORLD_GROUP
      );

      /*
       * HMPI_Comm_world_performances 
       * holds the recently estimated 
       * performances of all processors
       */
      My_processor_performance = HMPI_Comm_world_performances[My_global_rank];

      pcomm = (MPI_Comm*)HMPI_Get_comm(
                             pgroup_t
      );

      if (pcomm == NULL)
      {
         return HMPI_ERR_INTERNAL;
      }

      rc = MPI_Allgather(
              &My_processor_performance,
              1,
              MPI_DOUBLE,
              relative_performances,
              1,
              MPI_DOUBLE,
              *pcomm
      );

      if (rc != MPI_SUCCESS)
      {
         return rc;
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Timeof() -- 
    * Predict the total time of execution of the algorithm 
    * on the underlying hardware without its real execution
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */
   double
   HMPI_Timeof
   (
       const HMPI_Model* _model,
       const int* _net_parameters,
       int _net_param_count
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       double est;
       int rc;
       int *net_params;
       int param_count;
       int i;

       {
          pnet = (HMPI_Net*)malloc(
                              sizeof(HMPI_Net)
          );

          if (pnet == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          {
             int rc = HMPI_Init_net(
                          pnet
             );

             if (rc != HMPI_OK)
             {
                return rc;
             }
          }
       }

       /*
        * Fill the network type.
        */
       net = &(pnet->_net);
       net->type = (HMPI_Model*)_model;

       /*
        * Fill the network parameters and the parameter count.
        */
       net->params = (int*)_net_parameters;
       net->count = _net_param_count;

       if (net->power==MPC_INIT_POWER)
       {
          net->power = (net->type->power)
                       (
                          net->params,
                          net->power,
                          &(net->nodes),
                          &(net->links)
          );
       }

       /*
        * Copy the network parameters and parameter count.
        * This is because net estimation deletes the parameters.
        * This is because mpC generates code that alocates
        * net paramters on heap when timeof operator is used.
        */
       net_params = net->params;
       param_count = net->count;

       if (net_params != NULL)
       {
          net->params = (int*)malloc(
                              sizeof(int)
                              *
                              param_count
          );

          if (net->params == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < param_count; i++)
          {
              net->params[i] = net_params[i];
          }

          net->count = param_count;
       }

       est = MPC_Net_estimate(
                   HMPI_Group_parent(HMPI_COMM_WORLD_GROUP),
                   net
       );

       /*
       * Fixing Memory Leaks
       * rreddy BUG# 00003 MODULE MPC
       * Heterogeneous ScaLAPACK
       */
       if (net->links != NULL)
       {
          free(net->links);
          net->links = NULL;
       }

       free(pnet);

       return est;
   }

   /*-----------------------------------------------------*/
   /*
    * Tries to return HMPI error messages first, then 
    * MPC messages and then MPI messages if possible.
    */
   int
   HMPI_Strerror
   (
      int errnum,
      char* message
   )
   {
      int rc;
      int errlen;

      /*
       * MPC errors
       */
      if (errnum < MPC_ERR_LAST)
      {
         errnum = I2C(errnum);
         RC2STR(
           message, 
           errnum
         );
         return HMPI_OK;
      }

      /*
       * HMPI errors
       */
      if (errnum >= MPC_ERR_LAST
          && errnum <= HMPI_LAST_ERROR
      )
      {
         int len = strlen(HMPI_Error_messages[(errnum - MPC_ERR_LAST)]);
         strncpy(message, HMPI_Error_messages[(errnum - MPC_ERR_LAST)], len);
         return HMPI_OK;
      }

      /*
       * MPI errors
       */
      rc = MPI_Error_string(
              errnum,
              message,
              &errlen
      );

      if (rc != MPI_SUCCESS)
      {
         return rc;
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Get_version
   (
       int* version,
       int* sub_version
   )
   {
       *version = HMPI_VERSION;
       *sub_version = HMPI_SUBVERSION;

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int
   HMPI_Debug
   (
       int yesno
   )
   {
       if (yesno)
       {
          HMPI_Debug_flag = 1;

          return HMPI_OK;
       }

       HMPI_Debug_flag = 0;

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   /*
    * HMPI_Group_timeof() -- Returns the estimated time of 
    *                        execution of the algorithm
    *                             
    * ARGUMENTS: - see HMPI documentation
    *
    * RETURN CODES: - see HMPI documentation
    */   
   double
   HMPI_Group_timeof
   (
       const HMPI_Group* pgroup_t
   )
   {
       HMPI_Net* pnet;
       MPC_Net* net;
       int index_of_net;
       int result;
       MPI_Comm* pcomm;

       if (!HMPI_Is_member(pgroup_t))
       {
          return HMPI_UNDEFINED;
       }

       result = HMPI_Test_net_id(pgroup_t);

       if (result != HMPI_OK)
       {
          return HMPI_INVALID_GROUP;
       }

       {
          pnet = HMPI_Get_net_from_map(
                     pgroup_t,
                     &index_of_net
          ); 

          if (pnet == NULL)
          {
             return HMPI_ERR_GROUP_NOT_EXIST;
          }
       }

       pcomm = (MPI_Comm*)HMPI_Get_comm(
                              pgroup_t
       );

       if (pcomm == NULL)
       {
          return HMPI_ERR_INTERNAL;
       }

       result = MPI_Bcast(
                   &(pnet->time),
                   1,
                   MPI_DOUBLE,
                   HMPI_Group_parent(pgroup_t),
                   *pcomm
       );

       if (result != MPI_SUCCESS)
       {
          return HMPI_UNDEFINED;
       }

       return pnet->time;
   }

   /*-----------------------------------------------------*/
