
/*************************************************************************
*                                                                        *
*                 HeteroMPI Programming Environment                      *
*                 =================================                      *
*                                                                        *
*  Copyright (c) 2005          Department of Computer Science,           *
*                              University College Dublin.                *
*                                                                        *
*  All rights reserved. We assume no responsibility for the use          *
*  or reliability of our software.                                       *
*                                                                        *
*************************************************************************/

   /***********************************************************/
   /* hmpi - Interfaces for the HeteroMPI library             */
   /*                                                         */
   /* Revision history                                        */
   /* DD-MM-YYYY                                              */
   /* ----------                                              */
   /* 01-02-2002  --      Initial version                     */
   /* 05-07-2002  --      Added HMPI_Timeof                   */
   /* 04-10-2007  --      Added HMPI_Group_timeof             */
   /* 16-10-2007  --      Added HMPI_Group_pauto_create       */
   /*                     HMPI_Group_heuristic_pauto_create   */
   /*                     Process arrangments are evaluated   */
   /*                     parallel                            */
   /***********************************************************/

   #ifndef __HMPI_HH
   #define __HMPI_HH

   #include <sys/types.h>
   #include <unistd.h>
   #include <limits.h>

   #include <mpi.h>

   #include <mpC.h>
   #include <topo.h>
   #include <mpc_macro.h>

   #include <hmpi_internal.h>
   #include <hmpi_err.h>
   #include <hmpi_ops.h>
   #include <hdpi.h>

   #ifndef DBL_MAX
   #define DBL_MAX  1.7976931348623157E+308
   #endif

   /*
    * HMPI specification version
    * Any changes to the HMPI interfaces, the version
    * must be incremented
    */
   #define HMPI_VERSION       1
   #define HMPI_SUBVERSION    0

   /*
    * Type of Sub group
    * Used in operation HMPI_Subgroup_create
    * useful for porting mpC applications to 
    * HMPI.
    */
   #define HMPI_FLEXIBLE      MPC_FLEXIBLE
   #define HMPI_NOT_FLEXIBLE  MPC_NOT_FLEXIBLE

   /* 
    * HMPI version
    */
   extern int
   HMPI_Get_version
   (
       int* version,
       int* sub_version
   );

   /* 
    * HMPI initialization and Finalization 
    */
   extern int 
   HMPI_Init
   (
       int *pargc,
       char*** pargv
   );

   extern int
   HMPI_Finalize
   (
       int exitcode
   );

   extern int
   HMPI_Abort
   (
       int exitcode
   );

   /*
    * Group manipulation functions 
    */
   extern int
   HMPI_Group_create
   (
       HMPI_Group* _gid,
       const HMPI_Model* model,
       int* model_parameters,
       int model_param_count
   );

   extern int
   HMPI_Group_auto_create
   (
       HMPI_Group* _gid,
       const HMPI_Model* model,
       const int* model_parameters,
       int model_param_count
   );

   extern int
   HMPI_Group_pauto_create
   (
       HMPI_Group* _gid,
       const HMPI_Model* model,
       const int* model_parameters,
       int model_param_count
   );

   typedef int (*HMPI_Heuristic_function)(
           int np, int *dp, const int *modelp, int paramc);

   extern int
   HMPI_Group_heuristic_auto_create
   (
       HMPI_Group* _gid,
       const HMPI_Model* model,
       HMPI_Heuristic_function hfunc,
       const int* model_parameters,
       int model_param_count
   );

   extern int
   HMPI_Group_heuristic_pauto_create
   (
       HMPI_Group* _gid,
       const HMPI_Model* model,
       HMPI_Heuristic_function hfunc,
       const int* model_parameters,
       int model_param_count
   );

   extern int
   HMPI_Group_free
   (
       HMPI_Group* _gid
   );

   /*
    * Subgroup manipulation functions 
    */
   extern int
   HMPI_Subgroup_create
   (
       HMPI_Group* subgroup_id,
       const HMPI_Group* parent_group_id,
       int subgroup_type,
       const char* conditions
   );

   /*
    *  Group accessors
    */
   extern int
   HMPI_Group_rank
   (
       const HMPI_Group* gid
   );

   extern int
   HMPI_Rank
   (
       const HMPI_Group* gid,
       const int* coord
   );

   extern int
   HMPI_Group_coordof
   (
       const HMPI_Group* gid,
       int* num_of_coordinates,
       int** coordinates
   );

   extern int
   HMPI_Coordof
   (
       const HMPI_Group* gid,
       int rank,
       int* num_of_coordinates,
       int** coordinates
   );

   extern int
   HMPI_Group_topo_size
   (
       const HMPI_Group* gid
   );

   extern int
   HMPI_Group_topology
   (
       const HMPI_Group* gid,
       int* num_of_coordinates,
       int** coordinates
   );

   extern int
   HMPI_Group_process_ranks
   (
       const HMPI_Group* pgid,
       int coord_count,
       const int* coordinates,
       int* rank_count,
       int** ranks
   );

   extern int
   HMPI_Group_parent
   (
       const HMPI_Group* gid
   );

   extern int
   HMPI_Group_size
   (
       const HMPI_Group* gid
   );

   extern unsigned char
   HMPI_Is_host();

   extern unsigned char
   HMPI_Is_parent
   (
       const HMPI_Group* gid
   );

   extern unsigned char 
   HMPI_Is_member
   (
       const HMPI_Group* gid
   );

   extern unsigned char
   HMPI_Is_free();

   extern
   const MPI_Comm*
   HMPI_Get_comm
   (
       const HMPI_Group* gid
   );

   extern int 
   HMPI_Get_number_of_processors();

   extern int 
   HMPI_Get_processors_info
   (
      double* relative_performances
   );

   extern int 
   HMPI_Get_processes_info
   (
      double* relative_performances
   );

   extern int 
   HMPI_Group_performances
   (
      const HMPI_Group* gid,
      double* relative_performances
   );

   extern int 
   HMPI_Get_number_of_free_processes();

   /*
    * Net synchronization functions 
    */
   extern
   int 
   HMPI_Barrier
   (
       const HMPI_Group* g
   );

   extern int
   HMPI_Host_rendezvous(int count);

   extern int
   HMPI_Notify_free_processes();

   extern int 
   HMPI_Sync_free_processes();

   #define  HMPI_Wait_free_processes   HMPI_Sync_free_processes

   /*
    * Recon functions
    */
   typedef void (*HMPI_Benchmark_function)(const void*, int, void*);

   extern
   int
   HMPI_Recon
   (
       HMPI_Benchmark_function func,
       const void* input_p,
       int num_of_parameters,
       void* output_p
   );

   /*
    * Algorithm time estimation functions
    */
   extern 
   double
   HMPI_Timeof
   (
       const HMPI_Model* _model,
       const int* model_parameters,
       int model_param_count
   );

   extern 
   double
   HMPI_Group_timeof
   (
       const HMPI_Group* gid
   );

   /*
    * Debugging routines
    */
   typedef int(*_hmpi_printf)(const char* format, ...);
   extern _hmpi_printf HMPI_Printf;

   _hmpi_printf HMPI_Printf;
   #define HMPI_Debug_printf printf 
    
   extern int
   HMPI_Strerror
   (
      int errnum,
      char* message
   );

   extern int 
   HMPI_Debug
   (
      int yesno
   );

   #endif /* __HMPI_HH */

