
   #include <math.h>
   #include "mxm_i.h"

   /*-----------------------------------------------------*/

   int
   Input_recon
   (
      double *a,
      double *b,
      double *c
   )
   {
     int i, j;
     for (i = 0; i < recon_r; i++)
     {
          for (j = 0; j < recon_n; j++)
          {
              a[i*recon_n + j] = 2.0;
          }
      }

      for (i = 0; i < recon_n; i++)
      {
          for (j = 0; j < recon_t; j++)
          {
              b[i*recon_t + j] = 2.0;
          }
      }

      for (i = 0; i < recon_r; i++)
      {
          for (j = 0; j < recon_t; j++)
          {
              c[i*recon_t + j] = 0.0;
          }
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int Perf_func
   (
       int x, /* --r-- */
       int y, /* --n-- */
       int z  /* --t-- */
   )
   {
      /*
       * Multiplication of r*n and n*t matrices.
       */
      int i,j,k;
      for (i = 0; i < x; i++)
      {
          for (j = 0; j < z; j++)
          {
              for (k = 0; k < y; k++)
              {
                  c[i*z + j] += a[i*x + k] * b[k*z + j];
              }
          }
      }

      return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   void Benchmark_function
   (
      const void* input_p,
      int num_of_p,
      void* output_p
   )
   {
      int* params = (int*)input_p;

      int result = Perf_func(
                       params[0],
                       params[1],
                       params[2]
      );

      *(int*)output_p = result;

      return;
   }

   /*-----------------------------------------------------*/

   int Do_recon()
   {
       a = (double*)malloc(
                    sizeof(double)
                    *
                    (recon_r*recon_n)
       );

       b = (double*)malloc(
                    sizeof(double)
                    *
                    (recon_n*recon_t)
       );

       c = (double*)malloc(
                    sizeof(double)
                    *
                    (recon_r*recon_t)
       );

       Input_recon(
         a,
         b,
         c
       );

       if (HMPI_Is_member(HMPI_COMM_WORLD_GROUP))
       {
            int rc;
            int output_p;
            int input_p[3] =
                {
                   recon_r,
                   recon_n,
                   recon_t
            };

            rc = HMPI_Recon(
                     &Benchmark_function,
                     input_p,
                     3,
                     &output_p
            );

            if  (rc != HMPI_OK)
            {
                printf("Panic: HMPI_Recon failed\n");
                return rc;
            }
       }

       if (HMPI_Is_host())
       {
          printf("Processor speeds refreshed\n");
       }

       free(a);
       free(b);
       free(c);

       return HMPI_OK;
   }

  /*-----------------------------------------------------*/

  int Pack_model_parameters
  (
     int Generalised_block_size,
     int *w,
     int *h,
     int *trow,
     int *model_params,
     int model_count
  )
  {
     int ind, i, j;
     
     model_params[0] = p; 
     model_params[1] = n; 
     model_params[1+1] = r; 
     model_params[1+1+1] = Generalised_block_size;

     for (ind = 0; ind < p; ind++) 
     {         
        model_params[1+1+1+1+ind] = w[ind];
     } 

     for (i = 0; i < (p); i++) 
     {         
        for (j = 0; j < (p); j++) 
        {         
           model_params[1+1+1+1+p+i*p+j] = h[i*p+j];
        }
     } 

     for (i = 0; i < (p); i++) 
     {         
        for (j = 0; j < (p); j++) 
        {         
           model_params[1+1+1+1+p+(p*p)+i*p+j] = trow[i*p+j];
        }
     } 

     return HMPI_OK;
  }

  /*-----------------------------------------------------*/

  int Execute_algorithm()
  {
     int rc, i, j, k, me;

     HMPI_Group gid;
     int *model_params;
     int model_count;
     double *dperf;
     int *iperf;

     me = HMPI_Group_rank(HMPI_COMM_WORLD_GROUP);

     model_count = 1+1+1+1+p+(p*p)+(p*p);
     model_params = (int*)malloc(
                          sizeof(int)
                          *
                          model_count
     );

     if (model_params == NULL)
     {
        return -1;
     }

     w = (int*)malloc(
               sizeof(int)
               *
               p
     );

     if (w == NULL)
     {
        return -1;
     }

     h = (int*)malloc(
               sizeof(int)
               *
               (p*p)
     );

     if (h == NULL)
     {
        return -1;
     }

     trow = (int*)malloc(
                  sizeof(int)
                  *
                  (p*p)
     );

     if (trow == NULL)
     {
        return -1;
     }

     row_allocations = (int*)malloc(
                                 sizeof(int)
                                 *
                                 (p*p)
     );

     if (row_allocations == NULL)
     {
        return -1;
     }

     column_allocations = (int*)malloc(
                                    sizeof(int)
                                    *
                                    p
     );

     if (column_allocations == NULL)
     {
        return -1;
     }

     dperf = (double*)malloc(
                     sizeof(double)
                     *
                     (p*p)
     );

     if (dperf == NULL)
     {
        return -1;
     }

     iperf = (int*)malloc(
                   sizeof(int)
                   *
                   (p*p)
     );

     if (iperf == NULL)
     {
        return -1;
     }

     rc = HMPI_Get_processors_info(
              dperf
     );

     if (rc != HMPI_OK)
     {
        return rc;
     }

     for (i = 0; i < (p*p); i++)
     {
        iperf[i] = dperf[i];
     }

     if (HMPI_Is_host())
     {
        printf("Performances are: ");

        for (i = 0; i < (p*p); i++)
        {
            printf("%d ", iperf[i]);
        }

        printf("\n");
     }

     if (HMPI_Is_host())
     {
        int bsize;
        double time, min_time = 1.7976931348623157E+308;

       // for (bsize = p; bsize <= n/r; bsize++)
        {
           /* The generalised blocks should fit    
            * the square matrix
            */
          // int fitted = (n/r)%bsize;

          // if (fitted != 0)
           //{
            // continue;
          // }
           bsize = 15;

           printf("=========Block size=%d=============\n", bsize);

           Generalised_block = (int(*)[2])malloc(
                                sizeof(int[2])
                                *
                                (bsize)*(bsize)
           );

           if (Generalised_block == NULL)
           {
              return -1;
           }

           rc = Distribute_load(
                  p,
                  iperf,
                  bsize,
                  row_allocations,
                  column_allocations
           );

           if (rc != HMPI_OK)
           {
              return rc;
           }

           rc = Determine_distribution_parameters(
                  p,
                  row_allocations,
                  column_allocations,
                  w,
                  h,
                  trow
           );

           if (rc != HMPI_OK)
           {
              return rc;
           }

           rc = Pack_model_parameters(
                    bsize,
                    w,
                    h,
                    trow,
                    model_params,
                    model_count
           );

           if (rc != HMPI_OK)
           {
              return rc;
           }

           printf("TIMEOF: time=");          

           time = HMPI_Timeof(
                      &MPC_NetType_ParallelAxB,
                      model_params,
                      model_count
           );

           printf(
              "%0.6f\n", 
              time
           );

           if (time < min_time)
           {
              Optimal_generalised_block_size = bsize;
              min_time = time;
           }

           free(Generalised_block);

           printf("===================================\n");
        }
     }

     rc = MPI_Bcast(
             &Optimal_generalised_block_size,
             1,
             MPI_INT,
             0,
             HMPI_COMM_WORLD
     );

     if (rc != MPI_SUCCESS)
     {
        return rc;
     }

     if (HMPI_Is_host())
     {
        printf("\n\n");
        printf("Optimal generalised block size = %d\n", Optimal_generalised_block_size);
        printf("\n\n");
     }

     Generalised_block = (int(*)[2])malloc(
                         sizeof(int[2])
                         *
                         (Optimal_generalised_block_size)
                         *
                         (Optimal_generalised_block_size)
     );

     if (Generalised_block == NULL)
     {
        return -1;
     }

     rc = Distribute_load(
             p,
             iperf,
             Optimal_generalised_block_size,
             row_allocations,
             column_allocations
     );

     if (rc != HMPI_OK)
     {
        return rc;
     }

     rc = Determine_distribution_parameters
          (
            p,
            row_allocations,
            column_allocations,
            w,
            h,
            trow
     );

     if (rc != HMPI_OK)
     {
        return rc;
     }

     if ((VERBOSE > 0)
         && HMPI_Is_host()
     )
     {
        printf("=======================\n");
        printf("Distribution parameters\n");
        printf("=======================\n");
        printf("Widths are:\n");

        for (i = 0; i < p; i++)
        {
           printf("%d ", w[i]);
        }

        printf("\n");

        printf("Heights are:\n");

        for (i = 0; i < p; i++)
        {
           for (j = 0; j < p; j++)
           {
              printf("%d ", h[i*p+j]);
           }
           printf("\n");
        }

        printf("top rows are:\n");

        for (i = 0; i < p; i++)
        {
           for (j = 0; j < p; j++)
           {
              printf("%d ", trow[i*p+j]);
           }
           printf("\n");
        }

        printf("\n");

        printf("=======================\n");
        printf("=======================\n");
     }

     rc = Pack_model_parameters(
              Optimal_generalised_block_size,
              w,
              h,
              trow,
              model_params,
              model_count
     );

     if (rc != HMPI_OK)
     {
        return rc;
     }

     if (HMPI_Is_host()
         || HMPI_Is_free()
     )
     {
        int rc = HMPI_Group_create(
                     &gid,
                     &MPC_NetType_ParallelAxB,
                     model_params,
                     model_count
        );

        if (rc != HMPI_OK)
        {
           return rc;
        }
     }

     if (HMPI_Is_free())
     {
        free(row_allocations);
        free(column_allocations);
        free(Generalised_block);
        free(model_params);

        free(dperf);
        free(iperf);

        free(h);
        free(w);
        free(trow);

        HMPI_Finalize(0);
     }

     /*
      * Calculate the barrier time for possible use.
      */
     {
        int rc;
        struct timeval start, end;
        double i_barrier_time, f_barrier_time;

        gettimeofday(&start, NULL);

        for (i = 0; i < BARRIER_ITERATIONS; i++)
        {
           rc = HMPI_Barrier(&gid);

           if (rc != HMPI_OK)
           {
              return rc;
           }
        }

        gettimeofday(&end, NULL);

        i_barrier_time = start.tv_sec + (start.tv_usec/pow(10, 6));
        f_barrier_time = end.tv_sec + (end.tv_usec/pow(10, 6));

        barrier_time = (f_barrier_time - i_barrier_time)/BARRIER_ITERATIONS;
     }

     if (HMPI_Is_host())
     {
        printf(
           "Starting the matrix-matrix multiplication\n"
        );
     }

     rc = Perform_mxm(
                 &gid
     );

     if (rc != HMPI_OK)
     {
        return rc;
     }

     rc = HMPI_Barrier(&gid);

     if (rc != HMPI_OK)
     {
        return rc;
     }

     free(row_allocations);
     free(column_allocations);
     free(Generalised_block);
     free(model_params);

     free(dperf);
     free(iperf);

     free(h);
     free(w);
     free(trow);

     /*
      * Destroy the group
      */
     if (HMPI_Is_member(&gid))
     {
        int rc = HMPI_Group_free(&gid);

        if (rc != HMPI_OK)
        {
           return rc;
        }
     }

     return HMPI_OK;
  }

  /*-----------------------------------------------------*/

  int
  Perform_mxm
  (
     const HMPI_Group* gid
  )
  {
     int rc;

     A = (double*)malloc(
                  sizeof(double)
                  *
                  (N*N)
     );

     if (A == NULL)
     {
        printf("PANIC: heap problems, Allocation of A\n");
        return -1;
     }

     B = (double*)malloc(
                  sizeof(double)
                  *
                  (N*N)
     );

     if (B == NULL)
     {
        printf("PANIC: heap problems, Allocation of B\n");
        return -1;
     }

     C = (double*)malloc(
                  sizeof(double)
                  *
                  (N*N)
     );

     if (C == NULL)
     {
        printf("PANIC: heap problems, Allocation of C\n");
        return -1;
     }

     rc = mxm(
            gid,
            A,
            B,
            C
     );

     if (rc != HMPI_OK)
     {
        printf(
          "MXM:Problems multiplying the matrices A and B "
          "...Exiting...\n"
        );

        return rc;
     }

     free(A);
     free(B);
     free(C);

     return HMPI_OK;
  }

  /*-----------------------------------------------------*/

  int
  mxm
  (
     const HMPI_Group* gid,
     double *a,
     double *b,
     double *c
  )
  {
     int result;
     int x, y;
     int icoord, jcoord;
     int i, j, k, l;
     int** my_coordinates = (int**)malloc(
                                   sizeof(int*)
     );
     int num_of_coordinates;

     result =  HMPI_Group_coordof(
                   gid,
                   &num_of_coordinates,
                   my_coordinates
     ); 

     if (result != HMPI_OK)
     {
        HMPI_Printf("Error while getting the coordinates\n");
        return result;
     }

     /*
      * Initialize the respective array elements A & B at
      * the processors.
      * Each array element is a r*r matrix.
      */
     icoord = (*my_coordinates)[0];
     jcoord = (*my_coordinates)[1];

     istart = 0;
     for (i = 0; i < icoord; i++)
     {
         istart += row_allocations[(i*p) + jcoord];
     }

     jstart = 0;
     for (j = 0; j < jcoord; j++)
     {
         jstart += (column_allocations[j]);
     }

     myialloc = row_allocations[(icoord*p) + jcoord];
     myjalloc = (column_allocations[jcoord]);

     for (x = (istart*r); x < N; x+=(Optimal_generalised_block_size*r))
     {
         for (y = (jstart*r); y < N; y+=(Optimal_generalised_block_size*r))
         {
             for (i = 0; i < (myialloc*r); i+=r)
             {
                 for (j = 0; j < (myjalloc*r); j+=r)
                 {
                     for (k = 0; k < r; k++)
                     {
                         for (l = 0; l < r; l++)
                         {
                             a[(x*N) + y + (i*N) + j + (k*N) + l] 
                             = 
                             MXM_CONSTANT_NUMBER
                             ;

                             b[(x*N) + y + (i*N) + j + (k*N) + l] 
                             = 
                             MXM_CONSTANT_NUMBER
                             ;

                             c[(x*N) + y + (i*N) + j + (k*N) + l] 
                             = 
                             0.0
                             ;
                         }
                     }
                 }
             }
         }
     }

     /*
      * Check to see if the matrix elements are properly
      * initialised.
      */
     if (VERBOSE > 1)
     {
        for (x = (istart*r); x < N; x+=(Optimal_generalised_block_size*r))
        {
            for (y = (jstart*r); y < N; y+=(Optimal_generalised_block_size*r))
            {
                for (i = 0; i < (myialloc*r); i+=r)
                {
                    for (j = 0; j < (myjalloc*r); j+=r)
                    {
                        for (k = 0; k < r; k++)
                        {
                            for (l = 0; l < r; l++)
                            {
                                printf(
                                    "a[%d][%d] = %0.2f\n",
                                    x + i + k,
                                    y + j + l,
                                    a[(x*N) + y + (i*N) + j + (k*N) + l]
                                );
                            }
                        }
                    }
                }
            }
        }
     }

     {
        int result = Grid_computations(
                         gid,
                         (*my_coordinates),
                         a,
                         b,
                         c
        );

        if (result != HMPI_OK)
        {
           HMPI_Printf("Error while performing grid computations\n");
           return result;
        }
     }

     free(my_coordinates[0]);
     free(my_coordinates);

     return HMPI_OK;
  }

  /*-----------------------------------------------------*/

  int
  Grid_computations
  (
     const HMPI_Group* gid,
     const int* my_coordinates,
     double *a,
     double *b,
     double *c
  )
  {
     int x, y, i, j, k, l, m, t;

     Block Ablock, Bblock;
     Processor RootProcessor, MyProcessor, ReceiverProcessor;

     MPI_Comm* grid_comm = (MPI_Comm*)HMPI_Get_comm(
                               gid
     );

     if (grid_comm == NULL)
     {
        HMPI_Printf("Panic: grid communicator is NULL\n");
        return -1;
     }

     MyProcessor.I = my_coordinates[0];
     MyProcessor.J = my_coordinates[1];

     for (k = 0; k < n; k+=r)
     {
         int Acolumn = ((k/r)%Optimal_generalised_block_size);
         int Brow = ((k/r)%Optimal_generalised_block_size);

         /*
          * P(i,k) broadcasts a(i,k) to p(i,*) horizontally.
          */
         for (i = 0; i < n; i+=r)
         {
             int Arow = ((i/r)%Optimal_generalised_block_size);
             GetBlock(Arow, Acolumn, &Ablock);
             GetProcessor(&Ablock, Optimal_generalised_block_size, &RootProcessor); 
             if (((MyProcessor.I) == (RootProcessor.I))
                 && ((MyProcessor.J) == (RootProcessor.J)
                )
             )
             {
                for (ReceiverProcessor.I = 0; ReceiverProcessor.I < p; ReceiverProcessor.I++)
                {
                    for (ReceiverProcessor.J = 0; ReceiverProcessor.J < p; ReceiverProcessor.J++)
                    {
                        if ((RootProcessor.I != ReceiverProcessor.I) 
			    || (RootProcessor.J != ReceiverProcessor.J))
                        {
                           int TopRow = trow[(ReceiverProcessor.I)*p + ReceiverProcessor.J];
                           int BottomRow = TopRow + h[(ReceiverProcessor.I)*p + ReceiverProcessor.J];
                           int Total_number_of_elements_sent_horizontally =
                               Get_number_of_elements_to_be_transferred(Arow, Arow+1, TopRow, BottomRow, w[RootProcessor.J]);
                           if (Total_number_of_elements_sent_horizontally > 0)
                           {
                              double temp[r*r];
                              int ProcessorCoords[2] = {ReceiverProcessor.I, ReceiverProcessor.J};
                              int dest = HMPI_Rank(gid, ProcessorCoords);
                               
                              for (x = 0; x < r; x++)
                              {
                                  for (y = 0; y < r; y++)
                                  {
                                      temp[x*r + y] = a[i*n + k + x*n + y];
                                  }
                              }

                              MPI_Send(&temp, r*r, MPI_DOUBLE, dest, HMPI_MSG_TAG,*grid_comm);
                           }
                        }
                    }
                }
             }
             else
             {
                int MyTopRow =  trow[(MyProcessor.I)*p+MyProcessor.J];
                int MyBottomRow =  MyTopRow+h[(MyProcessor.I)*p+MyProcessor.J];
                int Total_number_of_elements_sent_horizontally =
                    Get_number_of_elements_to_be_transferred(Arow, Arow+1, MyTopRow, MyBottomRow, w[RootProcessor.J]);
                if (Total_number_of_elements_sent_horizontally > 0)
                {
                    MPI_Status stat;
                    double temp[r*r];
                    int RootProcessorCoords[2] = {RootProcessor.I, RootProcessor.J};
                    int root = HMPI_Rank(gid, RootProcessorCoords);
                    MPI_Recv(&temp, r*r, MPI_DOUBLE, root, HMPI_MSG_TAG, *grid_comm, &stat);

                    for (x = 0; x < r; x++)
                    {
                        for (y = 0; y < r; y++)
                        {
                            a[i*n + k + x*n + y] = temp[x*r + y];
                        }
                    }
                }
             }
          }

          /*
           * P(k,j) broadcasts a(k,j) to p(*,j) vertically.
           */
          for (j = 0; j < n; j+=r)
          {
              int Bcolumn = ((j/r)%Optimal_generalised_block_size); 
              GetBlock(Brow, Bcolumn, &Bblock);
              GetProcessor(&Bblock, Optimal_generalised_block_size, &RootProcessor);
              if (MyProcessor.J == RootProcessor.J)
              {
                  if (MyProcessor.I == RootProcessor.I)
                  {
                     for (ReceiverProcessor.I = 0; ReceiverProcessor.I < p; ReceiverProcessor.I++)
                     {
                         if (RootProcessor.I != ReceiverProcessor.I)
                         {
                            double temp[r*r];
                            int ProcessorCoords[2] = { ReceiverProcessor.I, RootProcessor.J};
                            int dest = HMPI_Rank(gid, ProcessorCoords);
                            for (x = 0; x < r; x++)
                            {
                                for (y = 0; y < r; y++)
                                {
                                    temp[x*r + y] = b[k*n + j + x*n + y];
                                }
                            }
                            MPI_Send(&temp, r*r, MPI_DOUBLE, dest, HMPI_MSG_TAG,*grid_comm);
                         }
                     }
                  }
                  else
                  {
                     MPI_Status stat;
                     double temp[r*r];
                     int RootProcessorCoords[2] = {RootProcessor.I, RootProcessor.J};
                     int root = HMPI_Rank(gid, RootProcessorCoords);
                     MPI_Recv(&temp, r*r, MPI_DOUBLE, root, HMPI_MSG_TAG, *grid_comm, &stat);
                     for (x = 0; x < r; x++)
                     {
                         for (y = 0; y < r; y++)
                         {
                             b[k*n + j + x*n + y] = temp[x*r + y];
                         }
                     }
                  }
              }
          }

          for (x = (istart*r); x < n; x+=(Optimal_generalised_block_size*r))
          {
              for (y = (jstart*r); y < n; y+=(Optimal_generalised_block_size*r))
              {
                  for (i = 0; i < (myialloc*r); i+=r)
                  {
                      for (j = 0; j < (myjalloc*r); j+=r)
                      {
                          /*
                           * Multiplication of a[i][k] * b[k][j]
                           * is equivalent to multiplying 2 r*r
                           * matrices.
                           */
                          for (l = 0; l < r; l++)
                          {
                              for (m = 0; m < r; m++)
                              {
                                  for (t = 0; t < r; t++)
                                  {
                                      c[x*n + y + i*n + j + l*n + m] 
                                      += 
                                      a[x*n + 0 + i*n + k + l*n + t]
                                      *
                                      b[0   + y + k*n + j + t*n + m]
                                      ;
                                  }
                              }
                          }
                      }
                  }
              }
          }
      }

      /*
       * The result of the computations.
       */
      if (VERBOSE > 1)
      {
         for (x = (istart*r); x < N; x+=(Optimal_generalised_block_size*r))
         {
             for (y = (jstart*r); y < N; y+=(Optimal_generalised_block_size*r))
             {
                 for (i = 0; i < (myialloc*r); i+=r)
                 {
                     for (j = 0; j < (myjalloc*r); j+=r)
                     {
                          for (l = 0; l < r; l++)
                          {
                              for (m = 0; m < r; m++)
                              {
                                  printf(
                                      "c[%d][%d] = %0.2f\n",
                                      x + i + l,
                                      y + j + m,
                                      c[x*N + y + i*N + j + l*N + m]
                                  );
                              }
                          }
                     }
                 }
             }
         }
      }

      return HMPI_OK;
   }

  /*-----------------------------------------------------*/

   int Get_number_of_elements_to_be_transferred
   (
       int top_row_1,
       int bottom_row_1,
       int top_row_2,
       int bottom_row_2,
       int width_1
   )
   {
       /*
        * One area contains the other
        */
       if ((top_row_1 >= top_row_2)
           && (bottom_row_1 <= bottom_row_2)
       )
       {
          return (bottom_row_1 - top_row_1)*width_1;
       }

       if ((top_row_1 <= top_row_2)
           && (bottom_row_1 >= bottom_row_2)
       )
       {
          return (bottom_row_2 - top_row_2)*width_1;
       }

       /*
        * One area is followed or preceded by another
        * with an overlap
        */
       if ((top_row_1 <= top_row_2)
           && (bottom_row_1 >= top_row_2)
           && (bottom_row_1 <= bottom_row_2)
       )
       {
          return (bottom_row_1 - top_row_2)*width_1;
       }

       if ((top_row_1 >= top_row_2)
           && (top_row_1 <= bottom_row_2)
           && (bottom_row_1 >= bottom_row_2)
       )
       {
          return (bottom_row_2 - top_row_1)*width_1;
       }

       /*
        * There is no overlap
        */
       if ((bottom_row_1 < top_row_2)
           || (top_row_1 > bottom_row_2)
       )
       {
          return 0;
       }

       if ((top_row_1 < top_row_2)
           && (bottom_row_1 < bottom_row_2)
       )
       {
          return 0;
       }

       if ((top_row_1 > top_row_2)
           && (bottom_row_1 > bottom_row_2)
       )
       {
          return 0;
       }

       return 0;
   }

  /*-----------------------------------------------------*/

  int
  GetBlock(int x, int y, Block *b)
  {
     b->I = x;
     b->J = y;

     return 0;
  }

  /*-----------------------------------------------------*/

  int
  GetProcessor(Block *b, int Generalised_block_size, Processor* p)
  {
     p->I = Generalised_block[((b->I)*Generalised_block_size) + (b->J)][0];
     p->J = Generalised_block[((b->I)*Generalised_block_size) + (b->J)][1];

     return 0;
  }

  /*-----------------------------------------------------*/

