

/* errors.h  Routines to aid error handling. */

#ifndef ERRORS
#define ERRORS 1

typedef int error; /* A type so that we can pretend C is strongly typed. */

/* Possible errors: */
#define NoError    0
  /* "Everything worked"   Most routines know that this is zero,
   * don't change it to another number.
   */

#define ERRfalse           1
        /* Condition was false */

#define ERRnotfound        2
        /* Not found */

#define ERRnil             3
        /* Nil pointer passed to routine */

#define ERRsame            4
        /* Arrays not distinct arrays
         * Some routines need distinct arrays to avoid problems overwriting
         * needed input with output.
         */
           
#define ERRsize            5
        /* Zero or negative sized array */

#define ERRrange           6
        /* index not in range
         * Array out of bounds, access outsize of the items of  
         * a list, etc.
         */

#define ERRnumeric         7
        /* numeric error
         * Result not meaningful (divide by zero) or not significant (Some
         * statistical operations)
         */

#define ERRconform          8
        /* Arguments don't conform
         * (In the sense that a matrix multiply needs conforming arrays.)
         */

#define ERRnomem            9
        /* Out of memory */

#define ERRnodisk          10
        /* Out of disk space */

#define ERRsysIO           11
        /* System error being passed back. */

#define ERRnotimplemented  12
       /* Code still being written */

#define ERRmeaning         13
       /* Meaningless operation.
        * For example, scaling a collection of values by zero
        */

#define ERRvalue           14
       /* Value out of range. */

#define ERRuninitialized   15
       /* Use of uninitialize value or package */

#define ERRreinitialized   16
       /* Attempt to initialize twice without finalize first */

#define ERRbaddata         17
       /* Data not acceptable */

#define ERRempty           18
       /* Data not provided   */

#define ERRnesting         19
       /* Calls badly nested */

#define ERRendoffile       20
       /* End of file        */

#define ERRoverflow        21
       /* More data than space available */

#define ERRcorrupt         22
       /* Impossible error, internals of program corrupt? */

#define ERRendofline       23
       /* Hit end of line while expecting more. */

#define maxERR             23 
       /* The highest error number in use. */


/* --------------------- History Support ------------------------------------ */

error lasterror; /* Last error code registered */
char *lastfile;  /* File it was in */
int   lastline;  /* Line of the file */

/* ---------------------  Support routines ---------------------------------- */

/* Note where an error occurred */
#ifdef __FILE__
#define noteerror(Error,Text) seterror (Error, __FILE__, __LINE__, Text)
#else
#define noteerror(Error,Text) seterror (Error, 0, 0, Text)
#endif

extern error seterror (error Which, char *whichfile, int line, char *message);
   /* This routine just returns the error code given, so
    * that one can write, for example,
    *  return noteerror(ERRwhatever);
    */

/* Output text of last error, if any.  (No print if NoError)  */
extern error dumperror();

/* Clear an error status */
extern error clearerror();

/* ---------------------- Raw print of error -------------------------------- */

/* Given a return status, print the error. */
extern error printerror  (error whichone); /* To standard out */
/* Returns zero if it is a known error. */

/* Exit the program, reporting on an error */
extern error aborterror (error whichone, char *text);

#endif
