/*
 *  topo.c -mpC run-time system. Node placement functions 
 *  Coded by Dm.Arapov for A.Lastovetsky 1996
 *  release 29.04.96 
 *  -- 1.07.96 new allocation strategy implemented
 *  --22.08.96 Var functions implemented
 *  --22.08.96 MPC_Optimize now takes parent parameter
 *  --23.12.96 MPC_Find_best_group coorected maxproc condition tested
 *  -----------------------------------------------------------------------------
 *  -- March-June 1999, re-designed and re-coded by A.Lastovetsky ---------------
 *  -- to take into account communication links ---------------------------------
 *  -----------------------------------------------------------------------------
 */
#include "mpC.h"
#include "mpCtopo.h"
#include <stdlib.h>
#include <stdio.h>
#include <float.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

const char MPC_Topofile[] = "MPCTOPOFILE";
const char MPC_Tracemapfile[] = "MPCTRACEMAPFILE";
const char MPC_Topo[] = "MPCTOPO";
const char MPC_Current_topo[] = "/current.topo";

#define MPC_MAX_ENVTOPOSTRINGLENGTH 32768

static FILE* tracemap = NULL;

#define  MPC_Debug_printf printf

/***new***MPC_Hierarchy_descr*****@@@*******************************/
typedef struct MPC_Link_hierarchy_descr {
  Int level;                            /* Level of link hierarchy */
  double linkspeed[MPC_TRANSFER_DATA_RANGE];
                                        /* 1.0/(link speed) on different sizes of */
                                        /* transferred data messages, sec/byte */
  struct MPC_Link_hierarchy_descr *top; /* points to higher link descriptor */
} MPC_Hi_descr;

/**corrected**@@@**/
typedef struct MPC_GROUP_DESCR {
  Int maxproc;                     /* total number of processess */
  Int numproc;                     /* number of hired processes*/ 
  Int scalability;                 /* number of processors */
  Int processor_power;             /* power of one processor */
  double max_nodepower;            /* current power of a free node */
  double* nodeweights;             /* weights of allocated virtual processors */
  Int*    nodenumbers;             /* and their linear numbers in mpC network */
  MPC_Hi_descr *link_descr;        /* points to link descriptor for the group @@@*/
} MPC_Group_descr; 

Int _iMPC_NGroups;
MPC_Group_descr* _iMPC_Groups;

Int _iMPC_NProcs = 0;
double* _iMPC_rperformance = NULL;

extern Int* _iMPC_recon_net_MASK;
extern Int  _iMPC_recon_net_V;

#define ngroups  _iMPC_NGroups
#define groups   _iMPC_Groups

#define nprocs       _iMPC_NProcs
#define rperformance _iMPC_rperformance

#define recon_net_V     _iMPC_recon_net_V
#define recon_net_MASK  _iMPC_recon_net_MASK

/*******************new****MPC_Free_topo_graph*********@@@*********************/
/*
 * rreddy
 * For each element in topo_graph linked list, we 
 * delete the structure members links, link_weights, hard_link_times.
 */
void MPC_Free_topo_graph(MPC_Topo_graph *topo_graph, int power) {
  if(topo_graph==NULL)
    return;
  else
    {
      int i;
      for (i = 0; i < power; i++)
      {
         free(topo_graph[i].links);
         free(topo_graph[i].link_weights);
         free(topo_graph[i].hard_link_times);
      }
      free(topo_graph);
    }
}/* MPC_Free_topo_graph */

/********************* MPC_The_free_topology **********************************/
Int MPC_The_free_topology(MPC_Net* net)
  {
    if (MPC_DEBUG > 1)
      MPC_Debug_printf("---[%2d]-> MPC_The_free_topology\n",
                       MPC_Global_rank());
    if ((net->temp_topo) != NULL)
      free (net->temp_topo);
    if (MPC_DEBUG > 1)
      MPC_Debug_printf("<--[%2d]-- MPC_The_free_topology\n",
                       MPC_Global_rank());
    return MPC_OK;
  } /* MPC_The_free_topology */


/********************* MPC_Upper *********************new*************@@@*****/
Int MPC_Upper(Int x, Int y) /* x-number of processes, y-number of processors */
{
  int i;
  i=x/y;
  if(i<=1) return 1;
  if(i*y==x)
    return i;
  else
    return i+1;
}/* returns coefficient to divide processor power */ 

/********************* MPC_Node_weight ***********************************/
Int MPC_Node_weight(MPC_Net* net, Int i)
  {
    Int w;

    
    w = (*(net->type->node))(i,net->params,
                             net->power, &(net->nodes), &(net->links));
    if(net->bench==MPC_NOT_BENCH) {
      w -= 2;
      w = w/4;
      w = (w==0)?1:w*2;
    }
    else if(w==0)
      w=1;
    return w;    
  } /* MPC_Node_weight */

/*
 * Structure of MPC_Topology if MPC_NOT_BENCH:
 *    - Number of nodes;
 *    - array of nodes relative weights (integers)
 */

/**new@@@**/
/*
 * Structure of MPC_Topology if MPC_BENCH:
 *    - 2*(Number of nodes + Number of links)+1;
 *    - number of nodes;
 *    - array of nodes absolute weights (integers);
 *    - array to hold numbers of links outgoing from virtual processors (integers),
 *      namely, 0-th element holds the number of links  
 *      outgoing from 0-th virtual processor, 
 *      i-th element = (i-1)-th element + the number of links
 *      outgoing from i-th virtual processor; 
 *    - array to hold destination nodes for links (integers),
 *      namely, i-th element holds the number of node in which i-th link comes;
 *    - array to hold volumes of data transferring through links (integers), namely,
 *      i-th element holds the volume of data (in bytes) to transfer over i-th link
 */

/**new**/
/******************* MPC_Num_of_links ************************************@@@*/
Int MPC_Num_of_links( MPC_Net* net )
{
  if (net->num_of_links==0)
    {
      int i, j;
        for(i=0; i<MPC_Power(net); i++)
          for(j=0; j<MPC_Power(net); j++)
            if(i!=j &&
               (net->type->link)(i, j, net->params,
                                 net->power, &(net->nodes), &(net->links)))
              net->num_of_links++;
    }
  return (net->num_of_links);
} /* MPC_Num_of_links */

/***new**/
/******************** MPC_Outgoing_links ****************************@@@*******/
Int MPC_Outgoing_links(MPC_Net* net, Int i)
{
  Int j, num_of_links;
  
  for(j=0, num_of_links=0; j<MPC_Power(net); j++)
    if(i!=j && (net->type->link)(i, j, net->params,
                                   net->power, &(net->nodes), &(net->links)))
      num_of_links++;
  if (MPC_DEBUG == 1)
    printf("virtual node = %d,  outgoing links = %d\n", i, num_of_links);
  return num_of_links;
}/* MPC_Outgoing_links */


/********* new new *********scheme***********@@@***/

MPC_Topo_graph **MPC_Ptgs; 
double (*MPC_Current_mapping_estimation)(MPC_Topo_graph **, const Int *,
                                         Int, Int **, Int **);
double frexp(double, int *);

double ldexp(double, int);

int MPC_Double2ints(double single_double, int *pair_of_ints)
{
  pair_of_ints[0]=frexp(single_double, pair_of_ints+1);
}

double MPC_Ints2double(int *pair_of_ints)
{
  return ldexp((double)pair_of_ints[0], pair_of_ints[1]);
}

/********* new new *********scheme***********@@@***/

extern Int MPC_Local_env_built;
extern Int MPC_Local_env_updated_by_net_operation;
extern Int MPC_Local_env_updated_by_recon;

/********************* iMPC_The_topology_data ***************************@@@*****/
Int iMPC_The_topology_data(              /* topology info encoding in int array */
    MPC_Net* net,                        /* the net which topology is described */
    Int* plength,                        /* size of message in ints */
    MPC_Topology* atopology,             /* message containing topology info */
    MPC_Topology_free* afree_topology)   /* function to free topology messsage */
  {
    Int size, power, num_of_links;
    Int i, index;

    power = MPC_Power(net);

    
/********* new new *********scheme***********@@@***/
    if(net->type->estim!=NULL) {
      MPC_Node parent;
      Int RC;
      Int size;
      Int i, index;
      Int parent_rank;
      static total_nodes;
      Int *topology;
      
      net->power = power;
      if(net->power<=0)
        return MPC_ERR_POWER;
      
      if(!MPC_Local_env_built) {
        RC=MPC_Build_local_env();
        if(RC!=MPC_OK)
          return RC;
      }

     /* rreddy
      * Currently only host can call this function
      * because this function is used for algorithm
      * estimation and called before abstract networks
      * are created.
      *
      * But when this function is equipped to be called
      * by a parent of an abstract network, the parent
      * should refresh its environment and topological
      * information from the dispatcher.
      */
      /* if(MPC_Local_env_updated_by_net_operation) */
      {
        RC=MPC_Update_local_env();
        if(RC!=MPC_OK)
          return RC;
      }
      if(MPC_Local_env_updated_by_recon) {
        RC=MPC_Update_local_powers();
        if(RC!=MPC_OK)
          return RC;
      }
      parent_rank = MPC_Parent(net);
      parent=MPC_Global_rank();
      net->num_of_links=MPC_Num_of_links(net);
      size=2*(net->power+net->num_of_links)+1;
      if ((topology = malloc((size+1)*sizeof(Int)))==NULL)
        return MPC_ERR_NOMEM;
      index=0;
      topology[index++] = size;
      topology[index++] = net->power;  
      for (i=0;i<net->power;i++)
        topology[index++] = MPC_Node_weight(net,i);
      {
        Int j, k;
        Int *pt1, *pt2;
        
        pt1=topology+2*net->power+2;
        pt2=pt1+net->num_of_links;
        for (i=0; i<net->power; i++, index++) {
          topology[index]=
            (i?topology[index-1]:0)+MPC_Outgoing_links(net,i);
          for(j=0; j<net->power; j++) {
            if(i!=j) {
              k=(net->type->link)(i, j, net->params,
                                  net->power, &(net->nodes), &(net->links));
              if(k) {
                *pt1=j;pt1++;
                *pt2=k;pt2++;
              }
            }
          }
        }
        
        for(i=0, k=2*power-1; i<ngroups; i++)
          {
            k++;
            for(j=0; j<groups[i].maxproc; j++)
                k+=3;
          }
        *plength=k;
        if ((net->temp_topo = malloc(k*sizeof(Int)))==NULL)
          return MPC_ERR_NOMEM;
      }
      
      *atopology = net->temp_topo;      
      (*atopology)[0]=MPC_COMPUTED_MAPPING;
      
      {
        double* absolute_weights;
        Int* linear_numbers;
        Int i;
        Int j, group_number;
        MPC_Topo_graph *topo_graph=NULL;
        Int *bench_map;
        
        if ((absolute_weights = malloc(sizeof(double)*net->power))==NULL)
          return MPC_ERR_NOMEM;
        if ((linear_numbers = malloc(sizeof(Int)*net->power))==NULL)
          return MPC_ERR_NOMEM;    
        for (i=0;i<net->power;i++)
          linear_numbers[i] = i;
        if((bench_map=calloc(ngroups, sizeof(Int)))==NULL)
          return MPC_ERR_NOMEM;
        { /* Create tree for abstract network */
          Int num_of_links, num_of_first_link;
          Int *cur_link_dest, *cur_link_weight;
          
          cur_link_dest=topology+2*net->power+2;
          cur_link_weight=topology+(topology[0]-1)/2+net->power+2;
          if ((topo_graph=calloc(net->power, sizeof(MPC_Topo_graph)))==NULL)
            return MPC_ERR_NOMEM;
          for(i=0;i<net->power;i++)
            {
              num_of_links=i?topology[net->power+i+2]-topology[net->power+i+1]:
                topology[net->power+2];
              num_of_first_link=i?topology[net->power+i+1]:0;
              topo_graph[i].number=i;
              topo_graph[i].node_weight=topology[i+2]/MPC_FIXPOINT_SCALE;
              topo_graph[i].number_of_links=num_of_links;
              if ((topo_graph[i].links=calloc(num_of_links, sizeof(MPC_Topo_graph *)))
                  ==NULL)
                return MPC_ERR_NOMEM;
              if ((topo_graph[i].link_weights=calloc(num_of_links, sizeof(double)))
                  ==NULL)
                return MPC_ERR_NOMEM;
              if ((topo_graph[i].hard_link_times=calloc(num_of_links, sizeof(double)))
                  ==NULL)
                return MPC_ERR_NOMEM;
              for(j=0; j<num_of_links; j++, cur_link_weight++, cur_link_dest++)
                {
                  topo_graph[i].links[j]=topo_graph+*cur_link_dest;
                  topo_graph[i].link_weights[j]=*cur_link_weight;
                }
            }
          RC = MPC_Relative_2_absolute(topology+1, absolute_weights,
                                       parent_rank, parent);
        }
        if (RC != MPC_OK)
          return RC;
        RC = MPC_Sort_absolute_weights(net->power, absolute_weights, linear_numbers);
        if (RC != MPC_OK)
          return RC;
        {
          MPC_Topo_graph *prev=NULL;
          int k;
          
          MPC_Current_mapping_estimation=net->type->estim;
          MPC_Ptgs=calloc(power, sizeof(MPC_Topo_graph *));
          
          group_number=MPC_Group_of(parent);
          bench_map[group_number]++;
          topo_graph[parent_rank].group_number=group_number;
          topo_graph[parent_rank].proc_weight=groups[group_number].max_nodepower;
          groups[group_number].max_nodepower =
            (groups[group_number].processor_power/MPC_POWER_COEFF)/
            MPC_Upper(bench_map[group_number],groups[group_number].scalability);
          MPC_Ptgs[parent_rank]=topo_graph+parent_rank;
          k=1;
          for (i=0, prev=topo_graph+parent_rank;i<net->power;i++) /*prev - redundant */
            {
              if (linear_numbers[i]!=parent_rank)
                {
                  prev->next=topo_graph+linear_numbers[i];
                  MPC_Ptgs[linear_numbers[i]]=topo_graph+linear_numbers[i];
                  RC=MPC_Find_good_group(topology, topo_graph, topo_graph+parent_rank,
                                         linear_numbers[i], &group_number, bench_map,
                                         NULL, net);
                  if(RC != MPC_OK)
                    return RC;
                  for (j=0;j<groups[group_number].maxproc;j++)
                    {
                      if(groups[group_number].nodeweights[j]==0.0)
                        {
                          groups[group_number].numproc++;
                          groups[group_number].nodeweights[j]=absolute_weights[i];
                          (*atopology)[k++]=groups[group_number].nodenumbers[j];
                          (*atopology)[k++]=linear_numbers[i];
                          break;
                        } /* if */ 
                    } /* for */
                  prev=prev->next;
                } /* if */
            }
          for(i=0; i<ngroups; i++)
            {
              (*atopology)[k++]=groups[i].numproc;
              for(j=0; j<groups[i].maxproc; j++)
                {
                  (*atopology)[k++]=groups[i].nodenumbers[j];
                  MPC_Double2ints(groups[i].nodeweights[j],*atopology+k);
                  k+=2;
                }
            }
          free(MPC_Ptgs);
        }
        free(absolute_weights);
        free(linear_numbers);
        free(bench_map);
        MPC_Free_topo_graph(topo_graph, power);
      }
      *afree_topology = MPC_The_free_topology;
      /*MPC_Local_env_updated_by_net_operation=0;*/
      MPC_Ptgs=NULL;
      MPC_Current_mapping_estimation=NULL;
      return MPC_OK;
    }
/********* new new *********scheme***********@@@***/

    
    if(net->bench==MPC_BENCH) {
      num_of_links=MPC_Num_of_links(net);
      size=2*(power+num_of_links)+1;
    }
    else
      size=power;
    if ((net->temp_topo = malloc((size+1)*sizeof(Int)))==NULL)
      return MPC_ERR_NOMEM;
    index=0;
    net->temp_topo[index++] = size;
    if(net->bench==MPC_BENCH) {
      net->temp_topo[index++] = power;  
    }
    for (i=0;i<power;i++)
      net->temp_topo[index++] = MPC_Node_weight(net,i);
    if(net->bench==MPC_BENCH) {
      Int j, k;
      Int *pt1, *pt2;

      pt1=net->temp_topo+2*power+2;
      pt2=pt1+num_of_links;
      for (i=0; i<power; i++, index++) {
        net->temp_topo[index]=
          (i?net->temp_topo[index-1]:0)+MPC_Outgoing_links(net,i);
        for(j=0; j<power; j++) {
          if(i!=j) {
            k=(net->type->link)(i, j, net->params,
                                net->power, &(net->nodes), &(net->links));
            if (MPC_DEBUG == 1)
              printf("%d -> %d = %d\n", i, j, k);
            if(k) {
              *pt1=j;pt1++;
              *pt2=k;pt2++;
            }
          }
        }
      }
    }
    *plength = size+1;
    *afree_topology = MPC_The_free_topology;
    *atopology = net->temp_topo;      
    return MPC_OK;      
  } /* iMPC_The_topology_data */

/********************* MPC_The_topology_data *************************************/
Int MPC_The_topology_data(              /* topology info encoding in int array */
    MPC_Net* net,                       /* the net which topology is described */
    Int* plength,                       /* size of message in ints */
    MPC_Topology *atopology,            /* message containing topology info */
    MPC_Topology_free *afree_topology)  /*function to free topology-messsage space*/
  {
    Int RC;
    if (MPC_DEBUG > 1)
      MPC_Debug_printf( "---[%2d]-> MPC_The_topology_data\n",
                        MPC_Global_rank());
    RC = iMPC_The_topology_data(net, plength, atopology, afree_topology);
    if (MPC_DEBUG > 1)
      MPC_Debug_printf( "<--[%2d]-- MPC_The_topology_data, RC = %d\n",
                        MPC_Global_rank(), RC);
    return RC;
  } /* MPC_The_topology_data */


/************************MPC_Get_hard_link_time**************new*************@@@***/
double MPC_Get_hard_link_time(Int source_group, Int dest_group, double data_volume)
{
  MPC_Hi_descr *s, *d;
  int i;
  double result, data_stone=MPC_SIZE_OF_DATA_PACK;
  
  s=groups[source_group].link_descr;
  d=groups[dest_group].link_descr;
  if(s==NULL || d==NULL)
    return -1.0;
  while(s!=d)
    if(s->level>d->level)
      s=s->top;
    else if(s->level<d->level)
      d=d->top;
    else
      {
        s=s->top;
        d=d->top;
      }
  result=s->linkspeed[0];
  for(i=0; i<MPC_TRANSFER_DATA_RANGE && data_volume>data_stone;
      i++, data_stone*=MPC_SIZE_OF_DATA_PACK)
    ;
  if(i)
    result+=
      data_volume*
      s->linkspeed[i<MPC_TRANSFER_DATA_RANGE?i:(MPC_TRANSFER_DATA_RANGE-1)];
  return result;
}/****MPC_Get_hard_link_time***/


/*********************** iMPC_The_processors_info ******************************/
Int iMPC_The_processors_info(
    Int* num_of_processors,
    double** relative_performance )
  {
    if (1)
      {
        Int i;
        Int index=0;
        Int j;

        for (i=0;i<ngroups;i++) nprocs += groups[i].scalability;
        if (nprocs <= 0) return MPC_ERR_INTERNAL;
        if ((rperformance == NULL)&&
            (rperformance = malloc(nprocs*sizeof(double))) == NULL)
          return MPC_ERR_NOMEM;        
        for(i=0;i<ngroups;i++)
          for (j=0;j<groups[i].scalability;j++)
            {
              rperformance[index]=(double)(groups[i].processor_power);
              index++;
            } /* for */           
      } /* if */
    (*num_of_processors) = nprocs;
    (*relative_performance) = rperformance;
    return MPC_OK;
  } /* iMPC_The_processors_info */

/*********************** MPC_The_processors_info ******************************/
Int MPC_The_processors_info(
    Int* num_of_processors,
    double** relative_performance )
  {  
    Int RC;
    RC = iMPC_The_processors_info(num_of_processors, relative_performance);
    return RC;
  } /* MPC_The_processors_info */



/*********************** iMPC_Relative_2_absolute_coeff **************************/
Int iMPC_Relative_2_absolute_coeff( Int relative_weight, MPC_Node node,
                                    double* coeff )
  {
     Int i;
     Int j;
     for (i=0;i<ngroups;i++)
       for( j=0;j<groups[i].numproc;j++)
         { 
            if (groups[i].nodenumbers[j] == node) 
              {
                (*coeff) = groups[i].nodeweights[j]/
                  ((relative_weight>0) ?
                   (double)relative_weight :
                   -1.0/(double)relative_weight);
                return MPC_OK;
              } /* if */
         } /* for */  
     return MPC_ERR_INTERNAL;
  } /* iMPC_Relative_2_absolute_coeff */ 

/*********************** MPC_Relative_2_absolute_coeff ****************************/
Int MPC_Relative_2_absolute_coeff( Int relative_weight, MPC_Node node,
                                   double* coeff )
  {
    Int RC;
    RC = iMPC_Relative_2_absolute_coeff(relative_weight, node, coeff);
    return RC;
  } /* MPC_Relative_2_absolute_coeff */ 

/*********************** iMPC_Relative_2_absolute ****************************/
Int iMPC_Relative_2_absolute( MPC_Topology topology, double* absolute_weights,
                              Int parent_rank, MPC_Node parent )
  {
    Int RC;
    Int power;
    double rel2abs;
    Int i;

    power = topology[0];
    if (topology[parent_rank+1] == 0)
      return MPC_ERR_INTERNAL;   
    RC = MPC_Relative_2_absolute_coeff( topology[parent_rank+1], parent, &rel2abs );
    if ( RC != MPC_OK)
      return RC;
    for (i=0;i<power;i++) {
      if (topology[i+1] == 0)
        return MPC_ERR_INTERNAL;
      absolute_weights[i] = ((topology[i+1]>0) ?
                             (double)topology[i+1] :
                             -1.0/(double)topology[i+1])*rel2abs;
    } /* for */
    return MPC_OK;  
  } /* iMPC_Relative_2_absolute */

/*********************** MPC_Relative_2_absolute *********************************/
Int MPC_Relative_2_absolute( MPC_Topology topology, double* absolute_weights,
                             Int parent_rank, MPC_Node parent )
  {
    Int RC;
    RC=iMPC_Relative_2_absolute(topology, absolute_weights, parent_rank, parent);
    return RC; 
  } /* MPC_Relative_2_absolute */


/********************** iMPC_Reset_groups *******************************@@@*****/
Int iMPC_Reset_groups(const Int* map)
  {
    Int i;
    Int j;
    
    for (i=0;i<ngroups;i++) {
      groups[i].numproc = 0;
      groups[i].max_nodepower=groups[i].processor_power/MPC_POWER_COEFF; /**@@@**/
      for(j=0;j<groups[i].maxproc;j++) { 
        if (map[groups[i].nodenumbers[j]] == MPC_FREE)
          groups[i].nodeweights[j] = 0.0;
        else
          groups[i].numproc++;
      } /* for */
    } /* for */    
    return MPC_OK;  
  }  /* iMPC_Reset_groups */

/********************** MPC_Reset_groups *******************************************/
Int MPC_Reset_groups(const Int* map)
  {
    Int RC;

    RC = iMPC_Reset_groups( map );
    return RC;
  }  /* MPC_Reset_groups */

/*********************** MPC_Sort_absolute_weights ********************************/
Int MPC_Sort_absolute_weights(Int power, double* absolute_weights,
                              Int* linear_numbers )
  {
    Int i,j;
    double temp_weight;
    Int temp_number;
    for (i = 0; i < power; i++ )
      for (j = 1; j < power; j++ )
        {
          if (absolute_weights[j-1] < absolute_weights[j])
            {
              temp_weight = absolute_weights[j-1];
              absolute_weights[j-1] = absolute_weights[j];
              absolute_weights[j] = temp_weight;
              temp_number = linear_numbers[j-1];
              linear_numbers[j-1] = linear_numbers[j]; 
              linear_numbers[j] = temp_number;
            } /* if */
        } /* for */ 
    return MPC_OK; 
  } /* MPC_Sort_absolute_weights */

/************************ iMPC_Find_best_group *****************************/
Int iMPC_Find_best_group(double weight, int* group_number)
  {
    Int RC = MPC_ERR_INTERNAL;
    Int i;
    Int j;
    double speed;
    double max_speed = 0.0;
    double summ;

    for (i=0;i<ngroups;i++)
      {
        if(groups[i].numproc < groups[i].maxproc) 
          {
            if (groups[i].scalability > groups[i].numproc)
              speed = groups[i].processor_power/weight;
             else
               {               
                 summ = 0.0;
                 for (j=0;j<groups[i].maxproc;j++)
                   summ += groups[i].nodeweights[j];
                 summ += weight;
                 speed = (groups[i].processor_power*groups[i].scalability/summ);
               } /* if */ 
             if (speed > max_speed)
               {
                 max_speed = speed;
                 (*group_number) = i;
                 RC = MPC_OK;
               } /* if */             
          } /* if */
      } /* for */
    return RC;
  } /* iMPC_Find_best_group */

/************************ MPC_Find_best_group ********************************/
Int MPC_Find_best_group(double weight, int* group_number)
  {
    Int RC;
    RC = iMPC_Find_best_group(weight,group_number); 
    return RC;
  } /* MPC_Find_best_group */

/************************ iMPC_Allocate **************************************/
Int iMPC_Allocate( Int* map, double weight, Int linear_number )
  {
    Int RC;
    Int group_number;
    Int j;

    RC = MPC_Find_best_group( weight, &group_number );
    if (RC != MPC_OK) return RC;
    for (j=0;j<groups[group_number].maxproc;j++)
      {
        if (map[groups[group_number].nodenumbers[j]] == MPC_FREE)
          {
            map[groups[group_number].nodenumbers[j]] = linear_number;
            groups[group_number].numproc++;
            groups[group_number].nodeweights[j] = weight;
            return MPC_OK;
          } /* if */ 
      } /* for */ 
    return MPC_ERR_INTERNAL;
  } /* iMPC_Allocate */

/************************ MPC_Allocate ***************************************/
Int MPC_Allocate( Int* map, double weight, Int linear_number )
  {
    Int RC;
    RC = iMPC_Allocate( map, weight, linear_number );
    return RC;
  } /* MPC_Allocate */

/************************ MPC_Is_involved ************************************/
Int MPC_Is_involved(Int i, const Int* map )
  {
    Int involved = 0;
    Int j;

    for (j = 0; j<groups[i].maxproc; j++)
      {
        involved |= (map[groups[i].nodenumbers[j]] >= 0);
      } /* for */
    return involved;
  } /* MPC_Is_involved */

/************************ MPC_Get_speed *************************************/
Int MPC_Get_speed(Int i, const Int* map, double* minspeed, double* maxspeed)
  {
    Int j;
    double summ = 0;
    double maxweight=0.0;
    double minweight=0.0;
    Int first_process =1;

    for (j = 0; j <groups[i].maxproc; j++)
      if (map[groups[i].nodenumbers[j]] != MPC_FREE) { 
        summ += groups[i].nodeweights[j];
        if (first_process) {
            maxweight = (minweight = groups[i].nodeweights[j]);
            first_process = 0;
        }
        else {
          maxweight = (maxweight >= groups[i].nodeweights[j])?
            maxweight:groups[i].nodeweights[j];
          minweight = (minweight <= groups[i].nodeweights[j])?
            minweight:groups[i].nodeweights[j];
        } /* if */
      } /* if */
    if ((maxweight == 0.0) || (minweight == 0.0))
      return MPC_ERR_INTERNAL;
    if (groups[i].scalability >= groups[i].numproc) {
        (*minspeed) = groups[i].processor_power/maxweight;
        (*maxspeed) = groups[i].processor_power/minweight; 
    }
    else {
      (*maxspeed)=
        ((*minspeed) = groups[i].processor_power*groups[i].scalability/summ);
    } /* if */
    return MPC_OK;          
  } /* MPC_Get_speed */  

/************************ MPC_Group_of ***************************************/
Int MPC_Group_of( MPC_Node node )
  {
    Int i;
    Int j;

    for( i=0;i<ngroups;i++ ) {
      for ( j=0;j<groups[i].maxproc;j++) {
        if (groups[i].nodenumbers[j] == node)
          return i; 
      }
    } /* if */  
    return ngroups;    
  }

/************************ iMPC_Optimize **************************************/
Int iMPC_Optimize( Int* map, Int parent )
  {
    Int RC;
    Int first_group = 1;
    Int i;
    double maxspeed;
    double minspeed;     
    double axspeed;
    double inspeed; 
    Int maxnode;
    Int minnode;   

    for (i=0;i<ngroups;i++)
      if (MPC_Is_involved(i,map) || (i == MPC_Group_of(parent)))
        { 
          if (first_group)
            {
              RC = MPC_Get_speed( i, map, &minspeed, &maxspeed);
              if (RC != MPC_OK)
                return RC;
              maxnode = (minnode = i); 
              first_group = 0;
            }
          else
            {
              RC = MPC_Get_speed( i, map, &inspeed, &axspeed);
              if (RC != MPC_OK) return RC;
              if (axspeed > maxspeed)
                {
                  maxspeed = axspeed;
                  maxnode = i;
                }
              if (inspeed < minspeed)
                {
                  minspeed = inspeed;
                  minnode = i;
                }
            } 
        } /* if */
    return MPC_OK;
  } /* iMPC_Optimize */

/************************ MPC_Optimize ******************************************/
Int MPC_Optimize( Int* map, Int parent )
  {
    Int RC;

    RC = iMPC_Optimize(map,parent);
    return RC;
  } /* MPC_Optimize */

/*******************new****MPC_Topo_graph*********@@@***********************-/
typedef struct MPC_TOPO_GRAPH {
  Int number;   /-* linear number of the virtual processor *-/
  double node_weight;   /-* weight of the virtual processor = *-/
                        /-* the number of benchs to compute *-/
  Int number_of_links;   /-* the number of outgoing virtual links *-/
  struct MPC_TOPO_GRAPH **links;   /-* array of pointers to virtual processors *-/
                                   /-* for the links *-/
  double *link_weights;   /-* array of weights of the virtual links, *-/
                          /-* link_weights[i] = the number of bytes to transfer *-/
                          /-* over i-th link *-/
  Int group_number;   /-* number of the group into which the virtual processor *-/
                      /-* is mapped *-/ 
  double proc_weight;   /-* the weight of process, which the virtual processor *-/
                        /-* is mapped to = MPC_POWER_COEFF/t(bench) *-/
  double *hard_link_times;   /-* hard_link_times[i] = time in sec to transfer *-/
                             /-* link_weights[i] bytes through the hard link *-/
                             /-* on which i-th virtual link mapped *-/
  struct MPC_TOPO_GRAPH *next;   /-* next virtual processor in the list *-/
} MPC_Topo_graph; ----------------*/



/*********************** MPC_Mapping_estimation *******new*************@@@***/

double MPC_Mapping_estimation(MPC_Topo_graph *root)
{
  double max_est, node_est;
  Int i;

  for(max_est=0.0; root!=NULL; root=root->next)
    {
      if (MPC_DEBUG == 1)
        printf("node_weight=%f proc_weight=%f comp_est=%f\n",root->node_weight,
               root->proc_weight, root->node_weight/root->proc_weight);
      for(node_est=root->node_weight/root->proc_weight, i=0;
          i<root->number_of_links;
          i++)
        {
          if (MPC_DEBUG == 1)
            printf("i=%d hard_link_times[i]=%f\n",i,root->hard_link_times[i]);
          node_est+=root->hard_link_times[i];
        }
      if(node_est>max_est)
        max_est=node_est;
    }
  return max_est;
}/* MPC_Mapping_estimation */



/********* new new *********scheme***********@@@***/

double MPC_Part_comp_est(double portion, MPC_Topo_graph *node)
{
  if(node==NULL)
    return 0.;
  else if(MPC_Ptgs) //Brand new Oct 2000
    return 0.01*portion*
           (node->node_weight/groups[node->group_number].processor_power*MPC_POWER_COEFF);
  else
    return 0.01*portion*(node->node_weight/node->proc_weight);
}

double MPC_Part_comm_est(double portion, MPC_Topo_graph *source, MPC_Topo_graph *dest)
{
  if(source==NULL || dest==NULL || source==dest)
    return 0.;
  else
    {
      int i;
      for(i=0; i<source->number_of_links; i++)
        if(source->links[i]==dest)
          break;
      if(i==source->number_of_links)
        return 1.e+16;
      else
        return 0.01*portion*source->hard_link_times[i];
    }
}

/********* new new *********scheme***********@@@***/

/************************ MPC_Get_number_of_computers **********Oct2000***/
Int MPC_Get_number_of_computers(void)
{
   return ngroups;
}

/************************ MPC_Get_number_of_VPs **********Oct2000***/
Int MPC_Get_number_of_VPs(Int group_number, Int total_number_of_VPs)
{
    int i, num_of_VPs;
    for(i=0, num_of_VPs=0; i< total_number_of_VPs; i++)
      if(MPC_Ptgs && MPC_Ptgs[i] && MPC_Ptgs[i]->group_number==group_number)
        num_of_VPs++;
    return num_of_VPs;
}

/************************ MPC_Get_scalability **********Oct2000***/
Int MPC_Get_scalability(Int number_of_group)
{
   return groups[number_of_group].scalability;
}

/************************ MPC_Computes_on **********Oct2000***/
Int MPC_Computes_on(char *involved_VPs, Int number_of_VPs, Int group_number)
{
    int i, j;
    for(i=0; i< number_of_VPs; i++)
      if(involved_VPs[i]) {
        for(j=0; j< number_of_VPs; j++)
          if(MPC_Ptgs && MPC_Ptgs[j] &&
             MPC_Ptgs[j]->number==i &&
             MPC_Ptgs[j]->group_number==group_number)
            return 1;        
      }
    return 0;
}
/************************ MPC_Par_estimation **********Nov2000***/
double MPC_Par_estimation(int ppower, int MPC_actions,
                          double *MPC_estimation,
                          char **MPC_involved_VPs,
                          char *MPC_enclosed_involved_VPs) {
  int *MPC_sorted_actions;
  int MPC_t0,MPC_t1,MPC_t2,MPC_comp_num,MPC_VPs_num,MPC_prc_num;
  double MPC_one_comp_est,MPC_Max_estimation=0.0;
  MPC_sorted_actions=(int *)malloc(MPC_actions*sizeof(int));
  for(MPC_t0=0; MPC_t0<MPC_actions; MPC_t0++) {
    MPC_sorted_actions[MPC_t0]=MPC_t0;
  }
  MPC_Sort_absolute_weights(MPC_actions,MPC_estimation,MPC_sorted_actions);
  MPC_comp_num=MPC_Get_number_of_computers();
  for(MPC_t0=0; MPC_t0<MPC_comp_num; MPC_t0++) {
    MPC_VPs_num=MPC_Get_number_of_VPs(MPC_t0,ppower);
    MPC_prc_num=MPC_Get_scalability(MPC_t0);
    MPC_t2=0;
    MPC_one_comp_est=0.0;
    for(MPC_t1=0; MPC_t2<MPC_Upper(MPC_VPs_num,MPC_prc_num) && MPC_t1<MPC_actions; MPC_t1++) {
      if(MPC_Computes_on(MPC_involved_VPs[MPC_sorted_actions[MPC_t1]],ppower,MPC_t0)) {
        MPC_one_comp_est+=MPC_estimation[MPC_t1];
        MPC_t2++;
      }
    }
    if(MPC_one_comp_est > MPC_Max_estimation) {
      MPC_Max_estimation=MPC_one_comp_est;
    }
  }
  for(MPC_t1=0; MPC_t1<MPC_actions; MPC_t1++) {
    for(MPC_t0=0; MPC_t0<MPC_comp_num; MPC_t0++) {
      if(MPC_Computes_on(MPC_involved_VPs[MPC_sorted_actions[MPC_t1]],ppower,MPC_t0)) {
        break;
      }
      if(MPC_t0 == MPC_comp_num && MPC_estimation[MPC_t1] > MPC_Max_estimation) {
        MPC_Max_estimation=MPC_estimation[MPC_t1];
      }
    }
  }
  if(MPC_enclosed_involved_VPs) 
    for(MPC_t1=0; MPC_t1<MPC_actions; MPC_t1++)   
      if(MPC_involved_VPs[MPC_t1])
        for(MPC_t2=0; MPC_t2<ppower; MPC_t2++)
          if(MPC_involved_VPs[MPC_t1][MPC_t2])
            MPC_enclosed_involved_VPs[MPC_t2]=1; 
  for(MPC_t1=0; MPC_t1<MPC_actions; MPC_t1++) {
    if(MPC_involved_VPs[MPC_t1]) {
      free(MPC_involved_VPs[MPC_t1]);
    }
  }
  free(MPC_sorted_actions);
  free(MPC_involved_VPs);
  free(MPC_estimation);
  return MPC_Max_estimation;
}

/************************ MPC_iFind_good_group **********new*************@@@***/
Int iMPC_Find_good_group(MPC_Topology topology, MPC_Topo_graph *topo_graph,
                         MPC_Topo_graph *topo_parent, Int linear_number,
                         Int *group_number, Int *bench_map, double *estimation, MPC_Net *net)
  {
    Int RC = MPC_ERR_INTERNAL;
    Int i, j, power;
    struct MPC_ESTIMATION {
      double est;
      Int group_num;
      Int num_of_nodes; //12/Dec/00. Added to take into account density of nodes in the same group.
    } minest;
    double curest;
    
    MPC_Topo_graph *current=NULL;

    power=topology[1];
    minest.est=DBL_MAX;
    minest.num_of_nodes=power; //12/Dec/00.
    
    for (i=0;i<ngroups;i++)
      {
        if(groups[i].numproc < groups[i].maxproc) 
          {
            topo_graph[linear_number].group_number=i;
            for(current=topo_parent; current!=NULL; current=current->next)
              {
                if(current->group_number==i)
                  current->proc_weight=(groups[i].processor_power/MPC_POWER_COEFF)/
                    MPC_Upper(bench_map[i]+1,groups[i].scalability);
                if(current!=topo_graph+linear_number)
                  for(j=0; j<current->number_of_links; j++)
                    {
                      if(current->links[j]==topo_graph+linear_number)
                        if((current->hard_link_times[j]=
                          MPC_Get_hard_link_time(current->group_number,
                                                 i, current->link_weights[j]))<0.0)
                          return RC;
                    }
                else
                  for(j=0; j<current->number_of_links; j++)
                    {
                      if(current->links[j]->next!=NULL)
                        if((current->hard_link_times[j]=
                          MPC_Get_hard_link_time(i,current->links[j]->group_number,
                                                 current->link_weights[j]))<0.0)
                          return RC;
                    }
              }
            
            
            /********* new new *********scheme***********@@@***/
            if(MPC_Ptgs)
              curest=(*MPC_Current_mapping_estimation)
                (MPC_Ptgs, net->params, power, &(net->nodes), &(net->links));
            else
              curest=MPC_Mapping_estimation(topo_parent);
            /********* new new *********scheme***********@@@***/

            
            if (MPC_DEBUG == 1)
              printf("\ngroup#=%d estimation=%f\n\n", i, curest);

            // 12/Dec/00. Changed the condition to distribute nodes over computers more evenly
            // if different mapping have the same estimation time, not to allow concentrating
            // nodes on a few computers. 
            if(curest<=1.01*minest.est && (curest<0.99*minest.est || bench_map[i]<minest.num_of_nodes))
              {
                minest.est=curest;
                minest.group_num=i;
                minest.num_of_nodes=bench_map[i];
              }
            // 12/Dec/00. End of changes.

            for(current=topo_parent; current!=NULL; current=current->next)
              {
                if(current->group_number==i)
                  current->proc_weight=groups[i].max_nodepower;
              }
          } /* if */
      } /* for */
    if(estimation!=NULL)
      *estimation=minest.est;
    
    *group_number=minest.group_num;
    bench_map[minest.group_num]++;
    topo_graph[linear_number].group_number=minest.group_num;
    groups[minest.group_num].max_nodepower=
      (groups[minest.group_num].processor_power/MPC_POWER_COEFF)/
      MPC_Upper(bench_map[minest.group_num],groups[minest.group_num].scalability);
    for(current=topo_parent; current!=NULL; current=current->next)
      {
        if(current->group_number==minest.group_num)
          current->proc_weight=groups[minest.group_num].max_nodepower;
        if(current!=topo_graph+linear_number)
          for(j=0; j<current->number_of_links; j++)
            {
              if(current->links[j]==topo_graph+linear_number)
                if((current->hard_link_times[j]=
                  MPC_Get_hard_link_time(current->group_number, minest.group_num,
                                         current->link_weights[j]))<0.0)
                  return RC;
            }
        else
          for(j=0; j<current->number_of_links; j++)
            {
              if(current->links[j]->next!=NULL)
                if((current->hard_link_times[j]=
                    MPC_Get_hard_link_time(minest.group_num,
                                           current->links[j]->group_number,
                                           current->link_weights[j]))<0.0)
                  return RC;
            }
      }
    RC=MPC_OK;
    
    return RC;
  } /* iMPC_Find_good_group */

/******************** MPC_Find_good_group **************new*************@@@***/
Int MPC_Find_good_group(MPC_Topology topology, MPC_Topo_graph *topo_graph,
                        MPC_Topo_graph *topo_parent, Int linear_number,
                        Int *group_number, Int *bench_map, double *estimation, MPC_Net *net)
  {
    Int RC;
    RC = iMPC_Find_good_group(topology, topo_graph, topo_parent, linear_number,
                              group_number, bench_map, estimation, net); 
    return RC;
  } /* MPC_Find_good_group */

/************************ iMPC_The_strategy *******************************@@@*/
Int iMPC_The_strategy(
  MPC_Node parent,           /* absolute number of the node, */
                             /* which will be parent of the net */
  Int parent_rank,           /* linear number of the parent in the net */
  Int power,                 /* number of nodes to allocate+1 */
  Int topology_length,       /* size of topology info for requested net */
  MPC_Topology topology,     /* the requested net topology */
  Int total_nodes,           /* number of nodes in the computational space */
  Int* map )                 /* inout parameter. global map, at the start */
  {
    Int RC;
    double* absolute_weights;
    Int* linear_numbers;
    Int i;
    Int j, group_number;
    Int bench=MPC_NOT_BENCH;
    MPC_Topo_graph *topo_graph=NULL;
    Int *bench_map;

/********* new new *********scheme***********@@@***/
    if(topology[0]==MPC_COMPUTED_MAPPING) {
      int k;
      for(i=1, k=1; i<power; i++, k+=2)
        map[topology[k]]=topology[k+1];
      for(i=0; i<ngroups; i++)
        {
          groups[i].numproc=topology[k++];
          for(j=0; j<groups[i].maxproc; j++)
            {
              groups[i].nodenumbers[j]=topology[k++];
              groups[i].nodeweights[j]=MPC_Ints2double(topology+k);
              k+=2;
            }
          groups[i].max_nodepower=groups[i].processor_power/MPC_POWER_COEFF;
        }
      return MPC_OK;
    }
/********* new new *********scheme***********@@@***/
    
    if (topology_length != (topology[0]+1))
      return MPC_ERR_INTERNAL;
    if (power != topology[0])
      bench=MPC_BENCH;
    if ((absolute_weights = malloc(sizeof(double)*power))==NULL)
      return MPC_ERR_NOMEM;
    if ((linear_numbers = malloc(sizeof(Int)*power))==NULL)
      return MPC_ERR_NOMEM;    
    for (i=0;i<power;i++)
      linear_numbers[i] = i;
    if(bench==MPC_BENCH && (bench_map=calloc(ngroups, sizeof(Int)))==NULL)
        return MPC_ERR_NOMEM;
    if(bench==MPC_BENCH) { /* Create tree for abstract network */
      Int num_of_links, num_of_first_link;
      Int *cur_link_dest, *cur_link_weight;

      cur_link_dest=topology+2*power+2;
      cur_link_weight=topology+(topology[0]-1)/2+power+2;
      if ((topo_graph=calloc(power, sizeof(MPC_Topo_graph)))==NULL)
        return MPC_ERR_NOMEM;
      for(i=0;i<power;i++)
        {
          num_of_links=i?topology[power+i+2]-topology[power+i+1]:topology[power+2];
          num_of_first_link=i?topology[power+i+1]:0;
          topo_graph[i].number=i;
          topo_graph[i].node_weight=topology[i+2]/MPC_FIXPOINT_SCALE;
          topo_graph[i].number_of_links=num_of_links;
          if ((topo_graph[i].links=calloc(num_of_links, sizeof(MPC_Topo_graph *)))
              ==NULL)
            return MPC_ERR_NOMEM;
          if ((topo_graph[i].link_weights=calloc(num_of_links, sizeof(double)))
              ==NULL)
            return MPC_ERR_NOMEM;
          if ((topo_graph[i].hard_link_times=calloc(num_of_links, sizeof(double)))
              ==NULL)
            return MPC_ERR_NOMEM;
          for(j=0; j<num_of_links; j++, cur_link_weight++, cur_link_dest++)
            {
              topo_graph[i].links[j]=topo_graph+*cur_link_dest;
              topo_graph[i].link_weights[j]=*cur_link_weight;
            }
        }
      RC = MPC_Relative_2_absolute(topology+1, absolute_weights,
                                   parent_rank, parent);
    }
    else
      RC = MPC_Relative_2_absolute(topology, absolute_weights, parent_rank,parent);
    if (RC != MPC_OK)
      return RC;
    RC = MPC_Sort_absolute_weights(power, absolute_weights, linear_numbers);
    if (RC != MPC_OK)
      return RC;
    RC = MPC_Reset_groups(map);
    if (RC != MPC_OK)
      return RC;
    if(bench==MPC_NOT_BENCH)
      for (i=0;i<power;i++)
        {
          if (linear_numbers[i]!=parent_rank)
            {
              RC = MPC_Allocate(map,absolute_weights[i],linear_numbers[i]);
              if (RC != MPC_OK) return RC; 
            } /* if */
        } /* for */
    /*
      RC = MPC_Optimize( map, parent );      
      if (RC != MPC_OK) return RC;
      */
    else {
      MPC_Topo_graph *prev=NULL;
      
      group_number=MPC_Group_of(parent);
      bench_map[group_number]++;
      topo_graph[parent_rank].group_number=group_number;
      topo_graph[parent_rank].proc_weight=groups[group_number].max_nodepower;
      groups[group_number].max_nodepower =
        (groups[group_number].processor_power/MPC_POWER_COEFF)/
        MPC_Upper(bench_map[group_number],groups[group_number].scalability);
      for (i=0, prev=topo_graph+parent_rank;i<power;i++)
        {
          if (linear_numbers[i]!=parent_rank)
            {
              prev->next=topo_graph+linear_numbers[i];
              RC=MPC_Find_good_group(topology, topo_graph, topo_graph+parent_rank,
                                     linear_numbers[i], &group_number, bench_map,
                                     NULL, NULL);
              if(RC != MPC_OK)
                return RC;
              for (j=0;j<groups[group_number].maxproc;j++)
                {
                  if(map[groups[group_number].nodenumbers[j]]==MPC_FREE)
                    {
                      map[groups[group_number].nodenumbers[j]]=linear_numbers[i];
                      groups[group_number].numproc++;
                      groups[group_number].nodeweights[j]=absolute_weights[i];
                      break;
                    } /* if */ 
                } /* for */
              prev=prev->next;
            } /* if */
        }
    }
    free(absolute_weights);
    free(linear_numbers);
    if(bench==MPC_BENCH) {
      int i;
      if (MPC_DEBUG == 1) {
        printf("bench_map::  ");
        for(i=0; i<ngroups; i++)
          printf("%d  ", bench_map[i]);
        printf("\n\n");
      }
      free(bench_map);
      MPC_Free_topo_graph(topo_graph, power);
    }
    return MPC_OK;  
  } /* iMPC_The_strategy */

/************************ MPC_The_strategy *************************************/
Int MPC_The_strategy(
  MPC_Node parent, /*absolute number of the node, which will be parent of the net*/
  Int parent_rank,               /* linear number of the parent in the net */
  Int power,                     /* number of nodes to allocate+1 */
  Int topology_length,           /* size of topology info for requested net */
  MPC_Topology topology,         /* the requested net topology */
  Int total_nodes,               /* number of nodes in the computational space */
  Int* map )                     /* inout parameter. global map, at the start */
  {
    Int RC;
    Int j;

    if (MPC_DEBUG > 1)
      MPC_Debug_printf( "---[ d]-> MPC_The_strategy, parent = %d, power = %d\n",
                        parent,power);
    if (tracemap != NULL) { 
      for (j=0;j<total_nodes;j++) {
        if (map[j] != -2)
          fprintf(tracemap,"| - ");
        else
          {
            if (j==parent)
              fprintf(tracemap,"| p ");
            else
              fprintf(tracemap,"| + ");
          }
      } /* for */
      fprintf(tracemap, "\n"); 
    } /* if */
    RC = iMPC_The_strategy(parent,parent_rank,power,topology_length,
                           topology,total_nodes,map);
    if (tracemap != NULL) { 
      for (j=0;j<total_nodes;j++) {
        if (map[j] == -1) fprintf(tracemap,"| - ");
        if (map[j] == -2) {
          if (j==parent)
            fprintf(tracemap,"|%2d ",parent_rank);
          else
            fprintf(tracemap,"| + ");
        }         
        if (map[j] >= 0)
          fprintf(tracemap,"|%2d ",map[j]);
      }
      fprintf( tracemap, "\n"); 
      for (j=0;j<total_nodes;j++)
        fprintf(tracemap,"|---");
      fprintf( tracemap, "\n");
    } /* if */
    if (MPC_DEBUG > 1)
      MPC_Debug_printf( "<--[ d]-- MPC_The_strategy, parent = %d, power = %d\n",
                        parent,power);
    return RC;
  } /* MPC_The_strategy */

#define MPC_HOST_WEIGHT 100.0
#ifndef MPC_OLDGETFILENAME
/************************** MPC_Get_filename ************************************/
Int MPC_Get_filename( char** filename, const char* envvar )
  {
    if (envvar == MPC_Topofile) { 
      (*filename) = getenv(MPC_Topo);
      (*filename) = strcat(*filename,MPC_Current_topo);
    } 
    else
      (*filename) = getenv(envvar);
    return ((*filename) == NULL) ? MPC_ERR_ENVPARAM : MPC_OK;
  } /* MPC_Get_filename */
#else
/************************** MPC_Get_filename ***********************************/
Int MPC_Get_filename( char** filename, const char* envvar )
  {
    (*filename) = getenv(envvar);
    return ((*filename) == NULL) ? MPC_ERR_ENVPARAM : MPC_OK;
  } /* MPC_Get_filename */
#endif

/***@@@********************** iMPC_The_init_strategy ***************************/
#define MPC_LINK_NODE_TAG (-2)
#define MPC_NO_PARENT_TAG (-1)
#define MPC_TEMP_GROUP_TAG (-10)
MPC_Hi_descr *link_nodes;

Int MPC_Index_of_top_node(Int *buf, Int i) {
  Int index=0, j;

  for(j=0; j<i; j++)
    if(buf[j]==MPC_LINK_NODE_TAG || buf[j]<=MPC_TEMP_GROUP_TAG)
      index++;
  return index;
}

void MPC_Reset_env_buffer(Int *buf, Int buf_len) {
  Int j;

  for(j=0; j<buf_len; j++)
    if(buf[j]<=MPC_TEMP_GROUP_TAG)
      buf[j]=(buf[j]-MPC_TEMP_GROUP_TAG)/MPC_TEMP_GROUP_TAG;
  return;  
}


Int *MPC_Environment_buffer; /***new***@@@***/
Int MPC_Size_of_environment_buffer; /***new***@@@***/

/***@@@********************** iMPC_The_init_strategy ***************************/
Int iMPC_The_init_strategy(void)
  {
     Int  RC;
     Int  env_length;
     Int* env_buffer;
     Int  i;
     Int  j;
     Int  k;
     char* tracefilename;
     Int  total_nodes;
     Int  size;
     Int num_of_link_nodes, lncount, l;

     RC = MPC_Env_recv(&env_length, &env_buffer); 
     if (RC != MPC_OK)
       return RC;
     ngroups = env_buffer[0];
     if ((groups = malloc(sizeof(MPC_Group_descr)*ngroups)) == NULL)
       return MPC_ERR_NOMEM;

     for(i=1, num_of_link_nodes=ngroups; i<env_length; i++)
       if(env_buffer[i]==MPC_LINK_NODE_TAG)
         num_of_link_nodes++;
     if ((link_nodes=malloc(sizeof(MPC_Hi_descr)*num_of_link_nodes))==NULL)
       return MPC_ERR_NOMEM;
     
     k = 1;
     total_nodes = 0;
     i=0;
     lncount=0;
     while(k<env_length) {
       if(env_buffer[k]==MPC_LINK_NODE_TAG) {
         k++;
         for(l=0; l<MPC_TRANSFER_DATA_RANGE; l++) {
           link_nodes[lncount].linkspeed[l]=1.0/(double)env_buffer[k++];
           if( MPC_DEBUG == 1)
             printf("link_nodes[%d].linkspeed[%d]=%f\n",
                    lncount, l, link_nodes[lncount].linkspeed[l]);
         }
         link_nodes[lncount].level=env_buffer[k++];
         if( MPC_DEBUG == 1)
           printf("LEVEL=%d\n", link_nodes[lncount].level);
         if(env_buffer[k]==MPC_NO_PARENT_TAG)
           link_nodes[lncount].top=NULL;
         else
           link_nodes[lncount].top=
             link_nodes+MPC_Index_of_top_node(env_buffer, env_buffer[k]);
         if( MPC_DEBUG == 1)
           printf("&link_nodes[%d]=%p, link_nodes[%d].top=%p\n\n",
                  lncount, link_nodes+lncount, lncount, link_nodes[lncount].top);
         k++;
         lncount++;
       }
       else if(k<env_length)
         {
           groups[i].maxproc = env_buffer[k];
           env_buffer[k]=MPC_TEMP_GROUP_TAG+env_buffer[k]*MPC_TEMP_GROUP_TAG;
           k++;
           groups[i].numproc = 0;
           groups[i].scalability = env_buffer[k++];
           groups[i].processor_power = env_buffer[k++]; 
           groups[i].max_nodepower = groups[i].processor_power/MPC_POWER_COEFF;
           if ((groups[i].nodeweights = malloc(sizeof(double)*groups[i].maxproc))
               ==NULL)
             return MPC_ERR_NOMEM;
           if ((groups[i].nodenumbers = malloc(sizeof(Int)*groups[i].maxproc))
               ==NULL)
             return MPC_ERR_NOMEM;
           
           recon_net_V = 1;
           
           for(j=0;j<groups[i].maxproc;j++) 
             {
               recon_net_MASK[total_nodes] = recon_net_V;
               recon_net_V = 0;
               
               groups[i].nodenumbers[j] = env_buffer[k++];
               groups[i].nodeweights[j] = 0.0;
               total_nodes++;
             }
           for(l=0; l<MPC_TRANSFER_DATA_RANGE; l++) {
             link_nodes[lncount].linkspeed[l]=
               1.0/(double)env_buffer[k++];
             if( MPC_DEBUG == 1)
               printf("link_nodes[%d].linkspeed[%d]=%f\n",
                      lncount, l, link_nodes[lncount].linkspeed[l]);
           }
           link_nodes[lncount].level=env_buffer[k++];
           if( MPC_DEBUG == 1)
             printf("LEVEL=%d\n", link_nodes[lncount].level);
           if(env_buffer[k]==MPC_NO_PARENT_TAG)
             link_nodes[lncount].top=NULL;
           else
             link_nodes[lncount].top=
               link_nodes+MPC_Index_of_top_node(env_buffer, env_buffer[k]);
           if( MPC_DEBUG == 1)
             printf("&link_nodes[%d]=%p, link_nodes[%d].top=%p\n\n",
                    lncount, link_nodes+lncount, lncount, link_nodes[lncount].top);
           groups[i].link_descr=link_nodes+lncount;
           if( MPC_DEBUG == 1)
             printf("\n-----------> groups[%d].link_descr = %p <-------\n\n",
                    i, groups[i].link_descr);
           k++;
           lncount++;
           i++;
         }
       else
         break;
     }
     MPC_Reset_env_buffer(env_buffer, env_length);
     groups[0].numproc++;
     groups[0].nodeweights[0] = MPC_HOST_WEIGHT;

     MPC_Environment_buffer=env_buffer; /***new***@@@***/
     MPC_Size_of_environment_buffer=env_length; /***new***@@@***/
     
     recon_net_MASK[total_nodes] = 0; /*not dispatcher*/
     RC = MPC_Get_filename(&tracefilename, MPC_Tracemapfile);   
     if (RC == MPC_OK) {
       tracemap = fopen( tracefilename, "w" );
       if (tracemap == NULL)
         return MPC_ERR_TRACEFILE;
       for (j=0;j<total_nodes;j++)
         fprintf(tracemap,"|%2d ", j);
       fprintf(tracemap, "\n"); 
       for (j=0;j<total_nodes;j++)
         fprintf(tracemap,"|---");
       fprintf(tracemap, "\n"); 
     } /* if */
     return MPC_OK;   
  } /* iMPC_The_init_strategy */

/************************** MPC_The_init_strategy ******************************/
Int MPC_The_init_strategy(void)
  {
    Int RC;

    if( MPC_DEBUG > 1)
      MPC_Debug_printf( "---[ d]-> MPC_The_init_strategy\n" );
    RC = iMPC_The_init_strategy();
    if( MPC_DEBUG > 1)
      MPC_Debug_printf( "<--[ d]-- MPC_The_init_strategy, RC=%d, ngroups=%d\n",
                        RC, ngroups);
    return RC;
  } /* MPC_The_init_strategy */


/*********************** MPC_Is_comment **************@@@***********************/
Int MPC_Is_comment(const char* buffer)
{
  if(strlen(buffer)==0)
    return 1;
  while(isspace(*buffer))
    buffer++;
  if (*buffer== '#')
    return 1;
  else
    return 0;
}  

/********************* MPC_Parse_string ***************@@@*********************/
Int MPC_Parse_string( 
    Int count, 
    char* buffer,
    Int link_weight[],
    Int tag,
    Int* pnprocesses, 
    Int** pprocesses, 
    Int* pscalability, 
    Int* ppower )
  {
    char* s;
    Int nprocesses = 1;
    Int scalability = 1;
    Int power = 0;
    Int i;
    char* start = buffer;
    Int j=0;

    while ((s = (char*)strtok(start," ")) != NULL) 
      { 
        start = NULL;
        switch (s[0])
          {
             case 's' : 
               {
                 if(tag)
                   return MPC_ERR_FILE;
                 else
                   scalability = atoi(s+1);
                  break;
               }
             case 'p' :
               {
                 if(tag)
                   return MPC_ERR_FILE;
                 else
                   power = atoi(s+1);
                  break;
               }
             case 'n' :
               {
                 if(tag)
                   return MPC_ERR_FILE;
                 else
                   nprocesses = atoi(s+1); 
                  break;
               }  
             case 'c' :
               {
                 link_weight[j++] = atoi(s+1);
                 if(j>MPC_TRANSFER_DATA_RANGE)
                   return MPC_ERR_FILE;
                 break;
               }  
             default : 
               {
                  return MPC_ERR_FILE;
               }
          } /* switch */
      } /* while */
    if(j!=MPC_TRANSFER_DATA_RANGE) return MPC_ERR_FILE;
    if(!tag) {
      if (power == 0)
        return MPC_ERR_FILE;          
      if ((*pprocesses = (Int*)malloc(nprocesses*sizeof(Int))) == NULL)
        return MPC_ERR_NOMEM;      
      for (i=0;i<nprocesses;i++)
        (*pprocesses)[i] = count+i;
      *pscalability = scalability;
      *pnprocesses = nprocesses;
      *ppower = power;
    }
    return MPC_OK; 
  } /* MPC_Parse_string */

/********************** MPC_Pack_environment **********************@@@*********/
Int MPC_Pack_environment( 
  Int** penvironment, 
  Int* penv_length, 
  Int nprocesses, 
  Int* processes, 
  Int scalability, 
  Int power,
  Int link_weight[],
  Int level,
  Int *stack)
  {
    Int i;
    Int k;

    k = *penv_length;
    (*penvironment)[0]++;
    *penv_length += 5+MPC_TRANSFER_DATA_RANGE+nprocesses;
    if ((*penvironment = realloc((*penvironment),sizeof(Int)*(*penv_length)))
        == NULL)
      return MPC_ERR_NOMEM;
    (*penvironment)[k++] = nprocesses;
    (*penvironment)[k++] = scalability;
    (*penvironment)[k++] = power;
    for (i=0;i<nprocesses;i++)
      (*penvironment)[k++] = processes[i];
    for (i=0;i<MPC_TRANSFER_DATA_RANGE;i++)
      (*penvironment)[k++] = link_weight[i];
    (*penvironment)[k++] = level+1;
    (*penvironment)[k++] = stack[level];
    return MPC_OK;                 
  } /* MPC_Pack_environment */

/***new**************** MPC_Pack_link_env ***********************@@@**********/
Int MPC_Pack_link_env( 
  Int** penvironment, 
  Int* penv_length, 
  Int link_weight[],
  Int level,
  Int *stack)
  {
    Int i;
    Int k;

    k = *penv_length;
    *penv_length += 3+MPC_TRANSFER_DATA_RANGE;
    if ((*penvironment = realloc((*penvironment),sizeof(Int)*(*penv_length)))
        == NULL)
      return MPC_ERR_NOMEM;
    stack[level]=k;
    (*penvironment)[k++] = MPC_LINK_NODE_TAG;
    for (i=0;i<MPC_TRANSFER_DATA_RANGE;i++)
      (*penvironment)[k++] = link_weight[i];
    (*penvironment)[k++] = level;
    (*penvironment)[k++] = level?stack[level-1]:MPC_NO_PARENT_TAG;
    return MPC_OK;                 
  } /* MPC_Pack_link_env */

/******************** MPC_Process_add *****************************************/
Int MPC_Process_add( Int nprocesses, Int* processes,
                     Int* ptotal_processes, Int** process_ctrl )
  {
    Int k;
    Int i;

    k = *ptotal_processes;
    for (i=0;i<nprocesses;i++)
      if ((*ptotal_processes) <= processes[i])
        *ptotal_processes =  processes[i]+1;
    if ((*process_ctrl =
         (Int*)realloc((char*)(*process_ctrl),(*ptotal_processes)*sizeof(Int)))
        == NULL)
      return MPC_ERR_NOMEM;
    for (i=0;i<nprocesses;i++)
      (*process_ctrl)[k+i] = 0;
    for (i=0;i<nprocesses;i++)
      if ((*process_ctrl)[processes[i]] != 0)
        return MPC_ERR_DUPPROCESS;
      else
        (*process_ctrl)[processes[i]] = 1;   
    return MPC_OK;
  } /* MPC_Process_add */

/************************ MPC_Process_ctrl ************************************/
Int MPC_Process_ctrl( Int power, Int total_processes, Int* process_ctrl )
  {
    Int i; 

    if (power != total_processes) 
      {
        if (power < total_processes)
          return MPC_ERR_MISSEDPROCESS;
        else
          return MPC_ERR_DUPPROCESS;
      } /* if */      
    for (i=0;i<total_processes;i++)
      if(!process_ctrl[i])
        return MPC_ERR_MISSEDPROCESS;
    return MPC_OK;    
  } /* MPC_Process_ctrl */


/*@@@
 * 1. mpC run-time system reads environment topology from a file. The file
 *    should be generated by an external program. Name of the file should be passed
 *    in environment variable which name is placed in MPC_Topofile 
 *    File contains info about first
 *    n-1 processes, where n is total number of MPI processes. 
 *
 * 2. File mpC_environment.topo is a text file. It contans 5 types of strings:
 *
 *    <comment string> = #<any sequence of symbols>
 *
 *    <processor group description> = [s<int>] [n<int>] p<int> c<int> {c<int>...} 
 *    where s<int> - is number of phisical processors, if 1 can be omitted,
 *    p<int> - power of one processor
 *    n<int> - number of processes resident on the node
 *    c<int> ... - communication speed in bytes/sec for different sizes of messages
 *                 between of processors of the group.
 *
 *   <link description> = c<int> {c<int>...}
 *   where c<int> - communication speed in bytes/sec for different sizes of messages
 *   between processors from different clusters.
 *
 *   {        - opens new cluster
 *
 *   }        - closes cluster
 *
 *   Empty strings which contain only spaces (or tabs) are treated as comments
 */
Int MPC_Is_link_description(char *buf)
{
  while(isspace(*buf))
    buf++;
  if(*buf=='c')
    return 1;
  else
    return 0;
}

Int MPC_Is_left_brace(char *buf)
{
  while(isspace(*buf))
    buf++;
  if(*buf=='{')
    return 1;
  else
    return 0;
}

Int MPC_Is_right_brace(char *buf)
{
  while(isspace(*buf))
    buf++;
  if(*buf=='}')
    return 1;
  else
    return 0;
}

Int iMPC_The_get_env(
    Int count,
    int* pargc, 
    LPPSTR* pargv, 
    Int* penv_length,
    Int** penvironment )
  {
    Int RC;
    FILE* f;
    char buffer[MPC_MAX_ENVTOPOSTRINGLENGTH];
    Int nprocessors = 0;
    Int nprocesses;
    Int* processes;
    Int total_processes;
    char* filename;
    Int scalability;
    Int power;
    Int stack_of_tops[MPC_MAX_LINK_LEVELS], level;
    Int link_weight[MPC_TRANSFER_DATA_RANGE], i;

    RC = MPC_Get_filename ( &filename, MPC_Topofile );
    if (RC != MPC_OK)
      return RC;
    f = fopen( filename, "r" );
    if (f == NULL)
      return MPC_ERR_FILE;        
    *penv_length = 1;
    if ((*penvironment = malloc(sizeof(Int)*(*penv_length))) == NULL)
      return MPC_ERR_NOMEM;
    (*penvironment)[0] = 0;    

    total_processes = 0;
    
    level=0;
    
    while (!feof(f))
      {
        buffer[0] = (char)0;
        fgets(buffer,MPC_MAX_ENVTOPOSTRINGLENGTH,f);
        if (MPC_Is_comment(buffer)) {;}
        else if(MPC_Is_left_brace(buffer))
          {
            level++;
            if(level==MPC_MAX_LINK_LEVELS) return MPC_ERR_FILE;
          }
        else if(MPC_Is_right_brace(buffer))
          {
            level--;
            if(level<0) return MPC_ERR_FILE;
          }
        else if(MPC_Is_link_description(buffer))
          {
            RC = MPC_Parse_string(total_processes,buffer,link_weight,1,
                                  &nprocesses,&processes,&scalability,&power);
            if (RC != MPC_OK) return RC;
            RC = MPC_Pack_link_env( penvironment, penv_length,
                                    link_weight, level, stack_of_tops);
            if (RC != MPC_OK) return RC;
         }
        else 
          {
             nprocessors++;
             RC = MPC_Parse_string(total_processes,buffer,link_weight,0,
                                   &nprocesses,&processes,&scalability,&power);
             total_processes += nprocesses;
             if (RC != MPC_OK) return RC;
             RC = MPC_Pack_environment( penvironment, penv_length,
                                        nprocesses, processes, scalability, power,
                                        link_weight, level, stack_of_tops);
             if (RC != MPC_OK) return RC;
             free(processes); 
          }  /* if */
      } /* while */
    fclose(f);
    
    if (MPC_DEBUG==1) 
      {
        int i;
        for(i=0; i<*penv_length; i++)
          printf("%d ", (*penvironment)[i]);
        printf("\n\n");
      }
    return MPC_OK;                     
  } /* iMPC_The_get_env */ 

/************************* MPC_The_get_env ************************************/
Int MPC_The_get_env(
    Int count,
    int* pargc, 
    LPPSTR* pargv, 
    Int* penv_length,
    Int** penvironment )
  {
    Int RC;
    char err_str[MAX_ERR_STR];

    if (MPC_DEBUG) 
      {
        MPC_Debug_printf( "---[%2d]-> MPC_The_get_env\n",
                          MPC_Net_global.rank);
      } /* if */
    RC = iMPC_The_get_env(count,pargc,pargv,penv_length,penvironment);
    if (RC != MPC_OK) 
      {
        RC2STR( err_str, RC );
        MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_The_get_env\n",
                          MPC_Net_global.rank, err_str );
        exit(RC);
      } /* if */
    if (MPC_DEBUG) 
      {
        MPC_Debug_printf( "<--[%2d]-- MPC_The_get_env\n",
                          MPC_Net_global.rank);
      } /* if */
    return MPC_OK;
  } /* MPC_The_get_env */



/**********new**** Network type estimation *********@@@*********************/

Int MPC_Local_env_built;
Int MPC_Local_env_updated_by_net_operation;
Int MPC_Local_env_updated_by_recon;

Int MPC_Recon_epilogue(void)
{
  MPC_Local_env_updated_by_recon=1;
  MPC_Global_barrier();
}

void MPC_Refresh_landscape(void)
{
  MPC_Local_env_updated_by_net_operation=1;
  return;
}

/**********new**** MPC_Get_local_env *********@@@*********************/
Int MPC_Get_local_env(Int *env_length, Int **env_buffer)
{
  Int RC;
  RC=MPC_Send_request_for_env();
  if (RC != MPC_OK)
    return RC;
  RC=MPC_Local_env_recv(env_length, env_buffer);
  if (RC != MPC_OK)
    return RC;
  return MPC_OK;
} /* MPC_Get_local_env */

/**********new**** MPC_Build_local_env *********@@@*********************/
Int MPC_Build_local_env(void)
  {
     Int  RC;
     Int  env_length;
     Int* env_buffer;
     Int  i;
     Int  j;
     Int  k;
     Int  total_nodes;
     Int  size;
     Int num_of_link_nodes, lncount, l;

     RC = MPC_Get_local_env(&env_length, &env_buffer); 
     if (RC != MPC_OK)
       return RC;
     ngroups = env_buffer[0];
     if ((groups = malloc(sizeof(MPC_Group_descr)*ngroups)) == NULL)
       return MPC_ERR_NOMEM;

     for(i=1, num_of_link_nodes=ngroups; i<env_length; i++)
       if(env_buffer[i]==MPC_LINK_NODE_TAG)
         num_of_link_nodes++;
     if ((link_nodes=malloc(sizeof(MPC_Hi_descr)*num_of_link_nodes))==NULL)
       return MPC_ERR_NOMEM;
     
     k = 1;
     total_nodes = 0;
     i=0;
     lncount=0;
     while(k<env_length) {
       if(env_buffer[k]==MPC_LINK_NODE_TAG) {
         k++;
         for(l=0; l<MPC_TRANSFER_DATA_RANGE; l++) {
           link_nodes[lncount].linkspeed[l]=1.0/(double)env_buffer[k++];
         }
         link_nodes[lncount].level=env_buffer[k++];
         if(env_buffer[k]==MPC_NO_PARENT_TAG)
           link_nodes[lncount].top=NULL;
         else
           link_nodes[lncount].top=
             link_nodes+MPC_Index_of_top_node(env_buffer, env_buffer[k]);
         k++;
         lncount++;
       }
       else if(k<env_length)
         {
           groups[i].maxproc = env_buffer[k];
           env_buffer[k]=MPC_TEMP_GROUP_TAG+env_buffer[k]*MPC_TEMP_GROUP_TAG;
           k++;
           groups[i].numproc = 0;
           groups[i].scalability = env_buffer[k++];
           groups[i].processor_power = env_buffer[k++]; 
           groups[i].max_nodepower = groups[i].processor_power/MPC_POWER_COEFF;
           if ((groups[i].nodeweights = malloc(sizeof(double)*groups[i].maxproc))
               ==NULL)
             return MPC_ERR_NOMEM;
           if ((groups[i].nodenumbers = malloc(sizeof(Int)*groups[i].maxproc))
               ==NULL)
             return MPC_ERR_NOMEM;
           
           for(j=0;j<groups[i].maxproc;j++) 
             {
               groups[i].nodenumbers[j] = env_buffer[k++];
               groups[i].nodeweights[j] = 0.0;
               total_nodes++;
             }
           for(l=0; l<MPC_TRANSFER_DATA_RANGE; l++) {
             link_nodes[lncount].linkspeed[l]=
               1.0/(double)env_buffer[k++];
           }
           link_nodes[lncount].level=env_buffer[k++];
           if(env_buffer[k]==MPC_NO_PARENT_TAG)
             link_nodes[lncount].top=NULL;
           else
             link_nodes[lncount].top=
               link_nodes+MPC_Index_of_top_node(env_buffer, env_buffer[k]);
           groups[i].link_descr=link_nodes+lncount;
           k++;
           lncount++;
           i++;
         }
       else
         break;
     }
     MPC_Reset_env_buffer(env_buffer, env_length);
     groups[0].numproc++;
     groups[0].nodeweights[0] = MPC_HOST_WEIGHT;
     MPC_Local_env_built=1;
     return MPC_OK;   
  } /* MPC_Build_local_env */

/****new************ MPC_Update_local_env *************************@@@*********/
Int MPC_Update_local_env(void)
{
  Int RC;
  RC=MPC_Send_request_for_refresh();
  if (RC != MPC_OK)
    return RC;
  RC=MPC_Local_env_refresh();
  if (RC != MPC_OK)
    return RC;
  MPC_Local_env_updated_by_net_operation=0;
  if (MPC_DEBUG == 1)
    printf("============ MPC_Update_local_env ===============\n");
  return MPC_OK;
} /* MPC_Update_local_env */

/****new************ MPC_Update_local_powers *************************@@@*********/
Int MPC_Update_local_powers(void)
{
  Int RC;
  Int nprocs, i, j;
  double *powers;

  /* March 1, 2000 */
  /*  MPC_Processors_static_info(&nprocs, &powers); */
  RC=MPC_Updated_local_processors_info(&nprocs, &powers);
  if (RC != MPC_OK) return RC;
  /* March 1, 2000 */

  for(i=0, j=0; i<ngroups; i++)
    {
      groups[i].processor_power = (int)powers[j];
      groups[i].max_nodepower = groups[i].processor_power/MPC_POWER_COEFF;
      j+=groups[i].scalability;
    }
  MPC_Local_env_updated_by_recon=0;
  free(powers);
  if (MPC_DEBUG == 1)
    printf("============ MPC_Update_local_powers ===============\n");
  return MPC_OK;
} /* MPC_Update_local_powers */


/****new************ iMPC_Net_estimate *************************@@@*********/

Int iMPC_Net_estimate(
  MPC_Node parent,          /* absolute number of the node,                */
                            /* which will be parent of the net             */
  MPC_Net* net,             /* net descriptor                              */
  double *est)              /* pointer to variable to hold the estimation  */
  { 
    Int RC;
    Int size;
    Int i, index;
    Int parent_rank;
    static total_nodes;

    if(net->bench==MPC_NOT_BENCH) {
      *est=1.0;
      return MPC_OK;
    }
    net->power = MPC_Power(net);
    if(net->power<=0)
      return MPC_ERR_POWER;
    
    if(!MPC_Local_env_built) {
      RC=MPC_Build_local_env();
      if(RC!=MPC_OK)
        return RC;
    }

     /* rreddy
      * Currently only host can call this function
      * because this function is used for algorithm
      * estimation and called before abstract networks
      * are created.
      *
      * But when this function is equipped to be called
      * by a parent of an abstract network, the parent
      * should refresh its environment and topological
      * information from the dispatcher.
      */
    /*if(MPC_Local_env_updated_by_net_operation)*/
    {
      RC=MPC_Update_local_env();
      if(RC!=MPC_OK)
        return RC;
    }
    if(MPC_Local_env_updated_by_recon) {
      RC=MPC_Update_local_powers();
      if(RC!=MPC_OK)
        return RC;
    }
    {
      int i, num_of_free_nodes;

      num_of_free_nodes=MPC_Total_nodes();
      for(i=0; i<ngroups; i++)
        num_of_free_nodes-=groups[i].numproc;
      if(num_of_free_nodes<net->power-1)
        {
          *est=-1.0;
          return MPC_OK;
        }
    }
    parent_rank = MPC_Parent(net);
    
    net->num_of_links=MPC_Num_of_links(net);
    size=2*(net->power+net->num_of_links)+1;
    if ((net->temp_topo = malloc((size+1)*sizeof(Int)))==NULL)
      return MPC_ERR_NOMEM;
    index=0;
    net->temp_topo[index++] = size;
    net->temp_topo[index++] = net->power;  
    for (i=0;i<net->power;i++)
      net->temp_topo[index++] = MPC_Node_weight(net,i);
    {
      Int j, k;
      Int *pt1, *pt2;

      pt1=net->temp_topo+2*net->power+2;
      pt2=pt1+net->num_of_links;
      for (i=0; i<net->power; i++, index++) {
        net->temp_topo[index]=
          (i?net->temp_topo[index-1]:0)+MPC_Outgoing_links(net,i);
        for(j=0; j<net->power; j++) {
          if(i!=j) {
            k=(net->type->link)(i, j, net->params,
                                net->power, &(net->nodes), &(net->links));
            if(k) {
              *pt1=j;pt1++;
              *pt2=k;pt2++;
            }
          }
        }
      }
    }
    {
      double* absolute_weights;
      Int* linear_numbers;
      Int i;
      Int j, group_number;
      MPC_Topo_graph *topo_graph=NULL;
      Int *bench_map;
      Int *topology=net->temp_topo;
      
      if ((absolute_weights = malloc(sizeof(double)*net->power))==NULL)
        return MPC_ERR_NOMEM;
      if ((linear_numbers = malloc(sizeof(Int)*net->power))==NULL)
        return MPC_ERR_NOMEM;    
      for (i=0;i<net->power;i++)
        linear_numbers[i] = i;
      if((bench_map=calloc(ngroups, sizeof(Int)))==NULL)
        return MPC_ERR_NOMEM;
      { /* Create tree for abstract network */
        Int num_of_links, num_of_first_link;
        Int *cur_link_dest, *cur_link_weight;
        
        cur_link_dest=topology+2*net->power+2;
        cur_link_weight=topology+(topology[0]-1)/2+net->power+2;
        if ((topo_graph=calloc(net->power, sizeof(MPC_Topo_graph)))==NULL)
          return MPC_ERR_NOMEM;
        for(i=0;i<net->power;i++)
          {
            num_of_links=i?topology[net->power+i+2]-topology[net->power+i+1]:
              topology[net->power+2];
            num_of_first_link=i?topology[net->power+i+1]:0;
            topo_graph[i].number=i;
            topo_graph[i].node_weight=topology[i+2]/MPC_FIXPOINT_SCALE;
            topo_graph[i].number_of_links=num_of_links;
            if ((topo_graph[i].links=calloc(num_of_links, sizeof(MPC_Topo_graph *)))
                ==NULL)
              return MPC_ERR_NOMEM;
            if ((topo_graph[i].link_weights=calloc(num_of_links, sizeof(double)))
                ==NULL)
              return MPC_ERR_NOMEM;
            if ((topo_graph[i].hard_link_times=calloc(num_of_links, sizeof(double)))
                ==NULL)
              return MPC_ERR_NOMEM;
            for(j=0; j<num_of_links; j++, cur_link_weight++, cur_link_dest++)
              {
                topo_graph[i].links[j]=topo_graph+*cur_link_dest;
                topo_graph[i].link_weights[j]=*cur_link_weight;
              }
          }
        RC = MPC_Relative_2_absolute(topology+1, absolute_weights,
                                   parent_rank, parent);
      }
      if (RC != MPC_OK)
        return RC;
      RC = MPC_Sort_absolute_weights(net->power, absolute_weights, linear_numbers);
      if (RC != MPC_OK)
        return RC;
      {
        MPC_Topo_graph *prev=NULL;

        if(net->type->estim!=NULL) {
          MPC_Current_mapping_estimation=net->type->estim;
          MPC_Ptgs=calloc(net->power, sizeof(MPC_Topo_graph *));
          MPC_Ptgs[parent_rank]=topo_graph+parent_rank;
        }
        group_number=MPC_Group_of(parent);
        bench_map[group_number]++;
        topo_graph[parent_rank].group_number=group_number;
        topo_graph[parent_rank].proc_weight=groups[group_number].max_nodepower;
        groups[group_number].max_nodepower =
          (groups[group_number].processor_power/MPC_POWER_COEFF)/
          MPC_Upper(bench_map[group_number],groups[group_number].scalability);
        
        /*----------*/        
        if(MPC_Ptgs)
          *est=(*MPC_Current_mapping_estimation)
            (MPC_Ptgs, net->params, net->power, &(net->nodes), &(net->links));
        else
          *est=MPC_Mapping_estimation(topo_graph+parent_rank);
        /*----------*/
            
        for (i=0, prev=topo_graph+parent_rank;i<net->power;i++)
          {
            if (linear_numbers[i]!=parent_rank)
              {
                prev->next=topo_graph+linear_numbers[i];
                if(net->type->estim!=NULL)
                  MPC_Ptgs[linear_numbers[i]]=topo_graph+linear_numbers[i];
                RC=MPC_Find_good_group(topology, topo_graph, topo_graph+parent_rank,
                                       linear_numbers[i], &group_number, bench_map,
                                       est, net);
                if(RC != MPC_OK)
                  return RC;
                for (j=0;j<groups[group_number].maxproc;j++)
                  {
                    if(groups[group_number].nodeweights[j]==0.0)
                      {
                        groups[group_number].numproc++;
                        groups[group_number].nodeweights[j]=absolute_weights[i];
                        break;
                      } /* if */ 
                  } /* for */
                prev=prev->next;
              } /* if */
          }
        if(net->type->estim!=NULL) {
          free(MPC_Ptgs);
          MPC_Ptgs=NULL;
        }
      }
      free(absolute_weights);
      free(linear_numbers);

      {
        int i,j,k;
        for(i=0, bench_map[MPC_Group_of(parent)]--; i<ngroups; i++)
          {
            groups[i].numproc -= bench_map[i];
            groups[i].max_nodepower = groups[i].processor_power/MPC_POWER_COEFF;
            for(j=0; j<groups[i].maxproc;j++)
              if(groups[i].nodeweights[j]==0.0)
                break;
            for(k=bench_map[i]; j>0 && k>0; j--, k--)
              groups[i].nodeweights[j-1]=0.0;
          }
        free(bench_map);
        MPC_Free_topo_graph(topo_graph, net->power);
      }
    }

    return MPC_OK;
  } /* iMPC_Net_estimate */

/**new************* MPC_Net_estimate ***********************@@@************/
double MPC_Net_estimate(
  MPC_Node parent,           /* absolute number of the node, */
                             /* which will be parent of the net */
  MPC_Net* net)             /* net descriptor                             */
  {
    Int RC;
    char err_str[MAX_ERR_STR];
    double est;

    if (MPC_DEBUG)
      {
        MPC_Debug_printf( "---[%2d]-> MPC_Net_estimate (est=%e)\n",
                          MPC_Net_global.rank, est);
      } /* if */
    RC = iMPC_Net_estimate(parent,net,&est);

    if(net->params!=NULL) {
      free(net->params);
      net->params=NULL;
    }
    
    if (RC != MPC_OK)
      {        
        RC2STR( err_str, RC );
        MPC_Debug_printf( "<==[%2d]== MPC error %s during MPC_Net_estimate\n",
                          MPC_Net_global.rank, err_str );
        return -1.0;
        /*exit(RC);*/
      } /* if */
    if (MPC_DEBUG)
      {
        MPC_Debug_printf( "<--[%2d]-- MPC_Net_estimate (est=%e)\n",
                          MPC_Net_global.rank , est);
      } /* if */
    return est; 
  } /* MPC_Net_estimate */


#undef  MPC_Debug_printf

