
   /************************************************/
   /* Helpers for Partitioning Interfaces of       */
   /*                                              */
   /* Revision history                             */
   /* 20-05-2003  --      Initial version          */
   /************************************************/

   #include <math.h>
   #include <hmpi.h>

   /*-----------------------------------------------------*/

   int __HMPI_Number_of_elements_proportional_to_speed
   (
       int p,
       int n,
       const double *speeds,
       int *allocations
   )
   {
       int i, j;
       int total = 0;
       double sum = 0.0;

       for (i = 0; i < p; i++)
       {
           sum += speeds[i];
       }

       for (i = 0; i < p; i++)
       {
           allocations[i] = (
	                      (double)speeds[i]
                              /
		              (double)sum
                            )
		            *
		            n;
       }

       for (i = 0; i < p; i++)
       {
           total += allocations[i];
       }

       if (total == n)
       {
          return HMPI_OK;
       }

       for (i = total; i < n; i++)
       {
           int optimal_p;
	   int *revised_allocations;
	   double *allocation_ratios;
	   double temp;

	   revised_allocations = (int*)malloc(
	 	                       sizeof(int)
                                       *
				       p
           );

	   if (revised_allocations == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   allocation_ratios = (double*)malloc(
		    	                sizeof(double)
                                        *
				        p
           );

	   if (allocation_ratios == NULL)
           {
              return MPC_ERR_NOMEM;
           }

           for (j = 0; j < p; j++)
           {
               revised_allocations[j] = allocations[j] + 1;      
	       allocation_ratios[j] = (double)revised_allocations[j]
		                      /
		                      (double)speeds[j];
           }

	   temp = allocation_ratios[0];
           optimal_p = 0;
           for (j = 1; j < p; j++)
           {
               if (temp > allocation_ratios[j])
               {
                  temp = allocation_ratios[j];
                  optimal_p = j;
               }
           }

	   allocations[optimal_p] = allocations[optimal_p] + 1;

	   free(revised_allocations);
	   free(allocation_ratios);
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   double __HMPI_System_defined_metric
   (
       int p,
       const double *speeds,
       const int *actual,
       const int *ideal
   )
   {
       int i;
       double metric;
       double sumd = 0.0;

       if (HMPI_Debug_flag)
       {
          printf(
             "HMPI===> __HMPI_System_defined_metric: "
             "speeds are\n");

          for (i = 0; i < p; i++)
          {
             printf("%.1f ", speeds[i]);
          }

          printf("\n");

          printf("HMPI===> __HMPI_System_defined_metric: cumulative sumd = \n");
       }

       for (i = 0; i < p; i++)
       {
          if ((int)speeds[i] == 0)
          {
             continue;
          }

          sumd += ((actual[i] - ideal[i])*(actual[i] - ideal[i]))
                  /
                  speeds[i];

          if (HMPI_Debug_flag)
          {
             printf("%.1f ", sumd);
          }
       }

       if (HMPI_Debug_flag)
       {
          printf("\n");
       }

       metric = sqrt(sumd);

       return metric;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Size_of_bins
   (
       int p,
       int n,
       const double *speeds,
       const int *w,
       int *wallocations,
       int *tsum
   )
   {
       int i, j, rc;
       double sump = 0.0;
       int sumw = 0;
       int totalw = 0;

       for (i = 0; i < p; i++)
       {
           sump += speeds[i];
       }

       for (i = 0; i < n; i++)
       {
           sumw += w[i];
       }

       *tsum = sumw;

       for (i = 0; i < p; i++)
       {
           wallocations[i] = (
                              (double)speeds[i]
           	              /
			      (double)sump
			     )
		             *
			     sumw;
       }

       for (i = 0; i < p; i++)
       {
           totalw += wallocations[i];
       }

       for (i = totalw; i < sumw; i++)
       {
           int optimal_p;
	   int *revised_allocations;
	   double *allocation_ratios;
	   double temp;

	   revised_allocations = (int*)malloc(
	 	                       sizeof(int)
                                       *
				       p
           );

	   if (revised_allocations == NULL)
           {
              return MPC_ERR_NOMEM;
           }

	   allocation_ratios = (double*)malloc(
		    	                sizeof(double)
                                        *
				        p
           );

	   if (allocation_ratios == NULL)
           {
              return MPC_ERR_NOMEM;
           }

           for (j = 0; j < p; j++)
           {
               revised_allocations[j] = wallocations[j] + 1;      
	       allocation_ratios[j] = (double)revised_allocations[j]
		                      /
		                      (double)speeds[j];
           }

	   temp = allocation_ratios[0];
           optimal_p = 0;

           for (j = 1; j < p; j++)
           {
               if (temp > allocation_ratios[j])
               {
                  temp = allocation_ratios[j];
                  optimal_p = j;
               }
           }

	   wallocations[optimal_p] = wallocations[optimal_p] + 1;

	   free(revised_allocations);
	   free(allocation_ratios);
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Sum_of_weights_for_ordered_set
   (
       int p,
       int n,
       const double *speeds,
       const int *w,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, rc, prev_proc;
       int *Size_of_bin;
       int *wallocationsc;
       int sumw;
       int sumcum = 0;
       int *Current_bin_capacity;

       Size_of_bin  = (int*)malloc(
	                    sizeof(int)
			    *
			    p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = __HMPI_Size_of_bins(
                p,
		n,
		speeds,
		w,
		Size_of_bin,
		&sumw
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       if (HMPI_Debug_flag)
       {
          printf("HMPI===> __HMPI_Sum_of_weights_for_ordered_set: Sizes of bins are: \n");
          printf("HMPI===> ");

          for (i = 0; i < p; i++)
          {
              printf("%d ", Size_of_bin[i]);
          }

          printf("\n");
       }

       wallocationsc = (int*)malloc(
		             sizeof(int)
			     *
			     (p+1)
       );

       if (wallocationsc == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       wallocationsc[0] = 0;
       for (i = 1; i <= p; i++)
       {
           wallocationsc[i] = wallocationsc[i-1] + Size_of_bin[i-1];
       }

       Current_bin_capacity = (int*)malloc(
		                    sizeof(int)
			            *
			            p
       );

       if (Current_bin_capacity == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           np[i] = 0;
           Current_bin_capacity[i] = 0;
       }

       for (i = 0; i < n; i++)
       {
           prev_proc = 0;
           sumcum += w[i];

           for (j = 0; j < p; j++)
           {
               int Wastej_1, Wastej;

               if ((sumcum > wallocationsc[j])
                   && (sumcum <= wallocationsc[j+1]
	          )
               )
               {
                  if (prev_proc == j)
                  {
                     np[j]++;
		     Current_bin_capacity[j] += w[i];
	             break;
                  }
		  
		  /*
		   * The elements preceding the current one
		   * exactly fit into partition (j-1)
		   */
		  if ((sumcum - w[i]) == wallocationsc[j])
                  {
                     prev_proc = j;
                     np[j]++;
		     Current_bin_capacity[j] += w[i];
	             break;
                  }

		  /*
		   * This is a border element.
		   * The waste is calculated if this element
		   * goes to j-1 or to j.
		   */
                  Wastej_1 = fabs(
				 Size_of_bin[j-1] 
				 - 
				 (
				   Current_bin_capacity[j-1]
                                   +
                                   w[i]
                                 )
                  );

                  Wastej = fabs( (
				  sumw - wallocationsc[j]
                                )
                                -
			        (
				  sumw - sumcum + w[i]
				)
		  );

		  if (Wastej_1 <= Wastej)
                  {
                     np[j-1]++;
		     Current_bin_capacity[j-1] += w[i];
                  }
                  else
                  {
                     np[j]++;
		     Current_bin_capacity[j] += w[i];
                  }

                  break;
               }
           }
       }

       if (metric == NULL)
       {
          free(wallocationsc);
          free(Size_of_bin);
          free(Current_bin_capacity);

          return HMPI_OK;
       }

       /*
        * The ideal sum of weights is given by 
        * elements of array Size_of_bin and the 
        * actual sum of weights is calculated for array elements
        * of Current_bin_capacity.
        */
       switch (type_of_metric)
       {
          case USER_SPECIFIED:
            {
               *metric = (*umf)(
                          p,
                          speeds,
                          Current_bin_capacity,
                          Size_of_bin
               );
            }
            break;
          case SYSTEM_DEFINED:
            {
               *metric = __HMPI_System_defined_metric(
                         p,
                         speeds,
                         Current_bin_capacity,
                         Size_of_bin
               );
            }
            break;
          default:
            {
               return HMPI_ERR_METRIC;
            }
            break;
       }

       free(wallocationsc);
       free(Size_of_bin);
       free(Current_bin_capacity);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Apply_mlimits_to_ordered_sum_of_weights
   (
       int p,
       int n,
       const double *speeds,
       const int *mlimits,
       const int *w,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, k, rc, indl;
       int *Size_of_bin;
       int sumw;
       int total_limits = 0;
       int mlimits_apply = 0;
       int x, y, l, m, opt_start;
       int total_sub_mlimits;
       int wastei, sumtmp, wastef;

       for (i = 0; i < p; i++)
       {
          total_limits += mlimits[i];
       }

       if (total_limits == n)
       {
          for (i = 0; i < p; i++)
          {
             np[i] = mlimits[i];
          }
          return HMPI_OK;
       }

       if (total_limits < n)
       {
          printf(
            "The number of elements in the set"
            " exceed the upper bounds of the processors\n"
          );

          return HMPI_ERR_MLIMITS;
       }

       rc = __HMPI_Sum_of_weights_for_ordered_set(
                p,
		n,
		speeds,
		w,
                type_of_metric,
                umf,
                metric,
		np
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           if (np[i] > mlimits[i])
           {
              mlimits_apply = 1;
	      break;
           }
       }

       if (mlimits_apply == 0)
       {
          return HMPI_OK;
       }

       Size_of_bin  = (int*)malloc(
	                    sizeof(int)
			    *
			    p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = __HMPI_Size_of_bins(
                p,
		n,
		speeds,
		w,
		Size_of_bin,
		&sumw
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       for (i = 0; i < p; i++)
       {
           if (np[i] <= mlimits[i])
           {
              continue;
           }
           
	   /*
	    * We try to distribute the remaining 
	    * elements to the processors following it
	    */
           if (i == 0)
           {
              int reduced_set_size;
              np[i] = mlimits[i];
              reduced_set_size = n - np[i];

              if (HMPI_Debug_flag)
              {
                 printf("HMPI===> __HMPI_Apply_mlimits_to_ordered_sum_of_weights: mlimits = %d, Reduced set size = %d\n", mlimits[i], reduced_set_size);
              }

              free(Size_of_bin);

	      return __HMPI_Apply_mlimits_to_ordered_sum_of_weights(
                       p-1,
	    	       reduced_set_size,
		       (speeds + 1),
		       (mlimits + 1),
		       (w + np[i]),
                       type_of_metric,
                       umf,
                       metric,
		       np + 1
              );
           }

	   /*
	    * If this is the last processor,
	    * we try to distribute the remaining 
	    * elements to the processors preceding it
	    */
	   if (i == (p - 1))
           {
              int reduced_set_size = 0;
                 
	      for (j = 0; j < i; j++)
              {
                  reduced_set_size += np[j];   
              }

	      reduced_set_size += (np[i] - mlimits[i]);
              np[i] = mlimits[i];

              free(Size_of_bin);

	      return __HMPI_Apply_mlimits_to_ordered_sum_of_weights(
                       p-1,
	               reduced_set_size,
		       speeds,
		       mlimits,
		       w,
                       type_of_metric,
                       umf,
                       metric, 
		       np
              );
	   }

           if (HMPI_Debug_flag)
           {
              printf(
                "HMPI===> "
                "__HMPI_Apply_mlimits_to_ordered_sum_of_weights: "
                "Processor %d has upper bound exceeded\n",
                i
              );

              printf("HMPI===> Allocations are: \n");
              for (k = 0; k < p; k++)
              {
                 printf("%d ", np[k]);
              }
              printf("\n");

              printf("HMPI===> element limits are:\n");
              for (k = 0; k < p; k++)
              {
                 printf("%d ", mlimits[k]);
              }
              printf("\n");
           }

           for (k = i+1, total_sub_mlimits = 0; k < p; k++)
           {
               total_sub_mlimits += mlimits[k];
           }

	   /*
	    * Find the maximum subsequence of elements, the number
	    * of elements being equal to mlimits[i] and packing these
	    * elements into bin i generates least amount of waste
	    */
           l = 0;
           for (k = 0; k < i; k++)
           {
               l += np[k];
           }

           indl = l;

           do
           {
               wastei = INT_MAX;

               for (x = indl; x < (indl+np[i]); x++)
               {
                  if (((indl+np[i]) - x) < mlimits[i])
                  {
                     break;
                  }

                  sumtmp = 0;

                  for (y = 0; y < mlimits[i]; y++)
                  {
                      sumtmp += w[x+y];
                  }

                  wastef = fabs(sumtmp - Size_of_bin[i]);

                  if (wastef < wastei)
                  {
                     wastei = wastef;
                     opt_start = x;
                  }
               }

               if (HMPI_Debug_flag)
               {
                  printf("x=%d ", x);
               }

               l = opt_start;
               m = l + mlimits[i];

               indl++;
           }
           while ((n-m) > total_sub_mlimits);

           if (HMPI_Debug_flag)
           {
              printf("\n");
           }

           np[i] = mlimits[i];

           if (HMPI_Debug_flag)
           {
              printf(
                "HMPI===> "
                "__HMPI_Sum_of_weights_for_ordered_set_speed_functions_with_mlimits: "
                "Total number of elements=%d,"
                "Number of elements to be redistributed before=%d,"
                " elements after the element %d to be redistributed\n",
                n,
                l,
                m
              );
           }

	   /*
	    * spread the elements {0, 1, ..., l-1}
	    * amongst the processors before i
	    */
	   rc = __HMPI_Apply_mlimits_to_ordered_sum_of_weights(
                    i,
		    l,
		    speeds,
		    mlimits,
		    w,
                    type_of_metric,
                    umf,
                    metric, 
		    np
           );

	   if (rc != HMPI_OK)
           {
              return rc;
           }

	   /*
	    * spread the elements {m, m+1, ..., n-1}
	    * amongst the processors following i
	    */
	   rc = __HMPI_Apply_mlimits_to_ordered_sum_of_weights(
                    p-(i+1),
		    (n-l-mlimits[i]),
		    (speeds+i+1),
		    (mlimits+i+1),
		    (w+l+mlimits[i]),
                    type_of_metric,
                    umf,
                    metric, 
		    (np+i+1)
           );

	   if (rc != HMPI_OK)
           {
              return rc;
           }
           
           free(Size_of_bin);

           break;
       }

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Sum_of_weights_for_nonordered_set
   (
       int p,
       int n,
       const double *speeds,
       const int *w,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, rc;
       int sumw;
       int *Size_of_bin, *Current_bin_capacity;

       Size_of_bin  = (int*)malloc(
	                    sizeof(int)
			    *
			    p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = __HMPI_Size_of_bins(
                p,
		n,
		speeds,
		w,
		Size_of_bin,
		&sumw
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       Current_bin_capacity = (int*)malloc(
		                    sizeof(int)
			            *
			            p
       );

       if (Current_bin_capacity == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
           Current_bin_capacity[i] = 0;
       }

       for (i = 0; i < n; i++)
       {
           int waste = INT_MAX;
	   int chosen;

	   for (j = 0; j < p; j++)
           {
               if (Current_bin_capacity[j] == Size_of_bin[j])
               {
                  continue;
               }

               if ((Current_bin_capacity[j] + w[i]) <= Size_of_bin[j])
               {
	          np[i] = j;
	          Current_bin_capacity[j] += w[i];
                  break;
               }
           }

	   if (j == p)
           {
	      for (j = 0; j < p; j++)
              {
                  int wastej = fabs(
				 Size_of_bin[j]
                                 -
				 (
                                   Current_bin_capacity[j]
                                   +
                                   w[i]
                                 )
                  );

	          if (wastej < waste)
                  {
                     chosen = j;
	             waste = wastej;
                  }
              }

	      np[i] = chosen;
	      Current_bin_capacity[chosen] += w[i];
           }
       }

       if (metric == NULL)
       {
          free(Size_of_bin);
          free(Current_bin_capacity);

          return HMPI_OK;
       }

       /*
        * The ideal sum of weights is given by 
        * elements of array Size_of_bin and the 
        * actual sum of weights is calculated for array elements
        * of Current_bin_capacity.
        */
       switch (type_of_metric)
       {
          case USER_SPECIFIED:
            {
               *metric = (*umf)(
                          p,
                          speeds,
                          Current_bin_capacity,
                          Size_of_bin
               );
            }
            break;
          case SYSTEM_DEFINED:
            {
               *metric = __HMPI_System_defined_metric(
                         p,
                         speeds,
                         Current_bin_capacity,
                         Size_of_bin
               );
            }
            break;
          default:
            {
               return HMPI_ERR_METRIC;
            }
            break;
       }

       free(Size_of_bin);
       free(Current_bin_capacity);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Apply_mlimits_to_unordered_sum_of_weights_algo_2
   (
       int p,
       int n,
       const double *speeds,
       const int *mlimits,
       const int *w,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, k, rc;
       int sumw;
       int total_limits = 0;
       int *Size_of_bin, *Current_bin_capacity;
       int *Open, *Number_in_bin, *shortlist;

       for (i = 0; i < p; i++)
       {
          total_limits += mlimits[i];
       }

       if (total_limits < n)
       {
          printf(
            "The number of elements in the set"
            " is greater than the sum of numbers of elements"
            " the processors can hold or"
	    " Partitioning cannot be done with the restrictions"
	    " provided\n"
          );

          return HMPI_ERR_MLIMITS;
       }

       Size_of_bin  = (int*)malloc(
	                    sizeof(int)
			    *
			    p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = __HMPI_Size_of_bins(
                p,
		n,
		speeds,
		w,
		Size_of_bin,
		&sumw
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       Current_bin_capacity = (int*)malloc(
		                    sizeof(int)
			            *
			            p
       );

       if (Current_bin_capacity == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          Current_bin_capacity[i] = 0;
       }

       if (total_limits == n)
       {
          int ind = 0, temp_number, temp_mlimit;
          int *rearranged_mlimits;
          int *rearrangedp;

          rearranged_mlimits = (int*)malloc(
                                     sizeof(int)
                                     *
                                     p
          );

          if (rearranged_mlimits == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_mlimits[i] = mlimits[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_mlimits[j-1] > rearranged_mlimits[j])
                  {
                     temp_number = rearrangedp[j-1];
                     rearrangedp[j-1] = rearrangedp[j];
                     rearrangedp[j] = temp_number;

                     temp_mlimit = rearranged_mlimits[j-1];
                     rearranged_mlimits[j-1] = rearranged_mlimits[j];
                     rearranged_mlimits[j] = temp_mlimit;
                  }
              }
          }

          /*
           * TBD:
           * This looks like a NP-hard problem.
           * We know the number of elements in each subset
           * given by the upper bound.
           * We provide a naive implementation here.
           * This is of complexity O(n*n).
           * We arrange the processors in increasing
           * order of their upper bounds and we arrange
           * the weights in decreasing order.
           */
          for (i = 0; i < p; i++)
          {
             for (j = 0; j < rearranged_mlimits[i]; j++)
             {
                 np[ind] = rearrangedp[i];
                 Current_bin_capacity[rearrangedp[i]] += w[ind];
                 ind++;
             }
          }

          if (metric == NULL)
          {
             free(Size_of_bin);
             free(Current_bin_capacity);
             free(rearranged_mlimits);
             free(rearrangedp);

             return HMPI_OK;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for array elements
           * of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
             {
                 *metric = (*umf)(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
             }
             break;
             case SYSTEM_DEFINED:
             {
                 *metric = __HMPI_System_defined_metric(
                           p,
                           speeds,
                           Current_bin_capacity,
                           Size_of_bin
                  );
             }
             break;
             default:
             {
                  return HMPI_ERR_METRIC;
             }
             break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(rearranged_mlimits);
          free(rearrangedp);

          return HMPI_OK;
       }

       Open = (int*)malloc(
                    sizeof(int)
                    *
		    p
       );

       if (Open == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       Number_in_bin = (int*)malloc(
                             sizeof(int)
                             *
		             p
       );

       if (Number_in_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          Open[i] = 1;
          Number_in_bin[i] = 0;
       }

       shortlist = (int*)malloc(
                         sizeof(int)
                         *
		         p
       );

       if (shortlist == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < n; i++)
       {
           int nslist = 0;
	   int chosen = -1;

	   for (j = 0; j < p; j++)
           {
               if (Current_bin_capacity[j] == Size_of_bin[j])
               {
                  continue;
               }
 
               if (((Current_bin_capacity[j] + w[i]) <= Size_of_bin[j])
                   && (Open[j] == 1
                  )
               )
               {
                  shortlist[nslist++] = j;
               }
           }

	   if (nslist > 0)
           {
              int temp = Size_of_bin[shortlist[0]] 
		         - 
			 Current_bin_capacity[shortlist[0]]
              ;
	      chosen = shortlist[0];

	      for (k = 1; k < nslist; k++)
              {
                  int tempk = Size_of_bin[shortlist[k]]
                              -
			      Current_bin_capacity[shortlist[k]]
		  ;

                  if ((tempk >= temp)
                      && (Open[shortlist[k]] == 1
                     )
                  )
                  {
                     temp = tempk;
		     chosen = shortlist[k];
                  }
              }
           }
           else
           {
              int waste = INT_MAX;
	      for (j = 0; j < p; j++)
              {
                  if (Open[j] == 1)
                  {
                     int wastej =    (
				       Current_bin_capacity[j]
				       +
				       w[i]
				       -
				       Size_of_bin[j]
                     );

                     if (wastej < waste)
                     {
                        chosen = j;
			waste = wastej;
                     }
                  }
              }
           }

	   np[i] = chosen;
	   Number_in_bin[chosen]++;

	   if (Number_in_bin[chosen] >= mlimits[chosen])
           {
	      Open[chosen] = 0;
           }

	   Current_bin_capacity[chosen] = Current_bin_capacity[chosen]
		                          +
					  w[i]
           ;
       }

       free(shortlist);
       free(Open);
       free(Number_in_bin);

       if (metric == NULL)
       {
          free(Size_of_bin);
          free(Current_bin_capacity);

          return HMPI_OK;
       }

       /*
        * The ideal sum of weights is given by 
        * elements of array Size_of_bin and the 
        * actual sum of weights is calculated for array elements
        * of Current_bin_capacity.
        */
       switch (type_of_metric)
       {
          case USER_SPECIFIED:
            {
               *metric = (*umf)(
                          p,
                          speeds,
                          Current_bin_capacity,
                          Size_of_bin
               );
            }
            break;
          case SYSTEM_DEFINED:
            {
               *metric = __HMPI_System_defined_metric(
                         p,
                         speeds,
                         Current_bin_capacity,
                         Size_of_bin
               );
            }
            break;
          default:
            {
               return HMPI_ERR_METRIC;
            }
            break;
       }

       free(Size_of_bin);
       free(Current_bin_capacity);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/

   int __HMPI_Apply_mlimits_to_unordered_sum_of_weights
   (
       int p,
       int n,
       const double *speeds,
       const int *mlimits,
       const int *w,
       int type_of_metric,
       User_defined_metric umf,
       double *metric,
       int *np
   )
   {
       int i, j, k, rc;
       int sumw;
       int total_limits = 0;
       int *Size_of_bin, *Current_bin_capacity;
       int *Open, *Number_in_bin, *shortlist;

       for (i = 0; i < p; i++)
       {
          total_limits += mlimits[i];
       }

       if (total_limits < n)
       {
          printf(
            "The number of elements in the set"
            " is greater than the sum of numbers of elements"
            " the processors can hold or"
	    " Partitioning cannot be done with the restrictions"
	    " provided\n"
          );

          return HMPI_ERR_MLIMITS;
       }

       Size_of_bin  = (int*)malloc(
	                    sizeof(int)
			    *
			    p
       );

       if (Size_of_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       rc = __HMPI_Size_of_bins(
                p,
		n,
		speeds,
		w,
		Size_of_bin,
		&sumw
       );

       if (rc != HMPI_OK)
       {
          return rc;
       }

       Current_bin_capacity = (int*)malloc(
		                    sizeof(int)
			            *
			            p
       );

       if (Current_bin_capacity == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          Current_bin_capacity[i] = 0;
       }

       if (total_limits == n)
       {
          int ind = 0, temp_number, temp_mlimit;
          int *rearranged_mlimits;
          int *rearrangedp;

          rearranged_mlimits = (int*)malloc(
                                     sizeof(int)
                                     *
                                     p
          );

          if (rearranged_mlimits == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          rearrangedp = (int*)malloc(
                              sizeof(int)
                              *
                              p
          );

          if (rearrangedp == NULL)
          {
             return MPC_ERR_NOMEM;
          }

          for (i = 0; i < p; i++)
          {
              rearrangedp[i] = i;
              rearranged_mlimits[i] = mlimits[i];
          }

          for (i = 0; i < p; i++)
          {
              for (j = 1; j < p; j++)
              {
                  if (rearranged_mlimits[j-1] > rearranged_mlimits[j])
                  {
                     temp_number = rearrangedp[j-1];
                     rearrangedp[j-1] = rearrangedp[j];
                     rearrangedp[j] = temp_number;

                     temp_mlimit = rearranged_mlimits[j-1];
                     rearranged_mlimits[j-1] = rearranged_mlimits[j];
                     rearranged_mlimits[j] = temp_mlimit;
                  }
              }
          }

          /*
           * TBD:
           * This looks like a NP-hard problem.
           * We know the number of elements in each subset
           * given by the upper bound.
           * We provide a naive implementation here.
           * This is of complexity O(n*n).
           * We arrange the processors in increasing
           * order of their upper bounds and we arrange
           * the weights in decreasing order.
           */
          for (i = 0; i < p; i++)
          {
             for (j = 0; j < rearranged_mlimits[i]; j++)
             {
                 np[ind] = rearrangedp[i];
                 Current_bin_capacity[rearrangedp[i]] += w[ind];
                 ind++;
             }
          }

          if (metric == NULL)
          {
             free(Size_of_bin);
             free(Current_bin_capacity);
             free(rearranged_mlimits);
             free(rearrangedp);

             return HMPI_OK;
          }

          /*
           * The ideal sum of weights is given by
           * elements of array Size_of_bin and the
           * actual sum of weights is calculated for array elements
           * of Current_bin_capacity.
           */
          switch (type_of_metric)
          {
             case USER_SPECIFIED:
             {
                 *metric = (*umf)(
                            p,
                            speeds,
                            Current_bin_capacity,
                            Size_of_bin
                  );
             }
             break;
             case SYSTEM_DEFINED:
             {
                 *metric = __HMPI_System_defined_metric(
                           p,
                           speeds,
                           Current_bin_capacity,
                           Size_of_bin
                  );
             }
             break;
             default:
             {
                  return HMPI_ERR_METRIC;
             }
             break;
          }

          free(Size_of_bin);
          free(Current_bin_capacity);
          free(rearranged_mlimits);
          free(rearrangedp);

          return HMPI_OK;
       }

       Open = (int*)malloc(
                    sizeof(int)
                    *
		    p
       );

       if (Open == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       Number_in_bin = (int*)malloc(
                             sizeof(int)
                             *
		             p
       );

       if (Number_in_bin == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < p; i++)
       {
          Open[i] = 1;
          Number_in_bin[i] = 0;
       }

       shortlist = (int*)malloc(
                         sizeof(int)
                         *
		         p
       );

       if (shortlist == NULL)
       {
          return MPC_ERR_NOMEM;
       }

       for (i = 0; i < n; i++)
       {
           int nslist = 0;
	   int chosen = -1;

	   for (j = 0; j < p; j++)
           {
               if (Current_bin_capacity[j] == Size_of_bin[j])
               {
                  continue;
               }
 
               if (((Current_bin_capacity[j] + w[i]) <= Size_of_bin[j])
                   && (Open[j] == 1
                  )
               )
               {
                  chosen = j;
                  break;
               }
           }

           if (chosen == -1)
           {
              int waste = INT_MAX;
	      for (j = 0; j < p; j++)
              {
                  if (Open[j] == 1)
                  {
                     int wastej =    (
				       Current_bin_capacity[j]
				       +
				       w[i]
				       -
				       Size_of_bin[j]
                     );

                     if (wastej < waste)
                     {
                        chosen = j;
			waste = wastej;
                     }
                  }
              }
           }

	   np[i] = chosen;
	   Number_in_bin[chosen]++;

	   if (Number_in_bin[chosen] >= mlimits[chosen])
           {
	      Open[chosen] = 0;
           }

	   Current_bin_capacity[chosen] = Current_bin_capacity[chosen]
		                          +
					  w[i]
           ;
       }

       free(shortlist);
       free(Open);
       free(Number_in_bin);

       if (metric == NULL)
       {
          free(Size_of_bin);
          free(Current_bin_capacity);

          return HMPI_OK;
       }

       /*
        * The ideal sum of weights is given by 
        * elements of array Size_of_bin and the 
        * actual sum of weights is calculated for array elements
        * of Current_bin_capacity.
        */
       switch (type_of_metric)
       {
          case USER_SPECIFIED:
            {
               *metric = (*umf)(
                          p,
                          speeds,
                          Current_bin_capacity,
                          Size_of_bin
               );
            }
            break;
          case SYSTEM_DEFINED:
            {
               *metric = __HMPI_System_defined_metric(
                         p,
                         speeds,
                         Current_bin_capacity,
                         Size_of_bin
               );
            }
            break;
          default:
            {
               return HMPI_ERR_METRIC;
            }
            break;
       }

       free(Size_of_bin);
       free(Current_bin_capacity);

       return HMPI_OK;
   }

   /*-----------------------------------------------------*/
