
/*************************************************************************
*                                                                       *
*                 Heterogeneous Data Partitioning Interface             *
*                 =========================================             *
*                                                                       *
*  Copyright (c) 2002          Department of Computer Science,          *
*                              University College Dublin.               *
*                                                                       *
*  All rights reserved. We assume no responsibility for the use         *
*  or reliability of our software.                                      *
*                                                                       *
*************************************************************************/

   /************************************************/
   /* Partitioning interfaces for matrices         */
   /*                                              */
   /* Revision history                             */
   /* 19-05-2003  --      Initial version          */
   /************************************************/

   #ifndef __HMPI_PARTITIONING_MATRICES_HH
   #define __HMPI_PARTITIONING_MATRICES_HH

   /*
    * Two-dimensional distributions
    */
   #define   ROW_BASED            1
   #define   COLUMN_BASED         2
   #define   CARTESIAN            3

   /*
    * General rectangular 1D distribution
    */
   #define   GENERAL              4

   /*
    * Types of formulation for general one-dimensional
    * rectangular distributions
    */
   #define   DYNAMIC              1
   #define   ITERATIVE            2
   #define   REFINING             3

   #define   HMPI_RECT_INDEX(a, b, c, d, p, q)  (a*p*q*q+b*p*q+c*q+d)
   #define   H(a, b, c, d, p, q)                (a*p*q*q+b*p*q+c*q+d)

   typedef double (*HMPI_Lower_bound) (
           int p,
	   const double *speeds,
	   int m,
	   int n
   );

   typedef double (*HMPI_DP_function) (
           int rowsorcolumns,
	   int rectangles,
	   int p,
	   const double *speeds,
           double **previous,
	   int *r
   );

   typedef double (*HMPI_Iterative_function) (
           int p,
	   const int *w,
	   const int *h,
	   const int *trow,
	   const int *tcol
   );

   typedef double (*HMPI_Refining_function) (
           int p,
	   const double *speeds,
	   int m,
	   int n,
	   const int *oldw,
	   const int *oldh,
	   const int *oldtrow,
	   const int *oldtcol,
	   const int *neww,
	   const int *newh,
	   const int *newtrow,
	   const int *newtcol
   );
   
   int HMPI_Partition_matrix_2d(
           int p,
           int q,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int m,
	   int n,
	   int type_of_distribution,
	   int *w,
	   int *h,
	   int *trow,
	   int *tcol,
           int *ci,
           int *cj
   );

   int HMPI_Partition_matrix_1d(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int m,
	   int n,
	   int formulation,
	   HMPI_Lower_bound lb,
	   HMPI_DP_function dpf,
	   HMPI_Iterative_function cf,
	   HMPI_Refining_function rf,
	   const int *iw,
	   const int *ih,
	   const int *itrow,
	   const int *itcol,
           int type_of_distribution,
	   int *w,
	   int *h,
	   int *trow,
	   int *tcol,
           int *ci
   );

   int HMPI_Partition_matrix_1d_dp(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int m,
	   int n,
	   HMPI_Lower_bound lb,
	   HMPI_DP_function dpf,
           int type_of_distribution,
	   int *w,
	   int *h,
	   int *trow,
	   int *tcol,
           int *ci
   );

   int HMPI_Partition_matrix_1d_iterative(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int m,
	   int n,
	   HMPI_Lower_bound lb,
	   HMPI_Iterative_function cf,
	   int *w,
	   int *h,
	   int *trow,
	   int *tcol,
           int *ci
   );

   int HMPI_Partition_matrix_1d_refining(
           int p,
           int pn,
	   const double *speeds,
	   const int *psizes,
	   const int *mlimits,
	   int m,
	   int n,
	   HMPI_Lower_bound lb,
	   HMPI_Refining_function rf,
	   int *w,
	   int *h,
	   int *trow,
	   int *tcol,
           int *ci
   );

   int HMPI_Get_processor_2d(
           int i, 
           int j, 
           int p, 
           int q, 
           const int *w, 
           const int *h,
           const int *trow,
           const int *tcol,
           int *ci, 
           int *cj
   );

   int HMPI_Get_processor_1d (
           int i, 
           int j, 
           int p, 
           const int *w,
           const int *h,
           const int *trow,
           const int *tcol,
           int *c
   );

   typedef struct {
           int I; 
           int J;
   } HMPI_Processor;

   int HMPI_Get_matrix_processor (
           int r, 
           int c, 
           int p, 
           int q, 
           const int *w,
           const int *h,
           const int *trow,
           const int *tcol,
           HMPI_Processor *root
   );

   int HMPI_Print_rectangle_1d(
        int p,
        int m,
        int n,
        const int *w,
        const int *h,
        const int *trow,
        const int *tcol,
        const int *ci
   );

   int HMPI_Print_rectangle_2d(
        int p,
        int q,
        int m,
        int n,
        const int *w,
        const int *h,
        const int *trow,
        const int *tcol,
        const int *ci,
        const int *cj
   );

   int HMPI_Common_height
   (
       int top_row_1,
       int bottom_row_1,
       int top_row_2,
       int bottom_row_2
   );

   int HMPI_Get_my_width
   (
       int i, 
       int j, 
       int p, 
       int q, 
       const double *speeds,
       int type_of_distribution,
       int m, 
       int n
   );

   int HMPI_Get_my_height
   (
       int i, 
       int j, 
       int p, 
       int q, 
       const double *speeds,
       int type_of_distribution,
       int m, 
       int n
   );

   int HMPI_Get_diagonal
   (
       int i, 
       int j, 
       int p, 
       int q, 
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol
   );

   int HMPI_Get_my_elements
   (
       int m,
       int n,
       int gm,
       int gn,
       int i, 
       int j, 
       int p, 
       int q, 
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int type_of_distribution,
       char upper_or_lower
   );

   int HMPI_Get_my_kk_elements
   (
       int k,
       int m,
       int n,
       int gm,
       int gn,
       int i, 
       int j, 
       int p, 
       int q, 
       const int *w,
       const int *h,
       const int *trow,
       const int *tcol,
       int type_of_distribution,
       char upper_or_lower
   );

   #endif  /* __HMPI_PARTITIONING_MATRICES_HH */

