
/*************************************************************************
*                                                                       *
*                 HMPI Programming Environment                          *
*                 ============================                          *
*                                                                       *
*  Copyright (c) 2002          Department of Computer Science,          *
*                              University College Dublin.               *
*                                                                       *
*  All rights reserved. We assume no responsibility for the use         *
*  or reliability of our software.                                      *
*                                                                       *
*************************************************************************/

   /************************************************/
   /* hmpi - Implementation of helper functions of */
   /*        the HeteroMPI library                 */
   /*                                              */
   /* Revision history                             */
   /* 01-02-2002  --      Initial version          */
   /************************************************/

   #ifndef __HMPI_INTERNAL_HH
   #define __HMPI_INTERNAL_HH

   /* 
    * A meaningful allocation scheme for network 
    * identifiers should be implemented.
    */
   #define HMPI_NUM_OF_NETS     256
   #define HMPI_STATIC_NET_NAME (~((unsigned int)0) >> 24)
   #define HMPI_GLOBAL_NET_NAME (~((unsigned int)0) << 1)
   #define HMPI_HOST_NET_NAME   (~((unsigned int)0) << 2)
   #define HMPI_FREE_NET_NAME   (~((unsigned int)0) << 2)
   #define HMPI_PROC_NET_NAME   (~((unsigned int)0) << 3)
   #define HMPI_UNDEFINED_NID   -1

   #define HMPI_GROUP_BASE_ID   100

   typedef unsigned int HMPI_Group;
   typedef MPC_NetType HMPI_NetType;
   typedef MPC_NetType HMPI_Model;

   typedef struct _hmpi_net
   {
       MPC_Name _name;
       MPC_Net _net;
   } HMPI_Net;

   /*
    * There are 3 abstract networks available after 
    * HMPI runtime initialization.
    *
    * They are:
    * HMPI_Net_global
    * HMPI_Net_host
    * HMPI_Net_free
    *
    * Their identifiers are respectively given by:
    * HMPI_GLOBAL (HMPI_global)
    * HMPI_HOST   (HMPI_host)
    * HMPI_FREE   (HMPI_free)
    *
    * _hmpi_num_of_nets represent the number of nets the process 
    * is member of.
    *
    */
   extern int _hmpi_net_base;
   extern unsigned int _hmpi_num_of_nets;

   extern HMPI_Group HMPI_global;
   extern HMPI_Group HMPI_proc;
   extern HMPI_Group HMPI_host;
   extern HMPI_Group HMPI_free;

   extern HMPI_Net HMPI_Net_global;
   extern HMPI_Net HMPI_Net_proc;
   extern HMPI_Net HMPI_Net_host;
   extern HMPI_Net HMPI_Net_free;

   /*
    * HMPI diagnostics
    */
   extern int HMPI_Debug_flag;

   /*
    * _nids and _hmpi_nets should probably be replaced by 
    * a map data structure where given a nid, we would get 
    * HMPI_Net.
    */
   extern HMPI_Group* _nids;
   extern HMPI_Net** _hmpi_nets;

   extern MPI_Comm HMPI_COMM_WORLD;
   extern double *HMPI_Comm_world_performances;

   #define HMPI_COMM_WORLD_GROUP ((HMPI_Group*)&HMPI_global)
   #define HMPI_PROC_WORLD_GROUP ((HMPI_Group*)&HMPI_proc)
   #define HMPI_FREE_GROUP ((HMPI_Group*)&HMPI_free)
   #define HMPI_HOST_GROUP ((HMPI_Group*)&HMPI_host)

   /*-----------------------------------------------------*/
   /*
    * Fetch the Net structure from the map.
    * The key is the network identifier.
    */
   HMPI_Net*
   HMPI_Get_net_from_map
   (
       const HMPI_Group* pgroup_t,
       int* index
   );

   /*-----------------------------------------------------*/
   /*
    * Basic checks on the network identifier.
    */
   int
   HMPI_Test_net_id
   (
       const HMPI_Group* pgroup_t
   );

   /*-----------------------------------------------------*/
   /*
    * Host sends a message to the dispatcher, which sends
    * a message to all unemployed nodes to leave the waiting
    * point.
    * This should be probably provided as a synchronisation
    * function to the User.
    */
   int
   HMPI_Host_signal_free_processes_to_exit_waiting_points();

   int
   HMPI_Signal_free_processes_to_exit_waiting_points();
   /*-----------------------------------------------------*/
   /*
    * Free nodes wait for commands from Dispatcher here which
    * is the waiting point.
    * This should be probably provided as a synchronisation
    * function to the User.
    */
   int
   HMPI_Offer_free_processes_to_net
   (
       MPC_Name name,
       MPC_Net* net
   );

   /*-----------------------------------------------------*/
   /*
    * Initialise an abstract network structure.
    */
   int
   HMPI_Init_net
   (
       HMPI_Net *pnet
   );

   /*-----------------------------------------------------*/
   /*
    * Get the rank of the process given its coordinates.
    * This is called by HMPI_Rank interface function and
    * should be made part of the implementation of HMPI_Rank.
    */
   int
   HMPI_Process_rank_given_coordinates
   (
       MPC_Net* net,
       const int* coord
   );

   /*-----------------------------------------------------*/

   /*
    * Determine the relative numbering of the same processes
    * in two different abstract networks.
    */
   int 
   HMPI_Translate_ranks
   (
       const HMPI_Group* nid1,
       int number,
       int *ranks1,
       const HMPI_Group* nid2,
       int *ranks2
   );

   /*-----------------------------------------------------*/

   #endif
